/*
 * Copyright (c) 2002 - 2006 IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 */
package com.ibm.wala.examples.analysis.js;

import com.ibm.wala.analysis.pointers.HeapGraph;
import com.ibm.wala.cast.ir.ssa.AstIRFactory;
import com.ibm.wala.cast.js.ipa.callgraph.JSAnalysisOptions;
import com.ibm.wala.cast.js.ipa.callgraph.JSCFABuilder;
import com.ibm.wala.cast.js.ipa.callgraph.JSCallGraphUtil;
import com.ibm.wala.cast.js.ipa.callgraph.JSZeroOrOneXCFABuilder;
import com.ibm.wala.cast.js.ipa.callgraph.PropertyNameContextSelector;
import com.ibm.wala.cast.js.ipa.callgraph.correlations.extraction.CorrelatedPairExtractorFactory;
import java.nio.file.Files;
import com.ibm.wala.cast.js.loader.JavaScriptLoader;
import com.ibm.wala.cast.js.loader.JavaScriptLoaderFactory;
import com.ibm.wala.cast.js.ssa.JavaScriptPropertyRead;
import com.ibm.wala.cast.tree.rewrite.CAstRewriterFactory;
import com.ibm.wala.classLoader.IField;
import com.ibm.wala.classLoader.IMethod;
import com.ibm.wala.classLoader.IMethod.SourcePosition;
import com.ibm.wala.classLoader.Module;
import com.ibm.wala.classLoader.SourceModule;
import com.ibm.wala.classLoader.SourceURLModule;
import com.ibm.wala.core.util.io.FileProvider;
import com.ibm.wala.ipa.callgraph.AnalysisScope;
import com.ibm.wala.ipa.callgraph.CGNode;
import com.ibm.wala.ipa.callgraph.CallGraph;
import com.ibm.wala.ipa.callgraph.Entrypoint;
import com.ibm.wala.ipa.callgraph.IAnalysisCacheView;
import com.ibm.wala.ipa.callgraph.propagation.ConstantKey;
import com.ibm.wala.ipa.callgraph.propagation.HeapModel;
import com.ibm.wala.ipa.callgraph.propagation.InstanceFieldKey;
import com.ibm.wala.ipa.callgraph.propagation.InstanceKey;
import com.ibm.wala.ipa.callgraph.propagation.LocalPointerKey;
import com.ibm.wala.ipa.callgraph.propagation.PointerAnalysis;
import com.ibm.wala.ipa.callgraph.propagation.PointerKey;
import com.ibm.wala.ipa.callgraph.propagation.PropagationCallGraphBuilder;
import com.ibm.wala.ipa.callgraph.propagation.cfa.ZeroXInstanceKeys;
import com.ibm.wala.ipa.cha.ClassHierarchyException;
import com.ibm.wala.ipa.cha.IClassHierarchy;
import com.ibm.wala.shrike.shrikeCT.InvalidClassFileException;
import com.ibm.wala.ssa.DefUse;
import com.ibm.wala.ssa.IR;
import com.ibm.wala.ssa.IRFactory;
import com.ibm.wala.ssa.SSAGetInstruction;
import com.ibm.wala.ssa.SSAInstruction;
import com.ibm.wala.ssa.SymbolTable;
import com.ibm.wala.util.CancelException;
import com.ibm.wala.util.WalaException;
import com.ibm.wala.util.collections.Iterator2Iterable;
import com.ibm.wala.util.intset.OrdinalSet;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.JarURLConnection;
import java.net.URL;
import java.nio.file.Paths;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

/** TODO this class is a mess. rewrite. */
public class JSCallGraphBuilderUtil extends com.ibm.wala.cast.js.ipa.callgraph.JSCallGraphUtil {

  public enum CGBuilderType {
    ZERO_ONE_CFA(false, true, true),
    ZERO_ONE_CFA_WITHOUT_CORRELATION_TRACKING(false, true, false),
    ZERO_ONE_CFA_NO_CALL_APPLY(false, false, true),
    ONE_CFA(true, true, true);

    private final boolean useOneCFA;

    private final boolean handleCallApply;

    private final boolean extractCorrelatedPairs;

    CGBuilderType(boolean useOneCFA, boolean handleCallApply, boolean extractCorrelatedPairs) {
      this.useOneCFA = useOneCFA;
      this.handleCallApply = handleCallApply;
      this.extractCorrelatedPairs = extractCorrelatedPairs;
    }

    public boolean useOneCFA() {
      return useOneCFA;
    }

    public boolean handleCallApply() {
      return handleCallApply;
    }

    public boolean extractCorrelatedPairs() {
      return extractCorrelatedPairs;
    }
  }

  /**
   * create a CG builder for script. Note that the script at dir/name is loaded
   * via the classloader,
   * not from the filesystem.
   */
  public static JSCFABuilder makeScriptCGBuilder(
      String dir, String name, CGBuilderType builderType, ClassLoader loader)
      throws IOException, WalaException {
    URL script = getURLforFile(dir, name, loader);
    CAstRewriterFactory<?, ?> preprocessor = builderType.extractCorrelatedPairs
        ? new CorrelatedPairExtractorFactory(translatorFactory, script)
        : null;
    JavaScriptLoaderFactory loaders = JSCallGraphUtil.makeLoaders(preprocessor);

    AnalysisScope scope = makeScriptScope(dir, name, loaders, loader);

    return makeCG(loaders, scope, builderType, AstIRFactory.makeDefaultFactory());
  }

  public static URL getURLforFile(String dir, String name, ClassLoader loader) throws IOException {
    File f = null;
    FileProvider provider = new FileProvider();
    if (dir.startsWith(File.separator)) {
      f = new File(dir + File.separator + name);
    } else {
      try {
        f = provider.getFile(dir + File.separator + name, loader);
      } catch (FileNotFoundException e) {
        // I guess we need to do this on Windows sometimes? --MS
        // if this fails, we won't catch the exception
      }
    }
    return f.toURI().toURL();
  }

  public static AnalysisScope makeScriptScope(
      String dir, String name, JavaScriptLoaderFactory loaders, ClassLoader loader)
      throws IOException {
    return makeScope(makeSourceModules(dir, name, loader), loaders, JavaScriptLoader.JS);
  }

  public static Module[] makeSourceModules(String dir, String name) throws IOException {
    return makeSourceModules(dir, name, JSCallGraphBuilderUtil.class.getClassLoader());
  }

  public static Module[] makeSourceModules(String dir, String name, ClassLoader loader)
      throws IOException {
    URL script = getURLforFile(dir, name, loader);
    Module[] modules = new Module[] {
        (script.openConnection() instanceof JarURLConnection)
            ? new SourceURLModule(script)
            : makeSourceModule(script, dir, name),
        getPrologueFile("prologue.js")
    };
    return modules;
  }

  public static HashSet<String> getBuggyLines(String dir, String name)
      throws IOException, IllegalArgumentException, CancelException, WalaException,
      InvalidClassFileException {
    return getBuggyLines(
        dir, name, CGBuilderType.ONE_CFA, JSCallGraphBuilderUtil.class.getClassLoader());
  }

  public static boolean doesFieldExist(String fieldName, OrdinalSet<? extends InstanceKey> pts, HeapGraph hg) {
    for (InstanceKey ik : pts) {
      if (ik.toString().contains("prologue.js") && ik.toString().contains("Object in Everywhere")) {
        Iterator<PointerKey> succs = hg.getSuccNodes(ik);
        while (succs.hasNext()) {
          PointerKey pk_field = succs.next();
          if (pk_field instanceof InstanceFieldKey) {
            InstanceFieldKey fpk = (InstanceFieldKey) pk_field;
            IField field = fpk.getField();
            if (field.getName().toString().equals(fieldName)) {
              return true;
            }
          }
        }
      }
    }
    for (InstanceKey ik : pts) {
      if (ik instanceof ConstantKey) {
        continue;
      }
      if (ik.toString().contains("prologue.js") && ik.toString().contains("Object in Everywhere")) {
        continue;
      }
      Iterator<PointerKey> succs = hg.getSuccNodes(ik);
      while (succs.hasNext()) {
        PointerKey pk_field = succs.next();
        if (pk_field instanceof InstanceFieldKey) {
          InstanceFieldKey fpk = (InstanceFieldKey) pk_field;
          IField field = fpk.getField();
          if (field.getName().toString().equals(fieldName)) {
            return true;
          }
        }
      }
    }
    return false;
  }

  public static HashSet<String> getBuggyLines(
      String dir, String name, CGBuilderType builderType, ClassLoader loader)
      throws IOException, IllegalArgumentException, CancelException, WalaException, InvalidClassFileException {
    HashSet<String> buggyLines = new HashSet<>();
    PropagationCallGraphBuilder b = makeScriptCGBuilder(dir, name, builderType, loader);
    CallGraph CG = b.makeCallGraph(b.getOptions());
    PointerAnalysis<?> PA = b.getPointerAnalysis();
    HeapModel HM = PA.getHeapModel();
    HeapGraph HG = PA.getHeapGraph();
    for (CGNode node : CG) {
      IR ir = node.getIR();
      if (ir == null)
        continue;
      IMethod method = node.getMethod();

      // Normal get instructions eg. obj.field
      for (SSAInstruction instr : ir.getInstructions()) {
        if (instr == null)
          continue;
        SourcePosition sourcePos = method.getSourcePosition(instr.iIndex());
        int line = (sourcePos != null) ? sourcePos.getFirstLine() : -1;
        if (instr instanceof SSAGetInstruction) {
          SSAGetInstruction get = (SSAGetInstruction) instr;
          String field = get.getDeclaredField().getName().toString();

          if (line != -1 && !sourcePos.toString().contains("prologue")) {
            List<String> lines = Files.readAllLines(Paths.get(dir, name));
            if (line <= lines.size()) {
              int objVar = get.getRef(); // e.g., v2
              int maxValue = node.getIR().getSymbolTable().getMaxValueNumber();
              if (objVar > maxValue || objVar < 0 || node.getIR().getSymbolTable().isConstant(objVar) == true) {
                continue;
              }
              PointerKey pk = HM.getPointerKeyForLocal(node, objVar);
              OrdinalSet<? extends InstanceKey> pts = PA.getPointsToSet(pk);

              if (!doesFieldExist(field, pts, HG)) {
                String bugString = "Bug on line " + line + ": " + field + " does not exist in object v" + objVar;
                buggyLines.add(bugString);
              }
            }
          }
        }
        // Dynamic property reads eg. obj[f]
        if (instr instanceof JavaScriptPropertyRead) {
          JavaScriptPropertyRead read = (JavaScriptPropertyRead) instr;
          int base = read.getObjectRef(); // the object (`obj`)
          int property = read.getMemberRef(); // the property name or variable holding it
          SymbolTable symtab = ir.getSymbolTable();
          String resolvedProp = null;
          if (symtab.isConstant(property) && symtab.getConstantValue(property) instanceof String) {
            resolvedProp = (String) symtab.getConstantValue(property);
          }

          if (line != -1 && !sourcePos.toString().contains("prologue")) {
            PointerKey pk = HM.getPointerKeyForLocal(node, base);
            OrdinalSet<? extends InstanceKey> pts = PA.getPointsToSet(pk);

            if (resolvedProp == null) {
              String bugString = "Bug on line " + line + ": Dynamic property read with no resolved property name";
              buggyLines.add(bugString);
            } else {
              if (!doesFieldExist(resolvedProp, pts, HG)) {
                String bugString = "Bug on line " + line + ": Dynamic property read with property name '" + resolvedProp
                    + "' does not exist in object v" + base;
                buggyLines.add(bugString);
              }
            }
          }
        }

      }
    }
    // dumpCG(b.getPointerAnalysis(), CG);
    return buggyLines;
  }

  protected static JSCFABuilder makeCG(
      JavaScriptLoaderFactory loaders,
      AnalysisScope scope,
      CGBuilderType builderType,
      IRFactory<IMethod> irFactory)
      throws WalaException {
    try {
      IClassHierarchy cha = makeHierarchy(scope, loaders);
      com.ibm.wala.cast.util.Util.checkForFrontEndErrors(cha);
      Iterable<Entrypoint> roots = makeScriptRoots(cha);
      JSAnalysisOptions options = makeOptions(scope, cha, roots);
      options.setHandleCallApply(builderType.handleCallApply());
      IAnalysisCacheView cache = makeCache(irFactory);
      JSCFABuilder builder = new JSZeroOrOneXCFABuilder(
          cha,
          options,
          cache,
          null,
          null,
          ZeroXInstanceKeys.ALLOCATIONS,
          builderType.useOneCFA());
      if (builderType.extractCorrelatedPairs())
        builder.setContextSelector(
            new PropertyNameContextSelector(
                builder.getAnalysisCache(), 2, builder.getContextSelector()));

      return builder;
    } catch (ClassHierarchyException e) {
      throw new RuntimeException("internal error building class hierarchy", e);
    }
  }

}