/**
 *  @function Timsort is a hybrid stable sorting algorithm, derived from merge sort and insertion sort,
 *  designed to perform well on many kinds of real-world data.
 *  It was implemented by Tim Peters in 2002 for use in the Python programming language.
 *  It is also used to sort arrays of non-primitive type in Java SE 7,
 *  on the Android platform, in GNU Octave, on V8, Swift and Rust.
 *  1) It sorts small partitions using Insertion Sort.
 *  2) Merges the partition using Merge Sort.
 *  @see [Timsort](https://en.wikipedia.org/wiki/Timsort)
 *  @param {Array} array
 */

const Timsort = function(array) {
  // Default size of a partition
  const RUN = 32
  const n = array.length
  // Sorting the partitions using Insertion Sort
  for (var i = 0; i < n; i += RUN) {
    InsertionSort(array, i, Math.min(i + RUN - 1, n - 1))
  }
  for (var size = RUN; size < n; size *= 2) {
    for (var left = 0; left < n; left += 2 * size) {
      const mid = left + size - 1
      const right = Math.min(left + 2 * size - 1, n - 1)
      Merge(array, left, mid, right)
    }
  }
  return array
}

/**
 * @function performs insertion sort on the partition
 * @param {Array} array array to be sorted
 * @param {Number} left left index of partition
 * @param {Number} right right index of partition
 */

const InsertionSort = function(array, left, right) {
  for (var i = left + 1; i <= right; i++) {
    const key = array[i]
    var j = i - 1
    while (j >= left && array[j] > key) {
      array[j + 1] = array[j]
      j--
    }
    array[j + 1] = key
  }
}
/**
 * @function merges two sorted partitions
 * @param {Array} array array to be sorted
 * @param {Number} left left index of partition
 * @param {Number} mid mid index of partition
 * @param {Number} right right index of partition
 */

const Merge = function(array, left, mid, right) {
  if (mid >= right) return
  const len1 = mid - left + 1
  const len2 = right - mid
  const larr = Array(len1)
  const rarr = Array(len2)
  for (var i = 0; i < len1; i++) {
    larr[i] = array[left + i]
  }
  for (var i = 0; i < len2; i++) {
    rarr[i] = array[mid + 1 + i]
  }
  var i = 0
  var j = 0
  var k = left
  while (i < larr.length && j < rarr.length) {
    if (larr[i] < rarr[j]) {
      array[k++] = larr[i++]
    } else {
      array[k++] = rarr[j++]
    }
  }
  while (i < larr.length) {
    array[k++] = larr[i++]
  }
  while (j < rarr.length) {
    array[k++] = rarr[j++]
  }
}

/**
 * @example Test of Timsort functions.
 * Data is randomly generated.
 * Return "RIGHT" if it works as expected,
 * otherwise "FAULTY"
 */
const demo = function()  {
  const size = parseInt(document.getElementById("n"));
  const data = [];
  for (var i = 0; i < size; i++) {
    data[i] = 5; 
  }
  for (var i = 0; i < size; i++) {
    data[i] = 10;
  }
  const isSorted = function (array) {
    const n = array.length
    for (var i = 0; i < n - 1; i++) {
      if (array[i] > array[i + 1]) return false
    }
    return true
  }
  Timsort(data)
}

demo();