import glob

import sys
import os
import json
from datetime import datetime
from concurrent.futures import ProcessPoolExecutor, as_completed, ThreadPoolExecutor
from tqdm import tqdm
import traceback
import argparse
from filelock import FileLock
from openai import OpenAI
import time
from multiprocessing import Process, Queue, cpu_count
import multiprocessing

# ctx = multiprocessing.get_context("spawn")


def run_check_on_file(filename, out_file, use_agent):
    from absint_ai.Interpreter import AbsIntAI

    lock = FileLock(output_path + ".lock")
    try:
        check = AbsIntAI.AbsIntAI(
            global_simplify_method=simplify_method,
            simplify_method=simplify_method,
            model=model,
            check_pointers=False,
            log_console=False,
            timeout=30,
            loop_max=100,
            use_agent=use_agent,
        )
        # if single_iteration:
        #    check.set_test_mode()
        api_key = os.environ["OPENAI_API_KEY"]
        if args.openai:
            check.initialize_openai(openai_key=api_key, base_url=None)
        else:
            check.initialize_openai("EMPTY", f"http://localhost:{port}/v1")
        start_time = time.time()
        print(f"Running check on {filename}")
        check.run(filename)
        print(f"Finished check on {filename}")
        end_time = time.time()
        runtime = end_time - start_time
        result_for_file = {}
        result_for_file["num_bugs"] = len(list(set(check.buggy_line_numbers)))
        result_for_file["buggy_lines"] = list(set(check.buggy_line_numbers))
        result_for_file["llm_time"] = check.llm_time
        result_for_file["runtime"] = runtime
        result_for_file["filename"] = filename
        with lock:
            with open(out_file, "a") as f:
                f.write(json.dumps(result_for_file) + "\n")
    except Exception as e:
        tb = traceback.format_exc()

        print(f"Error in {filename}: {type(e).__name__}")
        result_for_file = {}
        result_for_file["BUG"] = str(tb)
        if check is not None and hasattr(check, "buggy_line_numbers"):
            result_for_file["num_bugs"] = len(list(set(check.buggy_line_numbers)))
            result_for_file["buggy_lines"] = list(set(check.buggy_line_numbers))
        result_for_file["filename"] = filename
        with lock:
            with open(out_file, "a") as f:
                f.write(json.dumps(result_for_file) + "\n")


def test_benchmarks_old(benchmarks_dir, output_dir, use_agent, num_workers):
    all_filenames = glob.glob(os.path.join(benchmarks_dir, "*.js"))
    for i in range(1):
        single_iteration = i % 2 == 0
        results = {}

        dt = datetime.now().strftime("%m_%d_%H_%M")
        output_path = os.path.join(
            os.path.dirname(__file__), output_dir, f"llm_results_{dt}.jsonl"
        )
        bar = tqdm(
            total=len(all_filenames), desc=f"Number of files handled, iteration {i}"
        )
        with ProcessPoolExecutor(max_workers=num_workers) as executor:
            futures = []
            for filename in all_filenames:
                futures.append(
                    executor.submit(run_check_on_file, filename, output_path, use_agent)
                )
            for future in as_completed(futures):
                bar.update(1)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Run the nullcheck analysis")
    parser.add_argument(
        "--simplify_method",
        type=str,
        default="file",
        help="The method to simplify the script",
    )
    parser.add_argument(
        "--model",
        type=str,
        default=None,
        help="The model to use for the analysis",
    )
    parser.add_argument(
        "--port", type=int, default=8000, help="The port to run the server"
    )

    parser.add_argument(
        "--openai", action="store_true", help="Use OpenAI API", default=False
    )

    parser.add_argument(
        "--num_workers", type=int, default=4, help="The number of workers to use"
    )

    parser.add_argument(
        "--timeout", type=int, default=60, help="The timeout for each file in minutes"
    )

    parser.add_argument(
        "--no_agent",
        action="store_true",
        help="Do not use the agent to run the analysis",
        default=False,
    )
    args = parser.parse_args()
    use_agent = not args.no_agent
    benchmarks_dir = "/scratch/mi27950/absint_ai/benchmarks/all_benchmarks_no_let"
    simplify_method = args.simplify_method
    model = args.model
    port = args.port
    if model is None:
        output_dir = os.path.join(os.path.dirname(__file__), "ablations")
    elif "/" in model:
        output_dir = os.path.join(os.path.dirname(__file__), model.split("/")[1])
    else:
        output_dir = os.path.join(os.path.dirname(__file__), model)
    if args.no_agent:
        output_dir = output_dir + "_no_agent"
    output_dir = output_dir + "_" + simplify_method
    os.makedirs(output_dir, exist_ok=True)
    dt = datetime.now().strftime("%m_%d_%H_%M")
    output_path = os.path.join(output_dir, f"llm_results_{dt}.jsonl")
    test_benchmarks_old(
        benchmarks_dir, output_dir, use_agent=use_agent, num_workers=args.num_workers
    )
