from absint_ai.Environment.types.Type import *
from absint_ai.Environment.memory.RecordResult import RecordResult
from absint_ai.utils.Util import *
from absint_ai.Environment.agents.merging import *
from absint_ai.utils.Logger import logger
import absint_ai.Environment.features.functions as functions
from typing import TYPE_CHECKING
from ordered_set import OrderedSet
import gc
import debugpy
import copy

if TYPE_CHECKING:
    from absint_ai.Environment.Environment import Environment


def check_no_pointers_from_abstract_to_concrete(env: "Environment") -> None:
    # logger.info("CHECKING!!!")
    for addr in env.abstract_heap.addresses():
        obj = env.abstract_heap.get(addr)
        for field, record_result in obj.items():
            if field == "__meta__":
                if "__parent__" in record_result:
                    parent_address = record_result["__parent__"]
                    if (
                        parent_address
                        and isinstance(parent_address, Address)
                        and parent_address.get_addr_type() == "concrete"
                    ):
                        raise Exception(
                            f"Found a pointer from abstract {addr} to concrete {parent_address}. {obj}"
                        )
                continue
            for value in record_result.get_all_values():
                if isinstance(value, Address) and value.get_addr_type() == "concrete":
                    raise Exception(
                        f"Found a pointer from abstract {addr} to concrete {value}. {obj}"
                    )
                    å


def add_exported_identifier(
    env: "Environment", file_path: str, identifier: str
) -> None:
    logger.info(f"Adding exported identifier {identifier} from {file_path}")
    if file_path not in env.exported_identifiers:
        env.exported_identifiers[file_path] = []
    values = env.lookup(identifier).get_all_values()
    for value in values:
        if isinstance(value, Address) and env.get_object_type(value) == "function":
            obj = env.lookup_address(value)
            obj["__meta__"]["file_path"] = file_path
    env.exported_identifiers[file_path].append((identifier, values))


def add_import(env: "Environment", file_path: str, identifier: str) -> None:
    logger.info(f"Adding import {identifier} from {file_path}")
    exported_identifiers_from_file = env.exported_identifiers.get(file_path, [])
    for exported_identifier, values in exported_identifiers_from_file:
        if exported_identifier == identifier:
            env.add(identifier, values)
            return
    raise Exception(
        f"Could not find identifier {identifier} in file {file_path}. exported identifiers: {exported_identifiers_from_file}"
    )


# simple mark and sweep garbage collection
def garbage_collect(
    env: "Environment", expr_loc: str = None, ignore_memoized=False
) -> None:
    # Recurse until nothing is removed
    removed_something = True
    iteration = 0
    while removed_something:
        iteration += 1
        removed_something = False
        reachable_concrete_addresses: OrderedSet[str] = OrderedSet()

        def get_reachable_addresses(addr: Address, add_to_reachable=True) -> None:
            if (
                addr
                and addr.get_addr_type() == "concrete"
                and addr not in reachable_concrete_addresses
                and addr.get_value() not in reachable_concrete_addresses
            ):
                if add_to_reachable:
                    reachable_concrete_addresses.add(addr.get_value())
                heap_frame = env.concrete_heap.get(addr)
                for key in heap_frame:
                    if key == "__meta__" and "__parent__" in heap_frame[key]:
                        parent_address = heap_frame[key]["__parent__"]
                        if parent_address and isinstance(parent_address, Address):
                            get_reachable_addresses(parent_address)
                    elif key != "__meta__":
                        for value in heap_frame[key].get_all_values():
                            if isinstance(value, Address):
                                get_reachable_addresses(value)
                    if addr.get_parent_heap_frame():
                        get_reachable_addresses(addr.get_parent_heap_frame())

        for module_name, stack in env.stack.items():
            for stack_frame in stack:
                heap_frame_addr = stack_frame.get_heap_frame_address()
                get_reachable_addresses(heap_frame_addr)
                stack_variables = stack_frame.get_variable_names()
                for var_name in stack_variables:
                    record_result = stack_frame.get_variable(var_name)
                    for value in record_result.get_all_values():
                        if isinstance(value, Address):
                            get_reachable_addresses(value)
                for allocated_address in stack_frame.get_allocated_addresses():
                    get_reachable_addresses(allocated_address)
                for return_value in stack_frame.get_return_values():
                    if isinstance(return_value, Address):
                        get_reachable_addresses(return_value)
        for func_address in env.entrypoint_functions:
            get_reachable_addresses(func_address)
        if not ignore_memoized:
            # Need to look through currently allocated addresses
            for memoized_function in env.memoized_functions:
                touched_addresses = memoized_function.get_touched_address_values()
                if touched_addresses:
                    for addr in touched_addresses:
                        get_reachable_addresses(addr)
                allocated_addresses = memoized_function.get_allocated_addresses()
                if allocated_addresses:
                    for addr in allocated_addresses:
                        get_reachable_addresses(addr)
                return_values = memoized_function.get_return_values()
                if return_values:
                    for return_value in return_values:
                        if isinstance(return_value, Address):
                            get_reachable_addresses(return_value)
        # for record_result in env.get_all_variable_record_results():
        #    for value in record_result.get_all_values():
        #        if isinstance(value, Address):
        #            get_reachable_addresses(value)
        addrs_to_remove = [
            addr.get_value()
            for addr in env.concrete_heap.addresses()
            if addr.get_value() not in reachable_concrete_addresses
        ]
        # logger.info(f"REMOVING ADDRESSES: {addrs_to_remove}")
        for (
            allocation_site_id,
            allocation_site,
        ) in env.allocation_sites.items():
            concrete_addresses_to_remove = copy.deepcopy(
                [
                    concrete_address
                    for concrete_address in allocation_site["concrete_addresses"]
                    if concrete_address.get_value() not in reachable_concrete_addresses
                    and concrete_address.get_addr_type() == "concrete"
                ]
            )
            allocation_site["concrete_addresses"] = OrderedSet(
                [
                    _
                    for _ in allocation_site["concrete_addresses"]
                    if _ not in concrete_addresses_to_remove
                ]
            )
        for module_name, stack in env.stack.items():
            for stack_frame in stack:
                touched_addresses_to_remove = copy.deepcopy(
                    [
                        touched_address
                        for touched_address in stack_frame.get_touched_addresses()
                        if touched_address.get_value()
                        not in reachable_concrete_addresses
                        and touched_address.get_addr_type() == "concrete"
                    ]
                )
                for touched_address in touched_addresses_to_remove:
                    stack_frame.remove_touched_address(touched_address)
        for addr_to_remove in addrs_to_remove:
            removed_something = True
            # logger.info(
            #    f"Popping {addr_to_remove} from concrete heap in garbage_collect"
            # )
            env.concrete_heap.pop(Address(value=addr_to_remove, addr_type="concrete"))
    gc.collect()


def move_all_objects_to_abstract_heap(env: "Environment") -> None:
    existing_addresses = env.concrete_heap.addresses()
    for address in existing_addresses:
        # Because move_object_to_abstract_heap runs recursively it could remove other addresses
        if address in env.concrete_heap.addresses():
            env.move_object_to_abstract_heap(address)


# Need to do this recursively to make sure there's no pointers from the abstract heap to the concrete heap
def move_object_to_abstract_heap(
    env: "Environment", concrete_address: Address, count=0, seen=None
) -> None:
    if seen is None:
        seen = []
    if concrete_address in seen:
        return
    if concrete_address.get_addr_type() == "abstract":
        return
    seen.append(concrete_address)
    original_address = copy.deepcopy(concrete_address)
    concrete_obj = env.lookup_address(concrete_address)
    for key in concrete_obj:
        if key == "__meta__":
            if "__parent__" in concrete_obj[key]:
                parent = concrete_obj[key]["__parent__"]
                if isinstance(parent, Address) and parent != concrete_address:
                    move_object_to_abstract_heap(env, parent, count + 1, seen=seen)
            continue
        assert isinstance(concrete_obj[key], RecordResult)
        concrete_obj[key].set_type("global")
        for obj in concrete_obj[key]:
            if (
                isinstance(obj, Address)
                and obj.get_addr_type() == "concrete"
                and obj != concrete_address
            ):
                move_object_to_abstract_heap(env, obj, count + 1, seen=seen)
    if concrete_address.get_parent_heap_frame():
        parent_heap_frame = concrete_address.get_parent_heap_frame()
        if (
            parent_heap_frame.get_addr_type() == "concrete"
            and parent_heap_frame != concrete_address
        ):
            move_object_to_abstract_heap(env, parent_heap_frame, count + 1, seen=seen)
    if "__meta__" not in concrete_obj:
        raise Exception(f"__meta__ not found in {concrete_obj}")
    abstract_id = env.abstract_heap.add(concrete_obj)
    # logger.info(f"Moving {concrete_address} to abstract heap with id {abstract_id}")
    concrete_address.set_value(abstract_id)
    concrete_address.set_addr_type("abstract")
    env.abstract_addresses(
        [original_address], Address(value=abstract_id, addr_type="abstract")
    )


def value_contained_in_record_result(
    env: "Environment", value_as_type: Type, record_result: RecordResult
) -> bool:
    value = env.type_to_value(value_as_type)
    all_values = [env.type_to_value(val) for val in record_result.get_all_values()]
    return item_contained_in_list(value, all_values)


def get_all_variable_names(env: "Environment") -> list[str]:
    variables: list[str] = []
    for stack_frames_for_module in env.stack.values():
        for stack_frame in stack_frames_for_module:
            for key in stack_frame.get_variable_names():
                variables.append(key)
            for key in env.lookup_address(stack_frame.get_heap_frame_address()):
                if key != "__meta__" and key != "__proto__":
                    variables.append(key)
    return variables


def get_reachable_allocation_sites(
    env: "Environment", variable_names: list[str], ignore_functions=False
) -> OrderedSet[str]:
    # only want objects and functions, ignore heap frames and classes
    def get_reachable_object_addresses(
        address: Address, found_addresses, seen
    ) -> set[Address]:
        if address in seen:
            return
        seen.add(address)
        if env.get_object_type(address) == "object":
            found_addresses.add(address)
        if not ignore_functions and env.get_object_type(address) == "function":
            found_addresses.add(address)
        obj = env.lookup_address(address)
        for field, record_result in obj.items():

            if field == "__meta__":
                if "__parent__" in record_result:
                    parent_address = record_result["__parent__"]
                    if parent_address and isinstance(parent_address, Address):
                        get_reachable_object_addresses(
                            parent_address, found_addresses, seen
                        )
                continue
            for value in record_result.get_all_values():
                if isinstance(value, Address) and value not in found_addresses:
                    get_reachable_object_addresses(value, found_addresses, seen)

    all_reachable_addresses = OrderedSet()
    seen = OrderedSet()
    all_allocation_sites = OrderedSet()
    for var_name in variable_names:
        record_result = env.lookup(var_name)
        for value in record_result.get_all_values():
            if isinstance(value, Address):
                get_reachable_object_addresses(value, all_reachable_addresses, seen)
    for addr in all_reachable_addresses:
        allocation_site_for_addr = env.get_allocation_site(addr)
        if allocation_site_for_addr:
            all_allocation_sites.add(allocation_site_for_addr)
    return all_allocation_sites


def get_all_variable_values(env: "Environment") -> dict[str : list[Type]]:
    variables: dict[str : list[Type]] = {}
    for stack_frames_for_module in env.stack.values():
        for stack_frame in stack_frames_for_module:
            for key in stack_frame.get_variable_names():
                variables[key] = stack_frame.get_variable(key).get_all_values()
            heap_frame = env.lookup_address(stack_frame.get_heap_frame_address())
            for key in heap_frame:
                if key != "__meta__" and key != "__proto__":
                    variables[key] = heap_frame[key].get_all_values()
    return variables


# Can determine if we are in a subroutine by how many stack frames are pus
def in_subroutine(env: "Environment", file_path: str) -> bool:
    file_path = convert_path_to_underscore(file_path)
    stack_for_file = env.stack[file_path]
    for stack_frame in stack_for_file:
        if stack_frame.is_function:
            return True
    return False


# Gets all record results that are pointed to by variables.
# get_all_variable_names doesn't handle same variable names across different scopes
def get_all_variable_record_results(env: "Environment") -> list[RecordResult]:
    all_record_results: list[RecordResult] = []
    for stack_frames_for_module in env.stack.values():
        for stack_frame in stack_frames_for_module:
            for key in stack_frame.get_variable_names():
                all_record_results.append(stack_frame.get_variable(key))
            for key in env.lookup_address(stack_frame.get_heap_frame_address()):
                if key != "__meta__" and key != "__proto__":
                    all_record_results.append(
                        env.lookup_address(stack_frame.get_heap_frame_address())[key]
                    )
    return all_record_results


def get_all_reachable_object_variable_names(env: "Environment") -> OrderedSet[str]:
    variable_names: OrderedSet[str] = []
    to_ignore = ["document", "element", "module"]
    for key in env.cur_stack_frame.get_variable_names():
        if key in to_ignore:
            continue
        for value in env.cur_stack_frame.get_variable(key).get_all_values():
            if (
                isinstance(value, Address) and env.get_object_type(value) == "object"
            ) and value not in env.builtin_addresses:
                variable_names.append(key)
                break
            elif isinstance(value, Primitive):
                variable_names.append(key)
                break
    cur_heap_frame = env.lookup_address(env.cur_stack_frame.get_heap_frame_address())
    cur_heap_frame_address = env.cur_stack_frame.get_heap_frame_address()
    while True:
        for key in cur_heap_frame:  # type: ignore
            if key != "__meta__" and key not in to_ignore:
                for value in cur_heap_frame[key].get_all_values():
                    if (
                        isinstance(value, Address)
                        and env.get_object_type(value) == "object"
                    ):
                        variable_names.append(key)
                        break
                    elif isinstance(value, Primitive):
                        variable_names.append(key)
                        break
        if "__parent__" not in cur_heap_frame["__meta__"]:
            logger.info(
                f"__parent__ not found in {cur_heap_frame}, address {cur_heap_frame_address}"
            )
            raise Exception(
                f"__parent__ not found in {cur_heap_frame}, address {cur_heap_frame_address}"
            )
        if not cur_heap_frame["__meta__"]["__parent__"]:
            break
        cur_heap_frame = env.lookup_address(cur_heap_frame["__meta__"]["__parent__"])  # type: ignore
        cur_heap_frame_address = cur_heap_frame["__meta__"]["__parent__"]
    return variable_names


def get_all_reachable_object_variable_values(
    env: "Environment", variable_names: OrderedSet[str]
) -> dict[str, list[Type]]:
    variable_values: dict[str, list] = {}
    for variable_name in variable_names:
        variable_values[variable_name] = env.lookup_and_derive(variable_name)
    return variable_values


def get_all_reachable_variable_names(env: "Environment") -> list[str]:
    variables: list[str] = []
    for key in env.cur_stack_frame.get_variable_names():
        for value in env.cur_stack_frame.get_variable(key).get_all_values():
            if (
                (isinstance(value, Address) and env.get_object_type(value) == "object")
                and value not in env.builtin_addresses
            ) or isinstance(value, Primitive):
                # logger.info(f"Found variable {key} with value {value}")
                variables.append(key)
                break
    cur_heap_frame = env.lookup_address(env.cur_stack_frame.get_heap_frame_address())
    cur_heap_frame_address = env.cur_stack_frame.get_heap_frame_address()
    while True:
        for key in cur_heap_frame:  # type: ignore
            if key != "__meta__":
                for value in cur_heap_frame[key].get_all_values():
                    if (
                        isinstance(value, Address)
                        and env.get_object_type(value) == "object"
                    ):
                        # logger.info(f"Found variable {key} with value {value}")
                        variables.append(key)
                        break
                    elif isinstance(value, Primitive):
                        variables.append(key)
                        break
        if "__parent__" not in cur_heap_frame["__meta__"]:
            logger.info(
                f"__parent__ not found in {cur_heap_frame}, address {cur_heap_frame_address}"
            )
            raise Exception(
                f"__parent__ not found in {cur_heap_frame}, address {cur_heap_frame_address}"
            )
        if not cur_heap_frame["__meta__"]["__parent__"]:
            break
        cur_heap_frame = env.lookup_address(cur_heap_frame["__meta__"]["__parent__"])  # type: ignore
        cur_heap_frame_address = cur_heap_frame["__meta__"]["__parent__"]
    # logger.info(f"Found variables: {variables}")
    return variables


def value_to_type(
    env: "Environment", value: object, abstract: bool = False, allocation_site=None
) -> Type:
    # import faulthandler

    # faulthandler.enable()
    if isinstance(value, list) or isinstance(value, OrderedSet):
        return [env.value_to_type(val) for val in value]  # type: ignore
    if isinstance(
        value, dict
    ):  # this is the weirdest fucking bug ever. At some point it loses the __eq__ method if it's from pythonmonkey...
        value = dict(value)
    if value == None or value == "null":
        return baseType.NULL
    if value == "TOP":
        return baseType.TOP
    elif value == "STRING":
        return baseType.STRING
    elif value == "NUMBER":
        return baseType.NUMBER
    elif (isinstance(value, str) and value.lower() == "null") or value == None:
        return baseType.NULL
    elif isinstance(value, str) and (
        value.lower() == "bool" or value.lower() == "boolean"
    ):
        return baseType.BOOLEAN
    elif isinstance(value, (int, float, str, bool)):
        return Primitive(value)
    elif isinstance(value, dict):
        result = {}
        for prop in value:
            result[prop] = env.value_to_type(value[prop])
        if abstract:
            return env.add_object_to_abstract_heap(
                result, allocation_site=allocation_site
            )
        return env.add_object_to_heap(
            result, allocation_site=allocation_site
        )  # This is an abstraction that's generated by an LLM, so no allocation site
    else:
        raise Exception(f"Unknown value {value}")


def type_to_value(
    env: "Environment", type: Type, seen: set = None, log=False
) -> object:
    if seen is None:
        seen = set()
    if isinstance(type, Address):
        if type in seen:
            return type
        seen.add(type)
    type_value = type.get_value()
    if isinstance(type, baseType):
        return type
    elif isinstance(type, Primitive):
        return type_value
    elif isinstance(type, Address):
        result: dict[object, list] = {}
        if env.get_object_type(type) == "function":
            return f"function({type})"
        addr_type = type.get_addr_type()
        if addr_type == "concrete":
            if not env.concrete_heap.contains(type_value):
                raise Exception(
                    f"Address {type_value} not found in the concrete heap"
                )
            heap_val = env.concrete_heap.get(type)
        elif addr_type == "abstract":
            if not env.abstract_heap.contains(type_value):
                raise Exception(
                    f"Address {type.get_value()} not found in the abstract heap"
                )
            heap_val = env.abstract_heap.get(type)
        else:
            raise Exception(f"Unknown address type {addr_type}")
        for key in heap_val:
            if key == "__proto__" or key == "__meta__":
                continue
            if isinstance(key, AbstractType):
                key_to_add = key.get_value()
            else:
                key_to_add = key
            """
            result[key_to_add] = []
            for val in heap_val[key]:
                if val == type:
                    result[key_to_add].append(str(val))
                else:
                    result[key_to_add].append(type_to_value(env, val, seen=seen))
            """
            result[key_to_add] = [
                str(val) if val == type else type_to_value(env, val, seen=seen)
                for val in heap_val[key]
            ]
        return result
    elif isinstance(type, AbstractType):
        return type_value
    else:
        raise Exception(f"Unknown type {type}")


def snapshot(env: "Environment") -> dict:
    def type_to_value_for_snapshot(type: Type, seen: list = None) -> object:
        if seen is None:
            seen = set()
        if isinstance(type, Address) and type in seen:
            return str(type)
        if isinstance(type, Address):
            seen.add(type)
        if isinstance(type, Primitive):
            return type.get_value()
        elif isinstance(type, AbstractType):
            return str(type.get_value())
        elif isinstance(type, Address):
            result: dict[object, dict | OrderedSet] = {}
            obj = env.lookup_address(type)
            for field, values in obj.items():
                if field == "__meta__":
                    continue
                result[str(field)] = {}
                for value in values.get_all_values():
                    value_key = str(value)
                    result[str(field)][value_key] = type_to_value_for_snapshot(
                        value, seen=seen
                    )
            return result
        else:
            raise Exception(f"Unknown type {type}")

    all_vars = env.get_all_reachable_variable_names()
    snapshot = {}
    for var in all_vars:
        try:
            record_result = env.lookup(var, add_touched=False)
            snapshot[var] = {
                str(val): type_to_value_for_snapshot(val)
                for val in record_result.get_all_values()
            }
        except Exception as e:
            logger.info(
                f"Failed to get snapshot for {var}, values are {env.lookup(var, add_touched=False)}"
            )
            raise e
    return snapshot


def update_address_mappings_in_snapshot(
    env: "Environment", snapshot: dict, depth=0
) -> dict:
    updated_snapshot = {}
    for key in snapshot:
        if key in env.abstracted_addresses_map:
            updated_key = env.abstracted_addresses_map[key]
        else:
            updated_key = key
        updated_snapshot[updated_key] = {}
        for value in snapshot[key]:
            if value in env.abstracted_addresses_map:
                updated_value = env.abstracted_addresses_map[value]
            else:
                updated_value = value
            if isinstance(snapshot[key][value], dict):
                updated_snapshot[updated_key][updated_value] = (
                    update_address_mappings_in_snapshot(
                        env, snapshot[key][value], depth + 1
                    )
                )
            elif value in env.abstracted_addresses_map:
                updated_snapshot[updated_key][updated_value] = (
                    env.abstracted_addresses_map[value]
                )
            else:
                updated_snapshot[updated_key][value] = snapshot[key][value]
    return updated_snapshot


def allocation_sites_snapshot(env: "Environment", only_abstract=False) -> dict:
    snapshot = {}
    reachable_allocation_sites = get_reachable_allocation_sites(
        env, env.get_all_reachable_variable_names()
    )
    for allocation_site in reachable_allocation_sites:
        snapshot[allocation_site] = {}
        summary_addresses = env.allocation_sites[allocation_site]["summary_addresses"]
        snapshot[allocation_site]["summary_addresses"] = {
            str(abstract_address): env.lookup_and_derive_address(
                abstract_address, log=True
            )
            for abstract_address in summary_addresses
        }
        if only_abstract:
            continue
        concrete_values = {
            str(addr): env.lookup_and_derive_address(addr, log=True)
            for addr in env.allocation_sites[allocation_site]["concrete_addresses"]
        }
        snapshot[allocation_site]["concrete_addresses"] = concrete_values
    return snapshot


def get_allocation_site_values(
    env: "Environment", allocation_sites: list[str], ignore_heap_frames=True
) -> dict:
    allocation_site_values = {}
    for allocation_site in allocation_sites:
        if (
            env.allocation_sites[allocation_site]["type"] == "heap_frame"
            and ignore_heap_frames
        ):
            continue
        allocation_site_values[allocation_site] = {}
        summary_addresses = env.allocation_sites[allocation_site]["summary_addresses"]
        allocation_site_values[allocation_site]["summary_addresses"] = {
            str(abstract_address): env.lookup_and_derive_address(
                abstract_address, log=True
            )
            for abstract_address in summary_addresses
        }
        concrete_values = {
            str(addr): env.lookup_and_derive_address(addr, log=True)
            for addr in env.allocation_sites[allocation_site]["concrete_addresses"]
        }
        allocation_site_values[allocation_site]["concrete_addresses"] = concrete_values

    return allocation_site_values


def changed_allocation_sites_from_snapshot(
    env: "Environment", allocation_sites_snapshot: dict, log: bool = True
) -> list:
    cur_allocation_sites_snapshot = env.allocation_sites_snapshot()
    changed = []
    for allocation_site in cur_allocation_sites_snapshot:
        if (
            allocation_site not in allocation_sites_snapshot
            or cur_allocation_sites_snapshot[allocation_site]
            != allocation_sites_snapshot[allocation_site]
        ):
            changed.append(allocation_site)
    return changed


def get_changed_object_allocation_sites_from_snapshot(
    env: "Environment", previous_allocation_sites_snapshot: dict, loop_id: str
) -> list:
    changed_allocation_site_ids = []
    cur_allocation_sites_snapshot = env.allocation_sites_snapshot()
    for allocation_site_id in cur_allocation_sites_snapshot:
        readable_allocation_site = env.get_readable_allocation_site(allocation_site_id)
        allocation_site_snapshot = cur_allocation_sites_snapshot[allocation_site_id]
        allocation_site = env.allocation_sites[
            allocation_site_id
        ]  # the current allocation site object, without anything derived
        if allocation_site["type"] == "heap_frame":
            continue
        if allocation_site_id not in previous_allocation_sites_snapshot:
            changed_allocation_site_ids.append(allocation_site_id)
            continue
        if (
            allocation_site_snapshot
            != previous_allocation_sites_snapshot[allocation_site_id]
        ):
            # logger.info(f"checking if {readable_allocation_site} has changed")
            # this means the LLM has not come up with a strategy for this but it's changing
            if loop_id not in allocation_site["merging_strategies"]:
                # logger.info(f"Allocation site {allocation_site_id} changed")
                changed_allocation_site_ids.append(allocation_site_id)
            else:
                merging_strategy: MergeStrategy = allocation_site["merging_strategies"][
                    loop_id
                ]
                if not merging_strategy.has_converged(
                    env,
                    previous_allocation_sites_snapshot[allocation_site_id],
                    allocation_site_snapshot,
                ):
                    changed_allocation_site_ids.append(allocation_site_id)
                    # logger.info(f"Allocation site {env.get_readable_allocation_site(allocation_site_id)} changed from:\n {previous_allocation_sites_snapshot[allocation_site_id]} to \n{allocation_site_snapshot}")
    return changed_allocation_site_ids


def changed_from_snapshot(
    env: "Environment", snapshot: dict, log: bool = True, primitives_only=False
) -> list:
    # We need to check if the old snapshot contained abstracted addresses and update them
    last_updated_snapshot = None
    updated_snapshot = copy.deepcopy(snapshot)
    count = 0
    while True:
        count += 1
        if count > 100:
            raise Exception("Infinite loop in changed_from_snapshot")
        updated_snapshot = update_address_mappings_in_snapshot(env, updated_snapshot)
        if updated_snapshot == last_updated_snapshot:
            break
        last_updated_snapshot = updated_snapshot
    cur_snapshot = env.snapshot()
    changed = []
    for var_name in cur_snapshot:
        if var_name == "arguments" or var_name == "element" or var_name == "document":
            continue
        if var_name not in updated_snapshot or not dicts_equal_modulo_keys(
            cur_snapshot[var_name], updated_snapshot[var_name]
        ):  # cur_snapshot[var_name] != updated_snapshot[var_name]
            if var_name == "this":
                continue
            if not env.lookup(var_name).contains_primitives() and primitives_only:
                continue
            if var_name in updated_snapshot:
                pass
                #logger.info(
                #   f"Variable {var_name} changed from {updated_snapshot[var_name]} to {cur_snapshot[var_name]}")
            changed.append(var_name)
    return changed


# region Lookup Functions


def lookup_address(
    env: "Environment", addr: Address
) -> dict[str | AbstractType, RecordResult]:
    if addr.get_addr_type() == "concrete":
        concrete_value = env.concrete_heap.get(addr)
        return concrete_value
    elif addr.get_addr_type() == "abstract":
        abstract_value = env.abstract_heap.get(addr)
        return abstract_value
    else:
        raise Exception(f"Unknown address type {addr}")


# Get all the variable names that point to this allocation site
def points_to_info_for_allocation_site(
    env: "Environment", allocation_site_id: str
) -> dict:
    allocation_site = env.allocation_sites[allocation_site_id]
    points_to_info = {}
    addresses = allocation_site["summary_addresses"].union(
        allocation_site["concrete_addresses"]
    )
    for var_name in env.get_all_variable_names():
        record_result = env.lookup(var_name)
        for value in record_result.get_all_values():
            if isinstance(value, Address) and value in addresses:
                if var_name not in points_to_info:
                    points_to_info[var_name] = OrderedSet()
                points_to_info[var_name].add(value)
    return points_to_info


def lookup(
    env: "Environment",
    name: str,
    add_touched: bool = True,
    loc: str = None,
    log: bool = False,
) -> RecordResult:
    # logger.info(f"looking for {name} in stack frame {env.cur_stack_frame}")
    if name in env.cur_stack_frame.get_variable_names():
        result = env.cur_stack_frame.get_variable(name)
        if add_touched:
            functions.add_touched_record_result(env, name, result, add_primitives=False)
        return result
    heap_frame_addr = env.cur_stack_frame.get_heap_frame_address()
    heap_frame = lookup_address(env, heap_frame_addr)
    if name in heap_frame:
        result = heap_frame[name]
        if add_touched:
            functions.add_touched_record_result(env, name, result, add_primitives=False)
        return result
    else:
        while heap_frame["__meta__"]["__parent__"]:
            heap_frame_addr = heap_frame["__meta__"]["__parent__"]  # type: ignore
            heap_frame = lookup_address(env, heap_frame_addr)
            if name in heap_frame:
                result = heap_frame[name]
                if add_touched:
                    functions.add_touched_record_result(
                        env, name, result, add_primitives=True
                    )  # We only want to take primitives that are from the parents, like global variables
                return result
    if log:
        if loc == "abstract_addresses":
            pass
        elif loc:
            logger.info(f"Variable {name} not found in environment at {loc}")
        else:
            logger.info(f"Variable {name} not found in environment")
    return RecordResult("local")


# TODO: Check that this works
def lookup_values_for_field_path(
    env: "Environment", address: Address, field_path: list[str]
) -> list[RecordResult]:
    result: list[RecordResult] = []
    obj = env.lookup_address(address)
    field = field_path[0]
    if field == "STRING":
        field = baseType.STRING
    elif field == "NUMBER":
        field = baseType.NUMBER
    elif field == "BOOLEAN":
        field = baseType.BOOLEAN
    if len(field_path) == 1:
        if field in obj:
            return [obj[field]]
    else:
        if field in obj:
            possible_values = obj[field]
            for possible_value in possible_values:
                if isinstance(possible_value, Address):
                    result += lookup_values_for_field_path(
                        env, possible_value, field_path[1:]
                    )
    return result


def lookup_values_for_depth(
    env: "Environment", address: Address, depth: int
) -> list[RecordResult]:
    obj = env.lookup_address(address)
    result: list[RecordResult] = []

    if depth == 1:
        for key, record_result in obj.items():
            if key == "__meta__" or key == "__proto__":
                continue
            result.append(record_result)
        return result
    else:
        for key, record_result in obj.items():
            if key == "__meta__" or key == "__proto__":
                continue
            for value in record_result.get_all_values():
                if isinstance(value, Address):
                    result += lookup_values_for_depth(env, value, depth - 1)
    return result


def lookup_field(
    env: "Environment",
    address: Address,
    field: Type,
    searched_addresses=None,
    line_number=None,
) -> RecordResult:
    if searched_addresses is None:
        searched_addresses = []
    if address in searched_addresses:
        return RecordResult("local")
    searched_addresses.append(address)
    obj = lookup_address(env, address)
    if isinstance(field, AbstractType):
        field_key = field
    else:
        field_key = field.get_value()
    possible_values = RecordResult("local")
    if baseType.TOP in obj:
        possible_values.merge_other_record_result(obj[baseType.TOP])
    if baseType.STRING in obj:
        if isinstance(field_key, str) or field_key == baseType.TOP:
            possible_values.merge_other_record_result(obj[baseType.STRING])
    if baseType.NUMBER in obj:
        if isinstance(field_key, (int, float)) or field_key == baseType.TOP:
            possible_values.merge_other_record_result(obj[baseType.NUMBER])
        
    if field_key in obj:
        possible_values.merge_other_record_result(obj[field_key])
        return possible_values

    if "__proto__" in obj and obj["__proto__"]:
        for proto in obj["__proto__"].get_all_values():
            if isinstance(proto, Address):
                return lookup_field(env, proto, field, searched_addresses)
    return RecordResult("local")


# endregion
