import beeprint
import os
from absint_ai.utils.Logger import logger
from typing import TYPE_CHECKING
import tiktoken
from typing import Tuple
import absint_ai.utils.Util as Util
import json
from absint_ai.Environment.types.Type import *
from absint_ai.Environment.agents.widening import *
from absint_ai.Environment.agents.merging import *

if TYPE_CHECKING:
    from absint_ai.Environment.Environment import (
        Environment,
    )  # Import the class only for type checking


def summarize_llm_loop(
    env: "Environment",
    model: str = "meta-llama/Llama-3.3-70B-Instruct",
    changed_variables: list[str] = [],
    code_window: str = "",
    loop_body: str = "",
) -> None:
    vars_to_abstract = get_variables_to_abstract_from_llm_loop(
        code=code_window,
        model=model,
        changed_variable_names=changed_variables,
        loop_body=loop_body,
        env=env,
    )
    for var_name in vars_to_abstract:
        if var_name in changed_variables:
            simplify_variable_loop(
                env,
                var_name,
                code=code_window,
                loop_body=loop_body,
                model=model,
                invoke_llm=True,
            )


def get_num_tokens(string: str) -> int:
    """Returns the number of tokens in a text string."""
    encoding = tiktoken.get_encoding("o200k_base")
    num_tokens = len(encoding.encode(string))

    return num_tokens


def get_openai_prompt(
    message: str, chat_history: list[tuple[str, str]], system_prompt: str
) -> list:
    messages = [
        {"role": "system", "content": system_prompt},
    ]
    for user_input, response in chat_history:
        messages.append({"role": "user", "content": user_input})
        messages.append({"role": "assistant", "content": response})
    messages.append({"role": "user", "content": message})
    return messages


def get_allocation_sites_to_abstract_from_human(
    env: "Environment", code: str, changed_allocation_sites: list
) -> Tuple[list[str], dict[str, str]]:

    allocation_site_values_raw = env.get_allocation_site_values(
        changed_allocation_sites, ignore_heap_frames=True
    )
    for allocation_site in allocation_site_values_raw:
        points_to_info = env.points_to_info_for_allocation_site(allocation_site)
        allocation_site_values_raw[allocation_site]["points_to"] = list(points_to_info)

    # Just to make it more readable
    allocation_site_mapping = {}
    allocation_site_mapping_backwards = {}
    for i, allocation_site in enumerate(changed_allocation_sites):
        allocation_site_mapping[allocation_site] = "alloc@" + str(i)
        allocation_site_mapping_backwards["alloc@" + str(i)] = allocation_site
    allocation_site_values = {
        allocation_site_mapping[k]: v for k, v in allocation_site_values_raw.items()
    }
    allocation_site_values_str = beeprint.pp(allocation_site_values, output=False)
    first_abstraction_message = f"""
The code is:
```javascript
{code}
```
The values for the allocation sites in JSON form so far are:
```json
{allocation_site_values_str}
```
What allocation sites should be abstracted? Separate each allocation site with backticks:

"""
    user_input = input(first_abstraction_message)
    allocation_sites_to_abstract = Util.get_variables_in_backticks(user_input)
    allocation_sites_to_abstract = list(set(allocation_sites_to_abstract))

    return allocation_sites_to_abstract, allocation_site_mapping_backwards


def get_merge_strategy_from_human(
    env: "Environment",
    code: str,
    allocation_site_from_llm: str,
    allocation_site_mapping: dict[str, str],
) -> MergeStrategy:
    allocation_site_id = allocation_site_mapping[allocation_site_from_llm]
    allocation_site_values_raw = env.get_allocation_site_values(
        [allocation_site_id], ignore_heap_frames=True
    )

    allocation_site_values = {
        allocation_site_from_llm: allocation_site_values_raw[allocation_site_id]
    }
    points_to_info = env.points_to_info_for_allocation_site(allocation_site_id)
    allocation_site_values_raw[allocation_site_id]["points_to"] = list(points_to_info)
    allocation_site_value_str = beeprint.pp(allocation_site_values, output=False)
    merge_strategy_message = f"""
You chose to abstract the following allocation site: `{allocation_site_from_llm}`. It has the following values:
```json
{allocation_site_value_str}
```
The code is:
```javascript
{code}
```

What merging strategy would you like to use for this allocation site? NOTE: right now the only one that works is the `naive`.
The possible strategies are:
- naive: simply merge all possible addresses and primitives together into one mega-object
- recency: merge all previous values of the allocation site into one object, and keep the most recent value as a separate object
- field_set: merge objects that have the same set of fields
- field_value: merge objects that have the same value for a particular field. 

Please choose only one of these strategies.
"""
    merge_strategy_str = input(merge_strategy_message)
    widening_policy_message = """
Thank you! You have chosen the following merging strategy: `{merge_strategy_str}`.
What widening strategy would you like to use for this allocation site?
The possible strategies are:
- field value: widen the value for a few particular fields. You might do this if only a few fields are growing.
- field key: merge a set of fields into one.
- depth limited: widen all values after a particular depth.
- all: widen the entire thing. Not recommended.
- none: do not widen anything. Do this if the object will converge without only merging and no widening.

Please choose one of these strategies.
"""
    widening_policy_str = input(widening_policy_message)
    widening_policy = None
    if widening_policy_str == "all":
        widening_policy = WidenAll()
    else:
        raise Exception(f"Widening policy {widening_policy_str} not supported yet.")

    if merge_strategy_str == "naive":
        merge_strategy = MergeByAllocationSite(widening_policy)
    else:
        raise Exception(f"Merging strategy {merge_strategy_str} not supported yet.")

    return merge_strategy


def get_variables_to_abstract_from_human(
    env: "Environment", code: str, changed_variables: list[str]
) -> list[str]:
    all_var_names_comma_separated = ", ".join(
        [f"`{var_name}`" for var_name in changed_variables]
    )
    all_var_values = env.get_all_reachable_object_variable_values(changed_variables)
    for var_name, var_values in all_var_values.items():
        for var_value in var_values:
            if not isinstance(var_value, Primitive):
                var_values.remove(var_value)
    all_var_values_str = beeprint.pp(all_var_values, output=False)

    first_abstraction_message = f"""
Please choose which variables you want to abstract. The code is:
```javascript
{code}
```

The variables you can abstract are: {all_var_names_comma_separated}. The values for the variables in JSON form so far are:
```json
{all_var_values_str}
```
What variables should be abstracted? Please enter a comma separated list with each variable in backticks. 
"""
    user_input = input(first_abstraction_message)
    variables_to_abstract = Util.get_variables_in_backticks(user_input)
    variables_to_abstract = [
        _.split(".")[0] for _ in variables_to_abstract
    ]  # this is a hack to remove fields
    variables_to_abstract = list(set(variables_to_abstract))
    return variables_to_abstract


# FUNCTION
def get_variables_to_abstract_from_llm(
    code: str, model: str, env: "Environment"
) -> list[str]:
    all_var_names = env.get_all_reachable_object_variable_names()
    all_var_names_comma_separated = ", ".join(
        [f"`{var_name}`" for var_name in all_var_names]
    )
    all_var_values_str = beeprint.pp(
        env.get_all_reachable_object_variable_values(all_var_names), output=False
    )
    first_abstraction_message = f"""
You are part of a static analyzer, where you are responsible for deciding which variables need to be abstracted and which ones can be kept concrete. Your goal is to abstract the necessary variables so that the analysis can converge to a fixpoint, but not to lose too much information through the abstractions. You are only allowed to abstract variable names, not fields. Here are some basic heuristics for when to abstract a variable and when to keep it concrete:
- Abstract a variable if the precise value of the variable is not important for the analysis.
- Abstract a variable if re-executing the program infinitely will cause the variable to increase in size. If re-executing the program does not change the shape or values of the variable, keep it concrete.
- Keep a variable concrete if the current values provided above would not change by re-executing the program.
- Keep a variable concrete if it is used frequently in control flow decisions.
- Whenever possible and if you are unsure, keep variables concrete and do not abstract. We can always abstract later if needed.


I will provide the code and the current state of the variables.
The code is:
```javascript
{code}
```

The variables you can abstract are: {all_var_names_comma_separated}. The values for the variables in JSON form so far are:
```json
{all_var_values_str}
```
What variables should be abstracted? You can abstract both primitives and objects. The program will converge when the variables do not change between executions. Please perform enough abstractions for the program converge but do not abstract too much information. We are about to re-execute the code I provided above with the variables initialized to the values above with no destructive updates to the variables. This means that any updates will be performed on the variables above.
"""

    system_prompt = "You are a program analyst specializing in deciding what variables to abstract for Javascript. You are tasked with analyzing a program that might throw a `TypeError: Cannot read properties of null or undefined` error. You are given the code and the state of all the variables.  Please decide which variables need to be abstracted to achieve the best balance of scalability and precision for detecting `TypeError: Cannot read properties of null or undefined`."
    chat_history = []
    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/var_merging/initial_prompt",
        ),
        "r",
    ) as f:
        initial_prompt = f.read().strip()
    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/var_merging/initial_response",
        ),
        "r",
    ) as f:
        initial_response = f.read().strip()

    logger.info(f"FIRST ABSTRACTION MESSAGE: {first_abstraction_message}")
    env.num_input_tokens += get_num_tokens(first_abstraction_message)
    chat_history.append((initial_prompt, initial_response))
    first_abstraction_prompt = get_openai_prompt(
        first_abstraction_message, chat_history, system_prompt
    )
    first_abstraction_response_full = env.openai_client.chat.completions.create(
        model=model, messages=first_abstraction_prompt, max_tokens=1000
    )
    first_abstraction_response = first_abstraction_response_full.choices[
        0
    ].message.content
    env.num_output_tokens += get_num_tokens(first_abstraction_response)
    logger.info(f"FIRST RESPONSE FROM LLM: {first_abstraction_response}")
    chat_history.append((first_abstraction_message, first_abstraction_response))
    second_abstraction_message = f"""
    {first_abstraction_response}
    
Based on the above explanation, what variables should be abstracted? Please respond with the variables that should be abstracted in backticks. If a variable should be kept concrete or kept unchanged, DO NOT INCLUDE IT IN YOUR RESPONSE.
"""
    get_variables_prompt = get_openai_prompt(
        second_abstraction_message, [], system_prompt
    )
    env.num_input_tokens += get_num_tokens(second_abstraction_message)
    get_variables_response_full = env.openai_client.chat.completions.create(
        model=model, messages=get_variables_prompt, max_tokens=100
    )
    get_variables_response = get_variables_response_full.choices[0].message.content
    env.num_output_tokens += get_num_tokens(get_variables_response)
    # logger.info(f"SECOND RESPONSE FROM LLM: {get_variables_response}")
    variables_to_abstract = Util.get_variables_in_backticks(get_variables_response)
    variables_to_abstract = [
        _.split(".")[0] for _ in variables_to_abstract
    ]  # this is a hack to remove fields
    variables_to_abstract = list(set(variables_to_abstract))
    logger.info(f"Variables to abstract from LLM: {variables_to_abstract}")
    return variables_to_abstract


# FUNCTION
def get_fields_to_abstract_from_llm_global(
    code: str, model: str, var_name: str, var_value: dict, env: "Environment"
) -> list[str] | str:
    system_prompt = "You are a program analyst specializing in deciding what variables to abstract for Javascript. You are tasked with analyzing a program that might throw a `TypeError: Cannot read properties of null or undefined` error. You are given the code and the state of all the variables.  Please decide which variables need to be abstracted to achieve the best balance of scalability and precision for detecting `TypeError: Cannot read properties of null or undefined`."
    var_value_str = beeprint.pp(var_value, output=False)
    chat_history = []
    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/var_merging/initial_prompt",
        ),
        "r",
    ) as f:
        initial_prompt = f.read().strip()
    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/var_merging/initial_response",
        ),
        "r",
    ) as f:
        initial_response = f.read().strip()
    chat_history.append((initial_prompt, initial_response))
    should_abstract_fields_message = f"""
You chose to abstract the following variable: `{var_name}`. It has multiple fields, the value of the object is:
```json
{var_value_str}
```
As a reminder, the code is:
```javascript
{code}
```
Do you want to pick individual fields to abstract, or should I abstract the whole thing? You would abstract individual fields if:
- You know that only some fields are relevant to the analysis
- Some fields are not being updated, and can stay concrete
- Some fields are being updated, and need to be abstracted
You would abstract all fields if:
- You are not sure which fields are relevant
- The fields are being added dynamically
If there are fields or properties being added dynamically, you'll have to abstract all fields.
Please respond with "INDIVIDUAL FIELDS" or "ALL FIELDS". DO NOT say anything else.
"""
    should_abstract_fields_prompt = get_openai_prompt(
        should_abstract_fields_message, chat_history, system_prompt
    )
    env.num_input_tokens += get_num_tokens(should_abstract_fields_message)
    should_abstract_fields_response_full = env.openai_client.chat.completions.create(
        model=model, messages=should_abstract_fields_prompt, max_tokens=50
    )
    should_abstract_fields_response = should_abstract_fields_response_full.choices[
        0
    ].message.content
    env.num_output_tokens += get_num_tokens(should_abstract_fields_response)
    if (
        should_abstract_fields_response != "INDIVIDUAL FIELDS"
        and should_abstract_fields_response != "ALL FIELDS"
    ):
        return "ALL FIELDS"
    if should_abstract_fields_response == "ALL FIELDS":
        return "ALL FIELDS"
    fields = list(var_value.keys())
    fields_comma_separated = ", ".join([f"`{field}`" for field in fields])
    fields_to_abstract_message = f"""
You chose to abstract the following variable: `{var_name}`. It has multiple fields, the value of the object is:
```json
{var_value_str}
```
As a reminder, the entire code is:
```javascript
{code}
```
What fields should be abstracted, and why? The possible fields are: {fields_comma_separated}. 
You should abstract a field if:
- The value for the field is being updated dynamically, and abstraction will help the analysis converge.
You should NOT abstract a field if:
- The precise value of the field is important for the analysis, for example if it is important for control flow.
- The value of the field is not being updated
"""
    fields_to_abstract_prompt = get_openai_prompt(
        fields_to_abstract_message, [], system_prompt
    )
    env.num_input_tokens += get_num_tokens(fields_to_abstract_message)
    fields_to_abstract_response_full = env.openai_client.chat.completions.create(
        model=model, messages=fields_to_abstract_prompt, max_tokens=1000
    )
    fields_to_abstract_response = fields_to_abstract_response_full.choices[
        0
    ].message.content
    env.num_output_tokens += get_num_tokens(fields_to_abstract_response)
    chat_history.append((fields_to_abstract_message, fields_to_abstract_response))
    final_fields_abstraction_message = f"""
Please respond with the field names that should be abstracted surrounded by backticks, and nothing else. DO NOT say anything else."""
    final_fields_abstraction_prompt = get_openai_prompt(
        final_fields_abstraction_message, chat_history, system_prompt
    )
    env.num_input_tokens += get_num_tokens(final_fields_abstraction_message)
    final_fields_abstraction_response_full = env.openai_client.chat.completions.create(
        model=model, messages=final_fields_abstraction_prompt, max_tokens=100
    )
    final_fields_abstraction_response = final_fields_abstraction_response_full.choices[
        0
    ].message.content
    env.num_output_tokens += get_num_tokens(final_fields_abstraction_response)
    fields_to_abstract = Util.get_variables_in_backticks(
        final_fields_abstraction_response
    )
    return fields_to_abstract


# FUNCTION
def get_variables_to_abstract_from_llm_loop(
    code: str,
    model: str,
    changed_variable_names: list[str],
    loop_body: str,
    env: "Environment",
) -> list[str]:
    all_var_names_comma_separated = ", ".join(
        [f"`{var_name}`" for var_name in changed_variable_names]
    )
    all_var_values_str = beeprint.pp(
        env.get_all_reachable_object_variable_values(changed_variable_names),
        output=False,
    )
    first_abstraction_message = f"""
You are part of a static analyzer, where you are responsible for deciding which variables need to be abstracted and which ones can be kept concrete. Your goal is to abstract the necessary variables so that the analysis can converge to a fixpoint, but not to lose too much information through the abstractions. You have encountered an unbounded loop, and are deciding which variables need to be abstracted.  You are only allowed to abstract variable names, not fields. Here are some basic heuristics for when to abstract a variable and when to keep it concrete:
- Abstract a variable if the precise value of the variable is not important for the analysis.
- Abstract a variable if re-executing the program infinitely will cause the variable to increase in size. If re-executing the program does not change the shape or values of the variable, keep it concrete.
- Keep a variable concrete if the current values provided above would not change by re-executing the program.
- Keep a variable concrete if it is used frequently in control flow decisions.
- Whenever possible and if you are unsure, keep variables concrete and do not abstract. We can always abstract later if needed.
I will provide the full program. the loop body, and the current state of the variables.

The loop body is:
```javascript
{loop_body}
```

The code is:
```javascript
{code}
```

The variables you can abstract are: {all_var_names_comma_separated}. The values for the variables in JSON form so far are:
```json
{all_var_values_str}
```
What variables should be abstracted? You can abstract both primitives and objects. The program will converge when the variables do not change between iterations. Please perform enough abstractions for the program converge but do not abstract too much information. We are about to re-execute the loop with the variables initialized to the values above. This means that any updates will be performed on the variables above.
"""

    system_prompt = "You are a program analyst specializing in deciding what variables to abstract for Javascript. You are tasked with analyzing a program that might throw a `TypeError: Cannot read properties of null or undefined` error. You are given the code and the state of all the variables.  Please decide which variables need to be abstracted to achieve the best balance of scalability and precision for detecting `TypeError: Cannot read properties of null or undefined`."
    chat_history = []
    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/var_merging/initial_prompt",
        ),
        "r",
    ) as f:
        initial_prompt = f.read().strip()
    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/var_merging/initial_response",
        ),
        "r",
    ) as f:
        initial_response = f.read().strip()

    # logger.info(f"FIRST ABSTRACTION MESSAGE: {first_abstraction_message}")
    env.num_input_tokens += get_num_tokens(first_abstraction_message)
    chat_history.append((initial_prompt, initial_response))
    first_abstraction_prompt = get_openai_prompt(
        first_abstraction_message, chat_history, system_prompt
    )
    first_abstraction_response_full = env.openai_client.chat.completions.create(
        model=model, messages=first_abstraction_prompt, max_tokens=1000
    )
    first_abstraction_response = first_abstraction_response_full.choices[
        0
    ].message.content
    env.num_output_tokens += get_num_tokens(first_abstraction_response)
    # logger.info(f"FIRST RESPONSE FROM LLM: {first_abstraction_response}")
    chat_history.append((first_abstraction_message, first_abstraction_response))
    second_abstraction_message = f"""
    {first_abstraction_response}
    
Based on the above explanation, what variables should be abstracted? Please respond with the variables that should be abstracted in backticks. If a variable should be kept concrete or kept unchanged, DO NOT INCLUDE IT IN YOUR RESPONSE.
"""
    env.num_input_tokens += get_num_tokens(second_abstraction_message)
    get_variables_prompt = get_openai_prompt(
        second_abstraction_message, [], system_prompt
    )
    get_variables_response_full = env.openai_client.chat.completions.create(
        model=model, messages=get_variables_prompt, max_tokens=100
    )
    get_variables_response = get_variables_response_full.choices[0].message.content
    env.num_output_tokens += get_num_tokens(get_variables_response)
    # logger.info(f"SECOND RESPONSE FROM LLM: {get_variables_response}")
    variables_to_abstract = Util.get_variables_in_backticks(get_variables_response)
    variables_to_abstract = [
        _.split(".")[0] for _ in variables_to_abstract
    ]  # this is a hack to remove fields
    logger.info(f"Variables to abstract from LLM: {variables_to_abstract}")
    return variables_to_abstract


# FUNCTION
def get_fields_to_abstract_from_llm_loop(
    code: str,
    loop_body: str,
    model: str,
    var_name: str,
    var_value: dict,
    env: "Environment",
) -> list[str] | str:
    system_prompt = "You are a program analyst specializing in deciding what variables to abstract for Javascript. You are tasked with analyzing a program that might throw a `TypeError: Cannot read properties of null or undefined` error. You are given the code and the state of all the variables.  Please decide which variables need to be abstracted to achieve the best balance of scalability and precision for detecting `TypeError: Cannot read properties of null or undefined`."
    var_value_str = beeprint.pp(var_value, output=False)
    chat_history = []
    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/var_merging/initial_prompt",
        ),
        "r",
    ) as f:
        initial_prompt = f.read().strip()
    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/var_merging/initial_response",
        ),
        "r",
    ) as f:
        initial_response = f.read().strip()
    chat_history.append((initial_prompt, initial_response))
    should_abstract_fields_message = f"""
You chose to abstract the following variable: `{var_name}`. It has multiple fields, the value of the object is:
```json
{var_value_str}
```
As a reminder, the loop body is:
```javascript
{loop_body}
```
Do you want to pick individual fields to abstract, or should I abstract the whole thing? You would abstract individual fields if:
- You know that only some fields are relevant to the analysis
- Some fields are not being updated, and can stay concrete
- Some fields are being updated, and need to be abstracted
You would abstract all fields if:
- You are not sure which fields are relevant
- The fields are being added dynamically
If there are fields or properties being added dynamically, you'll have to abstract all fields.
Please respond with "INDIVIDUAL FIELDS" or "ALL FIELDS". DO NOT say anything else.
"""
    should_abstract_fields_prompt = get_openai_prompt(
        should_abstract_fields_message, chat_history, system_prompt
    )
    env.num_input_tokens += get_num_tokens(should_abstract_fields_message)
    should_abstract_fields_response_full = env.openai_client.chat.completions.create(
        model=model, messages=should_abstract_fields_prompt, max_tokens=50
    )
    should_abstract_fields_response = should_abstract_fields_response_full.choices[
        0
    ].message.content
    env.num_output_tokens += get_num_tokens(should_abstract_fields_response)
    if (
        should_abstract_fields_response != "INDIVIDUAL FIELDS"
        and should_abstract_fields_response != "ALL FIELDS"
    ):
        return "ALL FIELDS"
    if should_abstract_fields_response == "ALL FIELDS":
        return "ALL FIELDS"
    fields = list(var_value.keys())
    fields_comma_separated = ", ".join([f"`{field}`" for field in fields])
    fields_to_abstract_message = f"""
You chose to abstract the following variable: `local_obj`. It has multiple fields, the value of the object is:
```json
{var_value_str}
```
As a reminder, the loop body is:
```javascript
{loop_body}
```
The entire code is:
```javascript
{code}
```
What fields should be abstracted, and why? The possible fields are: {fields_comma_separated}. 
You should abstract a field if:
- The value for the field is being updated dynamically, and abstraction will help the analysis converge.
You should NOT abstract a field if:
- The precise value of the field is important for the analysis, for example if it is important for control flow.
- The value of the field is not being updated
"""
    env.num_input_tokens += get_num_tokens(fields_to_abstract_message)
    fields_to_abstract_prompt = get_openai_prompt(
        fields_to_abstract_message, [], system_prompt
    )
    fields_to_abstract_response_full = env.openai_client.chat.completions.create(
        model=model, messages=fields_to_abstract_prompt, max_tokens=1000
    )
    fields_to_abstract_response = fields_to_abstract_response_full.choices[
        0
    ].message.content
    env.num_output_tokens += get_num_tokens(fields_to_abstract_response)
    chat_history.append((fields_to_abstract_message, fields_to_abstract_response))
    final_fields_abstraction_message = f"""
Please respond with the field names that should be abstracted surrounded by backticks, and nothing else. DO NOT say anything else."""
    env.num_input_tokens += get_num_tokens(final_fields_abstraction_message)
    final_fields_abstraction_prompt = get_openai_prompt(
        final_fields_abstraction_message, chat_history, system_prompt
    )
    final_fields_abstraction_response_full = env.openai_client.chat.completions.create(
        model=model, messages=final_fields_abstraction_prompt, max_tokens=100
    )
    final_fields_abstraction_response = final_fields_abstraction_response_full.choices[
        0
    ].message.content
    env.num_output_tokens += get_num_tokens(final_fields_abstraction_response)
    fields_to_abstract = Util.get_variables_in_backticks(
        final_fields_abstraction_response
    )
    return fields_to_abstract


def simplify_variable_global(
    env: "Environment", var_name: str, code: str, model: str, invoke_llm=False
) -> None:
    record_result = env.lookup(var_name)
    record_result.merge_primitives()
    if env.record_result_contains_objects(record_result) and invoke_llm:
        address_to_abstract = env.merge_all_addresses_in_record_result(
            record_result, simplify=False
        )
        if not address_to_abstract:
            return
        obj_to_abstract = env.lookup_and_derive_address(address_to_abstract)
        fields_to_abstract = get_fields_to_abstract_from_llm_global(
            code=code,
            model=model,
            var_name=var_name,
            var_value=obj_to_abstract,
            env=env,
        )
        logger.info(f"Fields to abstract for {var_name}: {fields_to_abstract}")
        if fields_to_abstract == "ALL FIELDS":
            env.simplify_address(address_to_abstract)
        else:
            env.simplify_fields_for_address(address_to_abstract, fields_to_abstract)
    else:
        env.merge_all_addresses_in_record_result(record_result, simplify=True)


def simplify_variable_loop(
    env: "Environment",
    var_name: str,
    code: str,
    loop_body: str,
    model: str,
    invoke_llm=False,
) -> None:
    record_result = env.lookup(var_name)
    record_result.merge_primitives()
    if env.record_result_contains_objects(record_result) and invoke_llm:
        address_to_abstract = env.merge_all_addresses_in_record_result(
            env, record_result, simplify=False
        )
        if not address_to_abstract:
            return
        obj_to_abstract = env.lookup_and_derive_address(address_to_abstract)
        fields_to_abstract = get_fields_to_abstract_from_llm_loop(
            code=code,
            loop_body=loop_body,
            model=model,
            var_name=var_name,
            var_value=obj_to_abstract,
            env=env,
        )
        logger.info(f"Fields to abstract for {var_name}: {fields_to_abstract}")
        if fields_to_abstract == "ALL FIELDS":
            env.simplify_address(address_to_abstract)
        else:
            env.simplify_fields_for_address(address_to_abstract, fields_to_abstract)
    else:
        env.merge_all_addresses_in_record_result(record_result, simplify=True)
