import beeprint
import os
from ordered_set import OrderedSet
from absint_ai.utils.Logger import logger
from typing import TYPE_CHECKING
import tiktoken
from absint_ai.utils.Util import *
import json
from absint_ai.Environment.types.Type import *
from absint_ai.Environment.memory.RecordResult import RecordResult

if TYPE_CHECKING:
    from absint_ai.Environment.Environment import (
        Environment,
    )  # Import the class only for type checking


def get_num_tokens(self, string: str) -> int:
    """Returns the number of tokens in a text string."""
    encoding = tiktoken.get_encoding("o200k_base")
    num_tokens = len(encoding.encode(string))

    return num_tokens


def get_openai_prompt(
    message: str, chat_history: list[tuple[str, str]], system_prompt: str
) -> list:
    messages = [
        {"role": "system", "content": system_prompt},
    ]
    for user_input, response in chat_history:
        messages.append({"role": "user", "content": user_input})
        messages.append({"role": "assistant", "content": response})
    messages.append({"role": "user", "content": message})
    return messages


# sometimes chatGPT doesn't return the values as a list, so we need to convert them to a list
def listify(abstraction: object) -> list:
    if isinstance(abstraction, list):
        return abstraction  # [self.listify(val) for val in abstraction]
    elif isinstance(abstraction, dict):
        new_abstraction = {}
        for var, values in abstraction.items():
            new_abstraction[var] = listify(values)
        return [new_abstraction]
    else:
        return [abstraction]


def generate_and_merge_address_abstraction_from_llm_global(
    model: str, code: str, env: "Environment"
) -> None:
    all_variables = env.get_all_reachable_object_variable_names()
    concrete_addresses, abstract_address = env.get_referenced_addresses_from_variables(
        all_variables
    )
    address_abstraction = generate_address_abstraction_from_llm(
        model, code, "", concrete_addresses, abstract_address, all_variables
    )
    try:
        logger.info(
            f"Merging address abstraction from LLM global: {address_abstraction}"
        )
        modified_abstract_addresses1 = env.merge_address_abstraction(
            address_abstraction
        )
    except Exception as e:
        logger.info(f"failed to merge {e}")
        modified_abstract_addresses1 = []


# returns address mappings from LLM
def generate_address_abstraction_from_llm(
    model: str,
    code: str,
    loop_body: str,
    concrete_addresses: list[Address],
    abstract_addresses: list[Address],
    changed_variables: list[str],
    env: "Environment",
) -> dict:
    if not concrete_addresses and not abstract_addresses:
        logger.info(f"No addresses or variables changed, returning empty dict")
        return {}
    concrete_heap_string = env.pretty_print_addresses(concrete_addresses)
    abstract_heap_string = env.pretty_print_addresses(abstract_addresses)
    if (
        len(json.loads(concrete_heap_string)) == 0
        and len(json.loads(abstract_heap_string)) == 0
    ):
        logger.info(f"No addresses changed, returning empty dict")
        return {}
    env_string = env.pretty_print_variables(changed_variables)
    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/in_context/address_merging/prompts/initial_setup_prompt",
        ),
        "r",
    ) as f:
        initial_setup_prompt = f.read().strip()

    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/in_context/address_merging/responses/initial_setup_response",
        ),
        "r",
    ) as f:
        initial_setup_response = f.read().strip()

    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/in_context/address_merging/prompts/initial_setup_prompt2",
        ),
        "r",
    ) as f:
        initial_setup_prompt2 = f.read().strip()

    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/in_context/address_merging/responses/initial_setup_response2",
        ),
        "r",
    ) as f:
        initial_setup_response2 = f.read().strip()

    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/in_context/address_merging/prompts/merge_query",
        ),
        "r",
    ) as f:
        merge_query = f.read().strip()

    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/in_context/address_merging/responses/merge_query_response",
        ),
        "r",
    ) as f:
        merge_query_response = f.read().strip()

    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/in_context/address_merging/prompts/json_conversion",
        ),
        "r",
    ) as f:
        json_conversion = f.read().strip()

    with open(
        os.path.join(
            os.path.dirname(__file__),
            "templates/in_context/address_merging/responses/json_conversion_response",
        ),
        "r",
    ) as f:
        json_conversion_response = f.read().strip()

    chat_history = []
    chat_history.append((initial_setup_prompt, initial_setup_response))
    chat_history.append((initial_setup_prompt2, initial_setup_response2))
    chat_history.append((merge_query, merge_query_response))
    chat_history.append((json_conversion, json_conversion_response))
    system_prompt = "You are a program analyst specializing in heap abstractions for Javascript. You are tasked with analyzing a program that might throw a `TypeError: Cannot read properties of null or undefined` error. You are given the code, the state of the environment, and the state of the concrete heap and abstract heap. Please merge nodes in the heap to achieve the best balance of scalability and precision for detecting `TypeError: Cannot read properties of null or undefined`. "
    loop_body_message = ""
    if loop_body:
        loop_body_message = f"""Here is the loop body we are analyzing:
```javascript
{loop_body}
```
"""
    message = f"""
I am going to provide you with the code, the environment, and the state of the concrete heap and abstract heap. The environment and heaps will be provided in JSON format. Please merge nodes in the heap to achieve the best balance of scalability and precision for detecting `TypeError: Cannot read properties of null or undefined`.
Here is the code: \n
```javascript
{code}
```
{loop_body_message}

Here is the current state of the environment:
```json
{env_string}
```

Here is the current state of the concrete heap:
```json
{concrete_heap_string}
```

Here is the current state of the abstract heap:
```json
{abstract_heap_string}
```

Please provide an abstraction by merging nodes in the concrete and abstract heap, and provide a mapping between which nodes have been merged in your output. Only specify the addresses you want to merge. 
The keys in your JSON object should be abstract address strings, and the values should be lists of concrete addresses that you want to merge. For example, if you want to merge addresses ["Concrete Address(1)", "Concrete Address(2)"] into an abstract address "Abstract Address(1)", you would provide the following JSON object:
{{"Abstract Address(1)": ["Concrete Address(1)", "Concrete Address(2)"]}}
You can also return an empty object if you do not want to merge any nodes.
"""
    # logger.info(f"Address abstraction information: {code}\n{env_string}\n{concrete_heap_string}\n{abstract_heap_string}")
    # logger.info(f"Address abstraction information: {message}")
    first_prompt = get_openai_prompt(message, chat_history, system_prompt)
    first_response_full = env.openai_client.chat.completions.create(
        model=model,
        messages=first_prompt,
        max_tokens=1000,
    )
    first_response: str = first_response_full.choices[0].message.content  # type: ignore
    # logger.info(f"first response from LLM for address mapping: {first_response}")
    result = Util.parse_json_from_message(first_response)
    logger.info(f"result from LLM for address mapping: {result}")
    return result


### TYPE ABSTRACTIONS FROM THE LLM
def generate_address_type_abstraction_from_llm(
    model: str,
    code: str,
    loop_body: str,
    concrete_addresses: list[Address],
    abstract_addresses: list[Address],
    changed_variables: list[str],
    env: "Environment",
) -> dict:
    chat_history = []
    if not concrete_addresses and not abstract_addresses:
        logger.info(f"No addresses or variables changed, returning empty dict")
        return {}
    initial_message_path = os.path.join(
        os.path.dirname(__file__),
        "templates/in_context/heap_var_abstractions/prompts/initial_message",
    )
    with open(initial_message_path, "r") as f:
        initial_message = f.read().strip()

    initial_response_path = os.path.join(
        os.path.dirname(__file__),
        "templates/in_context/heap_var_abstractions/responses/initial_response",
    )
    with open(initial_response_path, "r") as f:
        initial_response = f.read().strip()

    in_context_abstraction_path = os.path.join(
        os.path.dirname(__file__),
        "templates/in_context/heap_var_abstractions/prompts/in_context_abstraction",
    )
    with open(in_context_abstraction_path, "r") as f:
        in_context_abstraction = f.read().strip()

    in_context_abstraction_response_path = os.path.join(
        os.path.dirname(__file__),
        "templates/in_context/heap_var_abstractions/responses/in_context_abstraction_response",
    )
    with open(in_context_abstraction_response_path, "r") as f:
        in_context_abstraction_response = f.read().strip()

    json_conversion_path = os.path.join(
        os.path.dirname(__file__),
        "templates/in_context/heap_var_abstractions/prompts/json_conversion",
    )
    with open(json_conversion_path, "r") as f:
        json_conversion = f.read().strip()

    json_conversion_response_path = os.path.join(
        os.path.dirname(__file__),
        "templates/in_context/heap_var_abstractions/responses/json_conversion_response",
    )
    with open(json_conversion_response_path, "r") as f:
        json_conversion_response = f.read().strip()

    chat_history.append((initial_message, initial_response))
    chat_history.append((in_context_abstraction, in_context_abstraction_response))
    chat_history.append((json_conversion, json_conversion_response))
    system_prompt = "You are an expert program analyzer and instructor. You are teaching a student how to analyze a program and perform heap abstractions."
    concrete_heap_string = env.pretty_print_addresses(concrete_addresses)
    abstract_heap_string = env.pretty_print_addresses(abstract_addresses)
    if (
        len(json.loads(concrete_heap_string)) == 0
        and len(json.loads(abstract_heap_string)) == 0
    ):
        logger.info(f"No addresses changed, returning empty dict")
        return {}
    env_string = env.pretty_print_variables(changed_variables)
    loop_body_message = ""
    if loop_body:
        loop_body_message = f"""Here is the loop body we are analyzing:
```javascript
{loop_body}
```
"""
    initial_abstraction_message = f"""You are performing heap abstractions to find a fixpoint in order to finish analyzing a loop. I am going to provide you with the code surrounding the loop body we are analyzing, the loop body we are analyzing, and the state of the heap. Please provide a heap abstraction that results in a fixpoint while keeping as much relevant information as possible.

1. Abstract a list of strings into an abstract STRING object. For example, ["foo1", "foo2", "foo3"] can become ["STRING"] 
2. Abstract a list of numbers into an abstract NUMBER object. For example, [1,2,3,4,5] can become ["NUMBER"]
3. Merging keys in an object following the above rules. For example, {{1: ["var"], 2: ["var"], 3: ["var"]}} can become {{"NUMBER": ["var"]}}. It can also become {{"NUMBER": ["STRING"]}}.
4. Leave the object as is and do not perform an abstraction.

Here is the code:
```javascript
{code}
```

{loop_body_message}

Here is the current state of the environment:
```json
{env_string}
```

Here is the current state of the concrete heap:
```json
{concrete_heap_string}
```

Here is the current state of the abstract heap:
```json
{abstract_heap_string}
````


Please provide a general heap abstraction for the relevant variables that results in a fixpoint. Perform as few abstractions as are necessary to reach a fixpoint. 

You can return an empty object if you do not want to perform any abstractions. Please use an abstraction that is as precise as possible while still maintaining a fixpoint and include a JSON of the abstraction in your response."""
    # logger.info(f"Address type abstraction info: {code}\n{env_string}\n{concrete_heap_string}\n{abstract_heap_string}")
    # logger.info(f"Address type abstraction info: {initial_abstraction_message}")
    first_prompt = get_openai_prompt(
        initial_abstraction_message, chat_history, system_prompt
    )
    first_response_full = env.openai_client.chat.completions.create(
        model=model,
        messages=first_prompt,
        max_tokens=1000,
    )
    first_response: str = first_response_full.choices[0].message.content  # type: ignore
    # logger.info(
    #    f"first response from LLM for address type abstraction: {first_response}"
    # )
    address_type_abstraction = Util.parse_json_from_message(first_response)
    logger.info(f"address type abstraction from LLM is: {address_type_abstraction}")
    return address_type_abstraction


def generate_env_abstraction_from_llm(
    model: str,
    code: str,
    loop_body: str,
    changed_variables: list[str],
    env: "Environment",
) -> dict:
    initial_message_path = os.path.join(
        os.path.dirname(__file__),
        "templates/in_context/env_abstraction/prompts/initial_message",
    )
    with open(initial_message_path, "r") as f:
        initial_message = f.read().strip()

    initial_response_path = os.path.join(
        os.path.dirname(__file__),
        "templates/in_context/env_abstraction/responses/initial_response",
    )
    with open(initial_response_path, "r") as f:
        initial_response = f.read().strip()

    abstraction_message_path = os.path.join(
        os.path.dirname(__file__),
        "templates/in_context/env_abstraction/prompts/abstraction_message",
    )
    with open(abstraction_message_path, "r") as f:
        abstraction_message = f.read().strip()

    abstraction_response_path = os.path.join(
        os.path.dirname(__file__),
        "templates/in_context/env_abstraction/responses/abstraction_response",
    )
    with open(abstraction_response_path, "r") as f:
        abstraction_response = f.read().strip()

    json_conversion_path = os.path.join(
        os.path.dirname(__file__),
        "templates/in_context/env_abstraction/prompts/json_conversion",
    )
    with open(json_conversion_path, "r") as f:
        json_conversion = f.read().strip()

    json_conversion_response_path = os.path.join(
        os.path.dirname(__file__),
        "templates/in_context/env_abstraction/responses/json_conversion_response",
    )
    with open(json_conversion_response_path, "r") as f:
        json_conversion_response = f.read().strip()

    chat_history = []
    chat_history.append((initial_message, initial_response))
    chat_history.append((abstraction_message, abstraction_response))
    chat_history.append((json_conversion, json_conversion_response))
    system_prompt = "You are an expert program analyzer and instructor. You are teaching a student how to analyze a program and perform heap abstractions."
    env_primitives_string = env.pretty_print_variables(changed_variables)
    loop_body_message = ""
    if loop_body:
        loop_body_message = f"""Here is the loop body we are analyzing:
```javascript
{loop_body}
```
"""
    initial_abstraction_message = f"""You are abstracting variables in order to finish analyzing a loop. I am going to provide you with the code and the state of the heap. Please provide an abstraction of the primitive variables that results in a fixpoint while keeping as much relevant information as possible.

1. Abstract a list of strings into an abstract STRING object. For example, ["foo1", "foo2", "foo3"] can become ["STRING"] 
2. Abstract a list of numbers into an abstract NUMBER object. For example, [1,2,3,4,5] can become ["NUMBER"]
3. Merging keys in an object following the above rules. For example, {{1: ["var"], 2: ["var"], 3: ["var"]}} can become {{"NUMBER": ["var"]}}. It can also become {{"NUMBER": ["STRING"]}}.
4. Leave the object as is and do not perform an abstraction.

Here is the code:
```javascript
{code}
```

{loop_body_message}


Here is the current state of the environment:
```json
{env_primitives_string}
```

Please provide a general abstraction for the relevant variables that results in a fixpoint. Perform as few abstractions as are necessary to reach a fixpoint. You can return an empty object if you do not want to perform any abstractions. Please use an abstraction that is as precise as possible while still maintaining a fixpoint, and include a JSON of the abstraction in your response."""
    # logger.info(f"env abstraction message: {code}\n{env_primitives_string}")
    # logger.info(f"env abstraction message: {initial_abstraction_message}")
    first_prompt = get_openai_prompt(
        initial_abstraction_message, chat_history, system_prompt
    )
    first_response_full = env.openai_client.chat.completions.create(
        model=model,
        messages=first_prompt,
        max_tokens=1000,
    )
    first_response: str = first_response_full.choices[0].message.content  # type: ignore
    # logger.info(f"First response from LLM for env abstraction is {first_response}")
    env_abstraction = Util.parse_json_from_message(first_response)
    logger.info(f"Env abstraction from LLM is: {env_abstraction}")
    return env_abstraction


def simplify_llm_naive(
    env: "Environment",
    model: str,
    code_window: str,
    loop_body: str,
    changed_variables: list[str],
) -> list[Address]:
    concrete_addresses, abstract_addresses = (
        env.get_referenced_addresses_from_variables(changed_variables)
    )
    address_abstraction = generate_address_abstraction_from_llm(
        model,
        code=code_window,
        loop_body=loop_body,
        concrete_addresses=concrete_addresses,
        abstract_addresses=abstract_addresses,
        changed_variables=changed_variables,
        env=env,
    )
    address_type_abstraction = generate_address_type_abstraction_from_llm(
        model,
        code=code_window,
        loop_body=loop_body,
        concrete_addresses=concrete_addresses,
        abstract_addresses=abstract_addresses,
        changed_variables=changed_variables,
        env=env,
    )
    env_abstraction = generate_env_abstraction_from_llm(
        model,
        code=code_window,
        loop_body=loop_body,
        changed_variables=changed_variables,
        env=env,
    )
    try:
        logger.info(f"Merging address abstraction from LLM: {address_abstraction}")
        modified_abstract_addresses1 = env.merge_address_abstraction(
            address_abstraction
        )
        if len(modified_abstract_addresses1) > 0:
            logger.info("SUCCESSFUL ADDRESS ABSTRACTION MERGE")
        else:
            logger.info("FAILED ADDRESS ABSTRACTION MERGE")
    except Exception as e:
        logger.info(f"failed to merge {e}")
        modified_abstract_addresses1 = []
    try:
        logger.info(
            f"Merging address type abstraction from LLM: {address_type_abstraction}"
        )
        modified_abstract_addresses2 = env.merge_address_type_abstraction(
            address_type_abstraction
        )
        if len(modified_abstract_addresses2) > 0:
            logger.info("SUCCESSFUL ADDRESS ABSTRACTION TYPE MERGE")
        else:
            logger.info("FAILED ADDRESS ABSTRACTION TYPE MERGE")
    except Exception as e:
        logger.info(f"failed to merge {e}")
        modified_abstract_addresses2 = []
    try:
        updated_vars = merge_env_abstraction(env, env_abstraction)
        if len(updated_vars) > 0:
            logger.info("SUCCESSFUL ENV ABSTRACTION MERGE")
        else:
            logger.info("FAILED ENV ABSTRACTION MERGE")
    except Exception as e:
        logger.info(f"failed to merge {e}")
    return list(OrderedSet(modified_abstract_addresses1 + modified_abstract_addresses2))


def merge_env_abstraction(env: "Environment", abstraction: dict) -> None:
    updated_vars = []
    for var, abstraction_values in abstraction.items():
        if var == "this":
            continue
        if not isinstance(abstraction_values, list) and not isinstance(
            abstraction_values, OrderedSet
        ):
            abstraction_values = [abstraction_values]
        if env.validate_variable_abstraction(var, abstraction_values):
            logger.info(f"var before: {env.lookup(var)}")
            env.merge_abstraction_for_primitive_var(var, abstraction_values)
            logger.info(
                f"MERGED {var} {abstraction_values} from merge_env_abstraction."
            )
            updated_vars.append(var)
    return updated_vars  # mostly for debugging


def merge_address_abstraction(
    env: "Environment", address_mapping: dict
) -> list[Address]:
    modified_abstract_addresses = OrderedSet()
    for abstracted_address_str, addresses in address_mapping.items():
        try:
            abstracted_address = address_from_string(abstracted_address_str)
        except Exception as e:
            logger.info(f"Failed to parse address {abstracted_address_str}")
            continue
        if not abstracted_address.get_addr_type() == "abstract":
            continue
        if env.abstract_heap.contains(abstracted_address.get_value()):
            abstracted_object = env.lookup_address(abstracted_address)
        else:
            abstracted_object = {}
        if is_heap_frame(abstracted_object):
            continue
        if not isinstance(addresses, list) and not isinstance(addresses, OrderedSet):
            addresses = [addresses]
        for address_str in addresses:
            try:
                addr = address_from_string(address_str)
            except Exception as e:
                logger.info(f"Failed to parse address {address_str} from {addresses}")
                continue
            try:
                obj = env.lookup_address(addr)
            except Exception as e:
                logger.info(f"Failed to lookup address {addr} {e}")
                continue
            if is_heap_frame(obj):
                continue
            for key, value in obj.items():
                if key == "__meta__":
                    abstracted_object[key] = value
                    continue
                if key not in abstracted_object:
                    abstracted_object[key] = RecordResult("abstract")
                for single_value in value:
                    if single_value not in abstracted_object[key].get_all_values():
                        abstracted_object[key].add_value(single_value)

        env.abstract_heap.add(abstracted_object, abstracted_address)
        for address_str in addresses:
            try:
                addr = address_from_string(address_str)
            except Exception as e:
                logger.info(f"Failed to parse address {address_str}")
                continue
            try:
                obj = env.lookup_address(addr)
                if is_heap_frame(obj):
                    continue
            except Exception as e:
                logger.info(f"Failed to lookup address {addr} {e}")
                continue
            if addr != abstracted_address:
                env.abstract_addresses([addr], abstracted_address)
                modified_abstract_addresses.add(abstracted_address)
    return list(modified_abstract_addresses)


def merge_address_type_abstraction(
    env: "Environment", address_type_abstraction: dict
) -> list[Address]:
    modified_abstract_addresses = OrderedSet()
    for address_str, abstraction in address_type_abstraction.items():
        try:
            address = address_from_string(address_str)
        except Exception as e:
            logger.info(f"Failed to parse address {address_str}")
            continue
        try:
            obj = env.lookup_address(address)
            if is_heap_frame(obj):
                continue
        except Exception as e:
            logger.info(f"Failed to lookup address {address} {e}")
            continue
        if env.validate_address_abstraction(address, abstraction):
            original_obj = env.lookup_address(address)
            logger.info(f"Validated address abstraction {address} {abstraction}")
            obj_from_abstraction: dict[str | AbstractType, RecordResult] = {}
            ignored_keys = ["__proto__", "__meta__", "exports", "module"]
            for key in ignored_keys:
                if key in original_obj:
                    obj_from_abstraction[key] = original_obj[key]
            if abstraction == "TOP":
                obj_from_abstraction[baseType.TOP] = baseType.TOP
                continue
            for field, values in abstraction.items():
                if field == "__meta__":  # Don't want meta to be record results
                    obj_from_abstraction[field] = values
                    continue
                if field == "NUMBER":
                    field_key = baseType.NUMBER
                elif field == "STRING":
                    field_key = baseType.STRING
                elif field == "TOP":
                    field_key = baseType.TOP
                else:
                    field_key = field
                assert isinstance(field_key, AbstractType) or isinstance(field_key, str)
                if address.get_addr_type() == "concrete":
                    obj_from_abstraction[field_key] = RecordResult("concrete")
                elif address.get_addr_type() == "abstract":
                    obj_from_abstraction[field_key] = RecordResult("abstract")
                else:
                    raise Exception(f"Unknown address type {address.get_addr_type()}")
                for value in values:
                    if value == "TOP":
                        value_to_add = baseType.TOP
                    elif value == "NUMBER":
                        value_to_add = baseType.NUMBER
                    elif value == "STRING":
                        value_to_add = baseType.STRING
                    elif is_address(value):
                        addr = address_from_string(value)
                        value_to_add = addr
                    else:
                        value_to_add = Primitive(value)
                    if value_to_add not in obj_from_abstraction[field_key]:
                        obj_from_abstraction[field_key].add_value(value_to_add)
            if address.get_addr_type() == "concrete":
                new_abstract_address = env.add_object_to_abstract_heap(
                    obj_from_abstraction
                )
                env.abstract_addresses([address], new_abstract_address)
                modified_abstract_addresses.add(new_abstract_address)
            elif address.get_addr_type() == "abstract":
                env.abstract_heap.overwrite_address(address, obj_from_abstraction)
                modified_abstract_addresses.add(address)
            logger.info(f"OBJ FROM ABSTRACTION {obj_from_abstraction}")
        else:
            logger.info(
                f"Failed to validate address abstraction {address} {abstraction}"
            )
    return list(modified_abstract_addresses)
