from dataclasses import dataclass
from absint_ai.Environment.agents.base_agent import Agent
from typing import TYPE_CHECKING
from absint_ai.Environment.agents.merging import *
from absint_ai.Environment.agents.widening import *

if TYPE_CHECKING:
    from absint_ai.Environment.Environment import Environment


@dataclass
class AgentAction:
    type: str  # "summarize", "strategy", "widen", "info", "exit", etc.
    args: dict
    tool_call_info: dict = None


def handle_info_action(env: "Environment", action: AgentAction, agent: Agent) -> None:
    """
    Handle the info action.
    """
    if action.args["type"] == "var":

        name = action.args["target"]
        var_lookup_results = env.lookup_and_derive(name, add_touched=False)
        if len(var_lookup_results) == 0:
            agent.receive_info(f"=== Variable {name} has no values at this point. ===")
        else:
            result = f"\n === Variable {name} has the following values: ===\n"

            result += "=====\n".join(
                beeprint.pp(_, output=False) for _ in var_lookup_results
            )
            agent.receive_info(result)

    elif action.args["type"] == "function":
        function_name = action.args["target"]
        function_lookup_results = env.get_function_definition(function_name)
        if len(function_lookup_results) == 0:
            agent.receive_info(
                f"=== Function {function_name} has no values at this point. ==="
            )
        else:
            result = f"\n === Function {function_name} has the following values: ===\n"

            result += function_lookup_results
            agent.receive_info(result)
    elif action.args["type"] == "alloc":
        env.print_alloc_info(action.args["target"])
    else:
        raise ValueError(f"Unknown info type: {action.args['type']}")


def apply_action_to_env(env: "Environment", action: AgentAction) -> None:
    """
    An action that changes the environment
    """
    if action.type == "summarize":
        env.summarize_allocation_sites(action.args["allocation_site_ids"])
    elif action.type == "strategy":
        env.set_strategy(action.args["strategy"])
    elif action.type == "widen":
        env.widen_allocation_sites(action.args["allocation_site_ids"])
    elif action.type == "info":
        env.print_info()
    elif action.type == "exit":
        env.exit()
    else:
        raise ValueError(f"Unknown action type: {action.type}")


def generate_strategy_for_site(
    merge_strategy_info: str, widening_strategy_info: str
) -> MergeStrategy:
    merge_strategy_str = merge_strategy_info["strategy"]
    widening_strategy_str = widening_strategy_info["strategy"]
    if widening_strategy_str == "all":
        widening_policy = WidenAll()
    elif widening_strategy_str == "field_value":
        field_paths = [
            field_path.split(".")
            for field_path in widening_strategy_info["field_paths"]
        ]
        widening_policy = WidenFields(field_paths)
    elif widening_strategy_str == "depth":
        depth = widening_strategy_info["depth"]
        widening_policy = WidenDepthLimited(depth)
    elif widening_strategy_str == "none":
        widening_policy = WidenNone()
    elif widening_strategy_str == "field_set":
        widening_policy = WidenFieldSet()
    else:
        raise Exception(
            f"Unknown widening strategy {widening_strategy_str}. Valid options are: all."
        )
    if merge_strategy_str == "all":
        return MergeByAllocationSite(widening_policy)
    elif merge_strategy_str == "recency":
        return MergeByRecency(widening_policy)
    elif merge_strategy_str == "role":
        field = merge_strategy_info["field"]
        return MergeByRole(field, widening_policy)
    else:
        raise Exception(
            f"Unknown merge strategy {merge_strategy_str}. Valid options are: naive."
        )
