import beeprint
import os
from absint_ai.utils.Logger import logger
from typing import TYPE_CHECKING, Optional
from absint_ai.Environment.memory.RecordResult import *
import tiktoken
import absint_ai.utils.Util as Util
import absint_ai.Environment.features.summarization as summarization
import json
from abc import ABC, abstractmethod
from absint_ai.Environment.types.Type import *

if TYPE_CHECKING:
    from absint_ai.Environment.Environment import (
        Environment,
    )  # Import the class only for type checking


"""Different type of widening rules 
- `field value`: widen the value for a few particular fields. You might do this if only a few fields are growing.
- `field key`: merge a set of fields into one.
- `depth limited`: widen all values after a particular depth.
- `all`: widen the entire thing. Not recommended.
- `none`: do not widen anything. Do this if the object will converge
"""


class WideningRule:
    def __init__(self, type, field_paths):
        self.type: str = type
        self.field_paths: list[list[str]] = (
            field_paths  # it's a list of lists because we want to handle nested fields
        )


def widen(env: "Environment", address: Address, widening_rule: WideningRule) -> None:
    if widening_rule.type == "fields":
        field_paths = widening_rule.field_paths
        ""
        for field_path in field_paths:
            value_record_results: list[RecordResult] = env.lookup_values_for_field_path(
                address, field_path
            )
            for value_record_result in value_record_results:
                summarization.simplify_record_result(
                    env=env, record_result=value_record_result
                )
    if widening_rule.type == "all":
        summarization.simplify_address(env, address)


class WideningPolicy(ABC):
    @abstractmethod
    def widen(self, env: "Environment", address: Address) -> None:
        """
        Widen the address with the given ID.
        """
        pass


class WidenAll(WideningPolicy):
    def __init__(self):
        pass

    def widen(self, env: "Environment", address: Address) -> None:
        env.simplify_address(address)


class WidenFields(WideningPolicy):
    def __init__(self, field_paths: list[list[str]]):
        self.field_paths = field_paths

    def widen(self, env: "Environment", address: Address) -> None:
        for field_path in self.field_paths:
            value_record_results: list[RecordResult] = env.lookup_values_for_field_path(
                address, field_path
            )
            for value_record_result in value_record_results:
                summarization.simplify_record_result(
                    env=env, record_result=value_record_result
                )


class WidenDepthLimited(WideningPolicy):
    def __init__(self, depth: int):
        self.depth = depth

    # Simplify all values after a particular depth. Search fields in the object for depth.
    def widen(self, env: "Environment", address: Address) -> None:
        # Get the current depth of the address
        value_record_results: list[RecordResult] = env.lookup_values_for_depth(
            address, self.depth
        )
        for value_record_result in value_record_results:
            # Simplify the record result
            summarization.simplify_record_result(
                env=env, record_result=value_record_result
            )

class WidenFieldSet(WideningPolicy):
    def __init__(self):
        pass
    
    # Join all the fields in the object into one field. This is useful for objects that are growing in size.
    def widen(self, env: "Environment", address: Address) -> None:
        summarization.merge_fields_in_address(env, address)
            


class WidenNone(WideningPolicy):
    def __init__(self):
        pass

    # Do nothing, it will converge on its own after merging.
    def widen(self, env: "Environment", address: Address) -> None:
        pass
