from absint_ai.Environment.types.Type import *
from ordered_set import OrderedSet
from typing import TYPE_CHECKING
from absint_ai.Environment.memory.RecordResult import RecordResult

if TYPE_CHECKING:
    from absint_ai.Environment.Environment import Environment


def execute_builtin(
    env: "Environment", obj: dict, args: OrderedSet[Type]
) -> OrderedSet[Type]:
    name = obj["__meta__"]["function_id"]
    if name == "getElementById":
        return getElementById(env).get_all_values()
    if name == "createElement":
        return createElement(env).get_all_values()
    if name == "querySelector":
        return querySelector(env).get_all_values()
    if name == "appendChild":
        return OrderedSet()
    if name == "setAttribute":
        return OrderedSet()
    if name == "addEventListener":
        # addEventListener(env, args[0], args[1])
        return OrderedSet()
    if name == "querySelectorAll":
        return querySelectorAll(env).get_all_values()
    return OrderedSet()


def add_builtin_function(
    env: "Environment", name: str, params: list[str], rest: str = None
) -> Address:
    function_info = {
        "type": "builtin_function",
        "enumerable_values": [],
        "params": params,
        "allocation_site": name,
        "rest": rest,
        "function_id": name,
    }
    function_object = env.abstract_heap.add(
        {
            "__meta__": function_info,
        }
    )
    function_object_address = Address(function_object, "abstract")
    if env.cur_stack_frame:
        env.cur_stack_frame.add_allocated_address(function_object_address)
    env.builtin_addresses.append(function_object_address)
    return function_object_address


# region document DOM builtins
def getElementById(env: "Environment") -> RecordResult:
    return env.lookup("element")


def createElement(env: "Environment") -> RecordResult:
    return env.lookup("element")


def querySelector(env: "Environment") -> RecordResult:
    return env.lookup("element")


def querySelectorAll(env: "Environment") -> RecordResult:
    query_result = {baseType.NUMBER: env.lookup("element")}
    addr = env.add_object_to_abstract_heap(query_result)
    return RecordResult("global", [addr])


def addEventListener(env: "Environment", event: str, callbacks: OrderedSet) -> None:
    for callback in callbacks:
        if env.is_function(callback):
            logger.info(f"adding entry point function {callback}")
            env.entrypoint_functions.add(callback)


def appendChild(env: "Environment") -> None:
    pass


def setAttribute(env: "Environment") -> None:
    pass


# endregion


def initialize_dom_builtin(env: "Environment") -> None:
    global_heap_frame = env.lookup_address(
        env.global_stack_frame.get_heap_frame_address()
    )

    document = {}
    document["querySelector"] = [
        add_builtin_function(env, "querySelector", ["selector"])
    ]
    document["querySelectorAll"] = [
        add_builtin_function(env, "querySelectorAll", ["selector"])
    ]
    document["getElementById"] = [
        add_builtin_function(env, "getElementById", ["element_id"])
    ]
    document["createElement"] = [
        add_builtin_function(env, "createElement", ["tag_name"])
    ]
    event_listener = add_builtin_function(
        env, "addEventListener", ["event", "callback"]
    )
    element = {}
    element["appendChild"] = [
        add_builtin_function(env, "appendChild", ["parent", "child"])
    ]
    element["setAttribute"] = [
        add_builtin_function(env, "setAttribute", ["attribute", "value"])
    ]
    element["addEventListener"] = [event_listener]
    document["addEventListener"] = [event_listener]
    element[baseType.STRING] = [baseType.STRING]
    element["innerHTML"] = []
    document_addr = env.add_object_to_abstract_heap(document)
    element_addr = env.add_object_to_abstract_heap(element)
    env.add_value_for_field(element_addr, baseType.STRING, element_addr)
    children_addr = env.add_object_to_abstract_heap({baseType.NUMBER: [element_addr]})
    env.add_value_for_field(element_addr, "children", children_addr)
    # env.builtin_addresses.append(document_addr)
    env.builtin_addresses.append(element_addr)
    global_heap_frame["document"] = RecordResult("global", [document_addr])
    global_heap_frame["element"] = RecordResult("global", [element_addr])


def add_object_prototype(env: "Environment") -> None:
    if env.object_prototype:
        return
    obj_prototype = {}
    """
    obj_prototype["toString"] = [baseType.TOP]
    obj_prototype["valueOf"] = [baseType.TOP]
    obj_prototype["hasOwnProperty"] = [baseType.TOP]
    obj_prototype["isPrototypeOf"] = [baseType.TOP]
    obj_prototype["propertyIsEnumerable"] = [baseType.TOP]
    obj_prototype["toLocaleString"] = [baseType.TOP]
    """
    obj_prototype["__proto__"] = []
    addr = env.add_object_to_abstract_heap(obj_prototype)
    env.object_prototype = addr
