"""All of the features for summarizing addresses and primitives."""

from typing import Tuple, Dict, Union
import esprima
import copy
from openai import OpenAI
import json
import beeprint
import gc
import logging
from ordered_set import OrderedSet
from typing import TYPE_CHECKING

from absint_ai.Environment.memory.RecordResult import RecordResult
from absint_ai.Environment.types.Type import *
from absint_ai.utils.Util import *

if TYPE_CHECKING:
    from absint_ai.Environment.Environment import Environment


# Use this to decide which variables to abstract.
def is_local(env: "Environment", name: str) -> bool:
    if name in env.cur_stack_frame.get_variable_names():
        return True
    heap_frame_addr = env.cur_stack_frame.get_heap_frame_address()
    heap_frame = env.lookup_address(heap_frame_addr)
    if name in heap_frame:
        return True
    return False


# Merges everything from record_result2 into record_result1
def merge_record_results(
    env: "Environment", record_result1: RecordResult, record_result2: RecordResult
) -> bool:
    record_result_address_values = [
        env.lookup_address(_)
        for _ in record_result1.get_all_values()
        if isinstance(_, Address)
    ]
    added_value = False
    all_values = record_result1.get_all_values()
    for value in record_result2.get_all_values():
        if value in all_values:
            continue
        if isinstance(value, Address) and env.is_object(value):
            value_obj = env.lookup_and_derive_address(value, ignore_proto=True)
            if not item_contained_in_list(value_obj, record_result_address_values):
                added_value = True
                record_result1.add_value(value)
        else:
            if not item_contained_in_list(value, all_values):
                added_value = True
                record_result1.add_value(value)
    return added_value


# We are assuming that the abstraction is valid at this point
def merge_abstraction_for_primitive_var(
    env: "Environment", var_name: str, abstraction_values: list
) -> None:
    values = [env.value_to_type(value) for value in abstraction_values]
    env.add(var_name, values, overwrite=True, is_abstraction=True)
    abstract_var_name(env, var_name)


def abstract_var_name(env: "Environment", name: str) -> None:
    record_result = env.lookup(name)
    for value in record_result.get_all_values():
        if isinstance(value, Address) and value.get_addr_type() == "concrete":
            env.move_object_to_abstract_heap(value)


# given a list of addresses, remap each address to the new address for everything in the environment
def abstract_addresses(
    env: "Environment",
    addresses: list[Address] | OrderedSet[Address],
    abstract_address: Address,
) -> None:
    addresses = [
        Address(_.get_value(), _.get_addr_type())
        for _ in addresses
        if _ != abstract_address
    ]
    addresses = list(OrderedSet(addresses))
    if not len(addresses) > 0:
        return
    original_addresses = [copy.deepcopy(address) for address in addresses]

    def update_addresses_in_iterable(
        record_result: RecordResult | list[Type],
    ) -> None:
        for value in record_result:
            if isinstance(value, Address):
                if value in original_addresses:
                    value.set_value(abstract_address.get_value())
                    value.set_addr_type("abstract")
        if isinstance(record_result, RecordResult):
            record_result.deduplicate_values()

    for record_result in env.get_all_variable_record_results():
        update_addresses_in_iterable(record_result)

    for func_addr in env.entrypoint_functions:
        if func_addr in original_addresses:
            func_addr.set_value(abstract_address.get_value())
            func_addr.set_addr_type("abstract")

    for addr in env.concrete_heap.addresses():
        obj = env.concrete_heap.get(addr)
        for field_name, values in obj.items():
            if field_name == "__meta__":
                if "__parent__" in values:
                    if values["__parent__"] in original_addresses:
                        values["__parent__"] = abstract_address
                continue
            if abstract_address in values.get_all_values():
                for original_address in original_addresses:
                    values.remove_value(original_address)
                continue
            update_addresses_in_iterable(values)
    for addr in env.abstract_heap.addresses():
        obj = env.abstract_heap.get(addr)
        for field_name, values in obj.items():
            if field_name == "__meta__":
                if "__parent__" in values:
                    if values["__parent__"] in original_addresses:
                        values["__parent__"] = abstract_address
                continue
            if abstract_address in values.get_all_values():
                for original_address in original_addresses:
                    values.remove_value(original_address)
                continue
            update_addresses_in_iterable(values)
    for module_name, stack in env.stack.items():
        for stack_frame in stack:
            update_addresses_in_iterable(stack_frame.touched_addresses)
            update_addresses_in_iterable(stack_frame.allocated_addresses)
            update_addresses_in_iterable(stack_frame.return_values)
    update_addresses_in_iterable(env.cur_stack_frame.touched_addresses)
    update_addresses_in_iterable(env.cur_stack_frame.allocated_addresses)
    update_addresses_in_iterable(env.cur_stack_frame.return_values)
    for allocation_site in env.allocation_sites:
        summary_addresses = env.allocation_sites[allocation_site]["summary_addresses"]
        update_addresses_in_iterable(summary_addresses)
        concrete_address_values = env.allocation_sites[allocation_site][
            "concrete_addresses"
        ]
        update_addresses_in_iterable(concrete_address_values)
    for memoized_function in env.memoized_functions:
        touched_addresses = memoized_function.get_touched_address_values()
        if touched_addresses:
            update_addresses_in_iterable(touched_addresses)
        allocated_addresses = memoized_function.get_allocated_addresses()
        if allocated_addresses:
            update_addresses_in_iterable(allocated_addresses)
        return_values = memoized_function.get_return_values()
        if return_values:
            update_addresses_in_iterable(return_values)
    # logger.info(f"popping {original_addresses}")
    for original_address in original_addresses:
        if original_address.get_addr_type() == "concrete":
            # logger.info(f"popping {original_address} from abstract_address")
            env.concrete_heap.pop(original_address)
        elif original_address.get_addr_type() == "abstract":
            # logger.info(f"popping {original_address} from abstract_address")
            env.abstracted_addresses_map[str(original_address)] = str(abstract_address)
            env.abstract_heap.pop(original_address)


def simplify_heap_frame(env: "Environment", heap_frame: Address) -> None:
    # logger.info(f"Simplifying heap frame {heap_frame} at {heap_frame}")
    obj = env.lookup_address(heap_frame)
    for field, values in obj.items():
        if field == "__meta__" or field == "exports" or field == "module":
            continue
        for value in values.get_all_values():
            if isinstance(value, Address):
                simplify_address(env, value)
    # The keys in the heap frame are just variable names, so we don't need to merge them. Just the values
    for field, values in obj.items():
        if field == "__meta__":
            continue
        primitive_values = values.get_all_primitives()
        address_values = values.get_all_addresses()
        objects = [_ for _ in address_values if env.is_object(_)]
        functions = [_ for _ in address_values if env.is_function(_)]
        classes = [_ for _ in address_values if env.is_class(_)]
        if len(primitive_values) > 1:
            values.merge_primitives()
        if len(objects) > 1:
            merge_all_addresses_in_record_result(env, values)
        if len(functions) > 1:
            merge_all_addresses_in_record_result(env, values)


def simplify_fields_for_address(
    env: "Environment", address: Address, fields_to_abstract
) -> None:
    obj = env.lookup_address(address)
    for field, values in obj.items():
        if field == "__meta__":
            continue
        if (
            field in fields_to_abstract or str(field) in fields_to_abstract
        ) and isinstance(values, RecordResult):
            simplify_record_result(env, values)


def simplify_record_result(env: "Environment", record_result: RecordResult) -> None:
    record_result.merge_primitives()
    merge_all_addresses_in_record_result(env, record_result, simplify=True)


def simplify_address(
    env: "Environment", address: Address, seen: list = None
) -> None:  # Recursively simplifies an address
    if seen is None:
        seen = []
    if address in seen or address in env.builtin_addresses:
        return
    if env.get_object_type(address) != "object":
        return
    seen.append(address)
    obj: dict[str | AbstractType, RecordResult] = env.lookup_address(address)
    # logger.info(f"simplifying {address} {beeprint.pp(obj,output=False)}")
    simplified_obj: dict[str | AbstractType, RecordResult] = {}
    new_enumerable_values = OrderedSet()
    for field, values in obj.items():
        if field == "__meta__" or field == "__proto__":
            simplified_obj[field] = values
            continue
        if isinstance(field, AbstractType):
            if field not in simplified_obj:
                simplified_obj[field] = values
            else:
                merge_record_results(env, simplified_obj[field], values)
                # simplified_obj[field].merge_other_record_result(values)
            new_enumerable_values.add(field)
        elif is_integer(field) or isinstance(field, (int, float)):
            if baseType.NUMBER in simplified_obj:
                merge_record_results(env, simplified_obj[baseType.NUMBER], values)
                # simplified_obj[baseType.NUMBER].merge_other_record_result(values)
            else:
                simplified_obj[baseType.NUMBER] = values
                new_enumerable_values.add(baseType.NUMBER)
        elif isinstance(field, str):
            if baseType.STRING in simplified_obj:
                merge_record_results(env, simplified_obj[baseType.STRING], values)
                # simplified_obj[baseType.STRING].merge_other_record_result(values)
            else:
                simplified_obj[baseType.STRING] = values
            new_enumerable_values.add(baseType.STRING)
        else:
            # logger.info(f"Field {field} is not a string or number, defaulting to TOP")
            if baseType.TOP in simplified_obj:
                merge_record_results(env, simplified_obj[baseType.TOP], values)
                # simplified_obj[baseType.TOP].merge_other_record_result(values)
            else:
                simplified_obj[baseType.TOP] = values
            new_enumerable_values.add(baseType.TOP)
    try:
        simplified_obj["__meta__"]["enumerable_values"] = new_enumerable_values
    except Exception:
        logger.info(f"{address}: {simplified_obj}")
        raise Exception("Failed to simplify")

    if address.get_addr_type() == "concrete":
        env.concrete_heap.overwrite_address(address, simplified_obj)
    elif address.get_addr_type() == "abstract":
        env.abstract_heap.overwrite_address(address, simplified_obj)
    else:
        raise Exception(f"Unknown address type {address.get_addr_type()}")

    # We've merged together all the fields of the object. Now we need to do it for the values
    fields = list(simplified_obj.keys())
    for field in fields:
        record_result = simplified_obj[field]
        if field == "__meta__":
            continue
        # logger.info(f"merging primitives for {address} with field {field}")
        record_result.merge_primitives()
        merge_all_addresses_in_record_result(
            env, record_result
        )  # This keeps on creating a new address
        # merge_all_functions_in_record_result(env, record_result)
        for value in record_result:
            if isinstance(value, Address):
                simplify_address(env, value, seen=seen)
    final_obj = env.lookup_address(address)
    # logger.info(
    #    f"Final simplified {address} with {beeprint.pp(final_obj,output=False)}"
    # )

def merge_fields_in_address(
    env: "Environment", address: Address
) -> None:  # Recursively simplifies an address
    if env.get_object_type(address) != "object":
        return
    obj: dict[str | AbstractType, RecordResult] = env.lookup_address(address)
    # logger.info(f"simplifying {address} {beeprint.pp(obj,output=False)}")
    simplified_obj: dict[str | AbstractType, RecordResult] = {}
    new_enumerable_values = OrderedSet()
    for field, values in obj.items():
        if field == "__meta__" or field == "__proto__":
            simplified_obj[field] = values
            continue
        if isinstance(field, AbstractType):
            if field not in simplified_obj:
                simplified_obj[field] = values
            else:
                merge_record_results(env, simplified_obj[field], values)
                # simplified_obj[field].merge_other_record_result(values)
            new_enumerable_values.add(field)
        elif is_integer(field) or isinstance(field, (int, float)):
            if baseType.NUMBER in simplified_obj:
                merge_record_results(env, simplified_obj[baseType.NUMBER], values)
                # simplified_obj[baseType.NUMBER].merge_other_record_result(values)
            else:
                simplified_obj[baseType.NUMBER] = values
                new_enumerable_values.add(baseType.NUMBER)
        elif isinstance(field, str):
            if baseType.STRING in simplified_obj:
                merge_record_results(env, simplified_obj[baseType.STRING], values)
                # simplified_obj[baseType.STRING].merge_other_record_result(values)
            else:
                simplified_obj[baseType.STRING] = values
            new_enumerable_values.add(baseType.STRING)
        else:
            # logger.info(f"Field {field} is not a string or number, defaulting to TOP")
            if baseType.TOP in simplified_obj:
                merge_record_results(env, simplified_obj[baseType.TOP], values)
                # simplified_obj[baseType.TOP].merge_other_record_result(values)
            else:
                simplified_obj[baseType.TOP] = values
            new_enumerable_values.add(baseType.TOP)
    try:
        simplified_obj["__meta__"]["enumerable_values"] = new_enumerable_values
    except Exception:
        logger.info(f"{address}: {simplified_obj}")
        raise Exception("Failed to simplify")

    if address.get_addr_type() == "concrete":
        env.concrete_heap.overwrite_address(address, simplified_obj)
    elif address.get_addr_type() == "abstract":
        env.abstract_heap.overwrite_address(address, simplified_obj)
    else:
        raise Exception(f"Unknown address type {address.get_addr_type()}")

# Given two parent addresses, add parent_addr2 to parent_addr1 as well as their parents
def merge_parent_addresses(
    env: "Environment", parent_addr1: Address, parent_addr2: Address
) -> Address:
    env.join_addresses(parent_addr1, parent_addr2, add_null=False)
    return parent_addr1


# Merges the prototype and function object and parents from func2 into func1
def join_functions(env: "Environment", func1: Address, func2: Address) -> None:
    func1_meta = env.get_meta(func1)
    func2_meta = env.get_meta(func2)
    func1_obj = env.lookup_address(func1)
    func2_obj = env.lookup_address(func2)
    if func1_meta.get("allocation_site") != func2_meta.get("allocation_site"):
        logger.info(
            f"not merging! {func1_meta.get('allocation_site')} {func2_meta.get('allocation_site')}"
        )
        return
    if func1_meta["type"] != "function" or func2_meta["type"] != "function":
        raise Exception("Trying to merge non-functions")
    func1_prototype = func1_obj["prototype"][0]
    func2_prototype = func2_obj["prototype"][0]
    env.move_object_to_abstract_heap(func1_prototype)
    env.join_addresses(func1_prototype, func2_prototype, add_null=False)
    abstract_addresses(env, [func2_prototype], func1_prototype)
    func1_parent = func1_meta.get("parent", None)
    func2_parent = func2_meta.get("parent", None)
    if not func1_parent and not func2_parent:
        return
    if not func1_parent:
        merged_parent = func2_parent
    elif not func2_parent:
        merged_parent = func1_parent
    elif func1_parent != func2_parent:
        merged_parent = merge_parent_addresses(env, func1_parent, func2_parent)
        abstract_addresses(env, [func1_parent, func2_parent], merged_parent)
    else:
        merged_parent = func1_parent
    env.set_parent_heap_frame(func1, merged_parent)
    logger.info(f"merging functions {func1} with {func2}")
    logger.info(f"calling join addresses with {func1} and {func2}")
    abstract_addresses(env, [func2], func1)


def merge_functions_with_same_allocation_site(
    env: "Environment", functions: list[Address]
) -> None:
    first_function = list(functions)[0]
    assert all(
        [
            env.get_meta(func).get("allocation_site")
            == env.get_meta(first_function).get("allocation_site")
            and env.get_meta(func)["type"] == "function"
            for func in functions
        ]
    )

    allocation_site = env.get_meta(list(functions)[0]).get("allocation_site", False)
    # logger.info(
    #    f"merging {len(functions)} functions with allocation site {allocation_site}"
    # )
    if allocation_site not in env.allocation_sites:
        logger.info(env.get_meta(list(functions)[0]))
        raise Exception(
            f"Allocation site {allocation_site} not found in allocation sites"
        )
    allocation_site_info = env.allocation_sites[allocation_site]
    if len(allocation_site_info["summary_addresses"]) == 0:
        env.move_object_to_abstract_heap(first_function)
        abstract_function = first_function
        allocation_site_info["summary_addresses"].add(abstract_function)
    else:
        abstract_function = allocation_site_info["summary_addresses"][0]
    for func in functions:
        if func == abstract_function:
            continue
        join_functions(env, abstract_function, func)
    abstract_addresses(env, functions, abstract_function)


def merge_all_functions_in_record_result(
    env: "Environment", record_result: RecordResult
) -> None:
    logger.info(f"MERGING FUNCTIONS IN RECORD RESULT")
    all_functions = [_ for _ in record_result.get_all_values() if env.is_function(_)]
    all_functions_copy = copy.deepcopy(all_functions)
    functions_map = (
        {}
    )  # Maps the function to the function that it should be merged with
    for function in all_functions:
        function_id = function.get_allocation_site()
        functions_map.setdefault(function_id, []).append(function)
    for function_id, functions in functions_map.items():
        merge_functions_with_same_allocation_site(env, functions)


# Merges all the objects in a record result into a single object, then returns the address
def merge_all_addresses_in_record_result(
    env: "Environment", record_result: RecordResult, simplify: bool = False
) -> Address:
    # This merges functions and objects together. We should be keeping
    # them separate
    addresses = [_ for _ in record_result.get_all_addresses() if env.is_object(_)]

    if len(addresses) < 2:
        if len(addresses) == 1 and simplify:
            simplify_address(env, addresses[0])
        if len(addresses) == 1:
            return addresses[0]
        return None
    abstract_address = env.add_object_to_abstract_heap({}, add_proto=False)
    addresses_to_abstract = copy.deepcopy(list(addresses))
    for address in addresses:
        join_addresses(env, abstract_address, address)
        record_result.remove_value(address)
    record_result.add_value(abstract_address)
    if simplify:
        simplify_address(env, abstract_address)
    abstract_addresses(
        env,
        [
            addr
            for addr in addresses_to_abstract
            if str(addr) not in env.abstracted_addresses_map
        ],
        abstract_address,
    )
    return abstract_address
    # self.abstract_addresses(record_result.get_all_addresses(), abstract_address)


def join_addresses(
    env: "Environment", addr1: Address, addr2: Address, add_null: bool = True
) -> (
    bool
):  # Adds all the values from addr2 to addr1 and returns whether anything changed
    if env.is_class(addr1) or env.is_class(addr2):
        return False
    assert addr1.get_addr_type() == "abstract"
    if addr2 == addr1:
        return False

    obj1 = env.lookup_address(addr1)
    obj2 = env.lookup_address(addr2)
    if obj1["__meta__"]["type"] != obj2["__meta__"]["type"]:
        return False
    is_empty = False
    if len(obj1) == 0 or (len(obj1) == 1 and "__meta__" in obj1):
        is_empty = True
    modified = False
    # logger.info(f"obj1 before: {beeprint.pp(obj1, output=False)}")
    for field, values in obj2.items():
        if field == "__meta__":
            if (
                "enumerable_values" in obj1["__meta__"]
                and "enumerable_values" in obj2["__meta__"]
            ):
                obj1[field]["enumerable_values"].update(values["enumerable_values"])
                obj1[field]["enumerable_values"] = OrderedSet(
                    obj1[field]["enumerable_values"]
                )
                # )
            if "__parent__" in obj2["__meta__"] and obj2["__meta__"]["__parent__"]:
                if "__parent__" in obj1["__meta__"] and obj1["__meta__"]["__parent__"]:
                    obj1[field]["__parent__"] = merge_parent_addresses(
                        env,
                        obj1["__meta__"]["__parent__"],
                        obj2["__meta__"]["__parent__"],
                    )
                else:
                    obj1[field]["__parent__"] = obj2["__meta__"]["__parent__"]
            if "__parent__" not in obj1["__meta__"]:
                if "__parent__" in obj2["__meta__"]:
                    obj1[field]["__parent__"] = obj2["__meta__"]["__parent__"]
            continue
        if field == "__proto__":
            if field in obj1:
                added_new_value = merge_record_results(env, obj1[field], values)
                # added_new_value = obj1[field].merge_other_record_result(values)
                # logger.info(
                #     f"ADDED NEW VALUE {values} {obj1[field]} {added_new_value}"
                # )
                if added_new_value:
                    modified = True
            else:
                obj1[field] = values
                modified = True
        field_in_obj1 = field_in_object(
            field, obj1
        )  # If it's __proto__ and there's a string it'll combine them
        if field_in_obj1:
            added_new_value = merge_record_results(env, obj1[field_in_obj1], values)
            # added_new_value = obj1[field_in_obj1].merge_other_record_result(values)
            # logger.info(
            #     f"ADDED NEW VALUE {values} {obj1[field_in_obj1]} {added_new_value}"
            # )
            if added_new_value:
                modified = True
        else:
            modified = True
            obj1[field] = values
            if add_null:
                obj1[field].add_value(baseType.NULL)
    env.abstract_heap.overwrite_address(addr1, obj1)
    # logger.info(f"obj1 after: {beeprint.pp(obj1, output=False)}")
    if not is_empty:
        return modified
    return False


def get_referenced_addresses_from_variables(
    env: "Environment",
    variables: list[str],
) -> Tuple[list[Address], list[Address]]:
    referenced_concrete_addresses = OrderedSet()
    referenced_abstract_addresses = OrderedSet()

    def get_referenced_address_from_obj(obj: dict):
        for _, values in obj.items():
            for value in values:
                if isinstance(value, Address):
                    if value.get_addr_type() == "concrete":
                        referenced_concrete_addresses.add(value)
                    elif value.get_addr_type() == "abstract":
                        referenced_abstract_addresses.add(value)

    for var in variables:
        record_result = env.lookup(var)
        for value in record_result:
            if isinstance(value, Address):
                if value.get_addr_type() == "concrete":
                    referenced_concrete_addresses.add(value)
                elif value.get_addr_type() == "abstract":
                    referenced_abstract_addresses.add(value)
                obj = env.lookup_address(value)
                get_referenced_address_from_obj(obj)
    return list(referenced_concrete_addresses), list(referenced_abstract_addresses)


def validate_address_abstraction(
    env: "Environment", address: Address, address_abstraction: dict
) -> bool:
    try:
        env.lookup_address(address)
    except Exception as e:
        logger.info(f"Failed to lookup address {address} {e}")
        return
    if (isinstance(address_abstraction, list) and "TOP" in address_abstraction) or (
        address_abstraction == "TOP"
    ):
        return True
    if not isinstance(address_abstraction, dict):
        return False
    if address.get_addr_type() == "concrete":
        # this breaks if it gives an address that doesn't exist TODO
        if not env.concrete_heap.contains(address.get_value()):
            return False
        obj = env.concrete_heap.get(address)
    elif address.get_addr_type() == "abstract":
        if not env.abstract_heap.contains(address.get_value()):
            return False
        obj = env.abstract_heap.get(address)
    else:
        raise Exception(f"Unknown address type {address.get_addr_type()}")
    try:
        keys_to_ignore = ["__proto__", "__meta__", "exports", "module"]
        new_obj = {
            k: v.get_all_values() for k, v in obj.items() if k not in keys_to_ignore
        }
    except Exception as e:
        raise Exception(
            f"Failed to validate address abstraction {address_abstraction}, error: {e}"
        )
    logger.info(f"validating {new_obj} with {address_abstraction}")
    if new_obj == {}:  # If the object is empty, object_is_superset always returns True
        return False
    return object_is_superset(new_obj, address_abstraction)


def validate_all_variable_abstractions(env: "Environment", abstractions: dict) -> bool:
    for var_name in abstractions:
        if not validate_variable_abstraction(env, var_name, abstractions[var_name]):
            logger.info(
                f"Failed to validate {var_name} with values {abstractions[var_name]}"
            )
            return False
    return True


def validate_variable_abstraction(
    env: "Environment", var_name: str, abstraction_values: list
) -> bool:
    # if "TOP" in abstraction_values:
    #    logger.info(
    #        f"We don't want to validate TOP {abstraction_values} for {var_name}"
    #    )
    #    return False
    all_values_for_var_name = env.lookup(var_name).get_all_values()
    if all(
        [isinstance(value, Address) for value in all_values_for_var_name]
    ):  # Don't bother abstracting this for now unless it's TOP. This should be handled by the address abstractions.
        return False
    logger.info(f"validating {all_values_for_var_name} with {abstraction_values}")
    for value in all_values_for_var_name:
        if isinstance(value, Primitive) or isinstance(value, AbstractType):
            if value == baseType.TOP:
                if "TOP" not in abstraction_values:
                    return False
            if (
                not isinstance(value, AbstractType)
                and isinstance(value.get_value(), str)
            ) or value == baseType.STRING:  # if it's a string, make sure either the value is in the abstraction or STRING or TOP
                if "STRING" in abstraction_values or "TOP" in abstraction_values:
                    continue
                elif value.get_value() not in abstraction_values:
                    return False
            if (
                isinstance(value.get_value(), (int, float)) or value == baseType.NUMBER
            ):  # if it's a number, make sure either the value is in the abstraction or NUMBER or TOP
                if "NUMBER" in abstraction_values or "TOP" in abstraction_values:
                    continue
                elif value.get_value() not in abstraction_values:
                    return False
            if value == baseType.NULL:
                if (
                    "NULL" in abstraction_values
                    or "TOP" in abstraction_values
                    or "null" in abstraction_values
                ):
                    continue
                else:
                    return False
            if value == baseType.BOOLEAN:
                if "BOOLEAN" in abstraction_values or "TOP" in abstraction_values:
                    continue
                else:
                    return False

        elif isinstance(value, Address) or is_address(value):
            value_processed = env.type_to_value(value)
            logger.info(
                f"Validating {var_name} with values {abstraction_values} and {value_processed}"
            )
            # value_processed = json.loads(json.dumps(value_processed))
            found_abstraction = False
            for abstraction_value in abstraction_values:
                if object_is_superset(value_processed, abstraction_value):
                    found_abstraction = True
                    break
            if not found_abstraction:
                return False
    return True
