from absint_ai.Environment.types.Type import *
from absint_ai.utils.Logger import logger  # type: ignore
from absint_ai.Environment.memory.RecordResult import RecordResult
from ordered_set import OrderedSet
from absint_ai.utils.Util import *

global abstract_heap_id
abstract_heap_id = 0


def abstract_heap_id_reset() -> None:
    global abstract_heap_id
    abstract_heap_id = 0


class AbstractHeap:
    def __init__(
        self,
        abstract_heap: dict[str, dict[str | AbstractType, RecordResult]] | None = None,
        abstracted_addresses: dict[str, str] | None = None,
        track_modified_addresses: bool = False,
    ):
        if abstract_heap is None:
            self.abstract_heap = {}
        else:
            self.abstract_heap = abstract_heap
        self.track_modified_addresses = track_modified_addresses
        self.modified_addresses = (
            OrderedSet()
        )  # use this to track what needs to be merged after a conditional

    def add(self, obj: dict, addr: Address = None) -> str:
        global abstract_heap_id
        converted_to_record_results = {}
        for key, value in obj.items():
            if key == "__meta__":
                converted_to_record_results[key] = value
            elif isinstance(value, list) or isinstance(value, OrderedSet):
                converted_to_record_results[key] = RecordResult("global", value)
            elif isinstance(value, RecordResult):
                converted_to_record_results[key] = value
            else:
                raise ValueError(
                    f"Invalid value type being added to heap: {type(value)}"
                )
        if addr is not None:
            self.abstract_heap[str(addr.get_value())] = converted_to_record_results
            if self.track_modified_addresses:
                self.modified_addresses.add(str(addr.get_value()))
            return str(addr.get_value())
        else:
            self.abstract_heap[str(abstract_heap_id)] = converted_to_record_results
            abstract_heap_id += 1
            return str(abstract_heap_id - 1)

    def contains(self, abstract_heap_id: str) -> int:
        return abstract_heap_id in self.abstract_heap

    def get(self, addr: Address) -> dict[str | AbstractType, RecordResult]:
        if not isinstance(addr, Address) or addr.get_addr_type() != "abstract":
            raise ValueError(f"Address {addr} is not abstract")
        abstract_heap_id = addr.get_value()
        if abstract_heap_id not in self.abstract_heap:
            raise ValueError(f"Address {abstract_heap_id} not found in abstract heap")
        return self.abstract_heap[abstract_heap_id]

    def merge(self, other: "AbstractHeap") -> None:
        for address in other.addresses():
            if address.get_value() not in self.abstract_heap:
                self.abstract_heap[address.get_value()] = other.get(address)
            else:
                for key, value in other.get(address).items():
                    if key == "__meta__":
                        continue
                    if key not in self.abstract_heap[address.get_value()]:
                        self.abstract_heap[address.get_value()][key] = value
                    else:
                        for val in value.get_all_values():
                            if not item_contained_in_list(
                                val,
                                self.abstract_heap[address.get_value()][
                                    key
                                ].get_all_values(),
                            ):
                                self.abstract_heap[address.get_value()][key].add_value(
                                    val
                                )
        if self.track_modified_addresses:
            self.modified_addresses.update(other.modified_addresses)

    def update(
        self,
        abstract_heap_id: str,
        property: str,
        values: list[Type],
    ) -> None:
        if isinstance(values, list) or isinstance(values, OrderedSet):
            values = RecordResult("global", values)  # type: ignore
        if property in self.abstract_heap[abstract_heap_id]:
            for value in values:
                if not item_contained_in_list(
                    value,
                    self.abstract_heap[abstract_heap_id][property].get_all_values(),
                ):
                    self.abstract_heap[abstract_heap_id][property].add_value(value)
        elif (
            isinstance(property, (int, float))
            and baseType.NUMBER in self.abstract_heap[abstract_heap_id]
        ):
            record_result_for_field = self.abstract_heap[abstract_heap_id][
                baseType.NUMBER
            ]
            for value in values:
                if not isinstance(value, Address):
                    if not item_contained_in_list(
                        value, record_result_for_field.get_all_values()
                    ):
                        record_result_for_field.add_value(value)
                else:
                    self.abstract_heap[abstract_heap_id][baseType.NUMBER].add_value(
                        value
                    )

        elif (
            isinstance(property, str)
            and baseType.STRING in self.abstract_heap[abstract_heap_id]
        ):
            record_result_for_field = self.abstract_heap[abstract_heap_id][
                baseType.STRING
            ]
            for value in values:
                if not isinstance(value, Address):
                    if not item_contained_in_list(
                        value, record_result_for_field.get_all_values()
                    ):
                        record_result_for_field.add_value(value)
                else:
                    self.abstract_heap[abstract_heap_id][baseType.STRING].add_value(
                        value
                    )
        else:
            if (
                property
                not in self.abstract_heap[abstract_heap_id]["__meta__"][
                    "enumerable_values"
                ]
            ):
                self.abstract_heap[abstract_heap_id]["__meta__"][
                    "enumerable_values"
                ].append(property)
            if property not in self.abstract_heap[abstract_heap_id]:
                self.abstract_heap[abstract_heap_id][property] = values  # type: ignore

    def overwrite_address(
        self, addr: Address, value: dict[str | AbstractType, RecordResult]
    ) -> None:
        if addr.get_addr_type() != "abstract":
            raise ValueError(f"Address {addr} is not abstract")
        heap_id = addr.get_value()
        if heap_id not in self.abstract_heap:
            raise ValueError(f"Address {heap_id} not found in abstract heap")
        self.abstract_heap[heap_id] = value

    def pop(self, addr: Address) -> None:
        if addr.get_addr_type() != "abstract":
            raise ValueError(f"Address {addr} is not abstract")
        heap_id = addr.get_value()
        if heap_id in self.abstract_heap:
            self.abstract_heap.pop(heap_id)
        else:
            raise ValueError(f"Address {heap_id} not found in abstract heap")

    def addresses(self) -> list[Address]:
        unmerged_addresses = [
            Address(value=addr, addr_type="abstract")
            for addr in self.abstract_heap.keys()
        ]
        return unmerged_addresses
