from absint_ai.utils.Logger import logger  # type: ignore
from absint_ai.Environment.types.Type import *
from .RecordResult import RecordResult


class StackFrame:
    def __init__(self, heap_frame: Address, variables: dict[str, RecordResult] = None, is_function=True):  # type: ignore
        self.heap_frame = heap_frame
        if variables is None:
            variables = {}
        self.variables: dict[str, RecordResult] = variables
        self.return_values: list[Type] = []
        self.touched_addresses: dict[Address, object] = {}
        self.touched_primitives: dict[str, list[Type]] = {}
        self.allocated_addresses: list[Address] = (
            []
        )  # If the address is allocated in this function call
        self.has_recursive_call: bool = False
        self.is_function: bool = is_function
        self._type: str = None  # type: ignore

    def __str__(self) -> str:
        return f"StackFrame({self.heap_frame})"

    def __repr__(self) -> str:
        return self.__str__()

    def get_variable_names(self) -> list[str]:
        return list(self.variables.keys())

    def get_variable(self, name: str) -> RecordResult:
        return self.variables[name]

    def add_variable(self, name: str, value: object) -> None:
        if name not in self.variables:
            self.variables[name] = RecordResult(self._type)
        self.variables[name].add_value(value)

    def add_allocated_address(self, address: Address) -> None:
        if address not in self.allocated_addresses:
            self.allocated_addresses.append(address)

    def get_allocated_addresses(self) -> list[Address]:
        return self.allocated_addresses

    def merge(self, other: "StackFrame") -> None:
        for name, record_result in other.variables.items():
            if name in self.variables:
                for value in record_result.get_all_values():
                    self.variables[name].add_value(value)
            else:
                self.variables[name] = record_result

    def add_touched_address(self, address: Address, value: object) -> None:
        if address not in self.touched_addresses:
            self.touched_addresses[address] = value

    def remove_touched_address(self, address: Address) -> None:
        if address in self.touched_addresses:
            del self.touched_addresses[address]

    def add_touched_primitive(self, var_name: str, value: Type) -> None:
        if var_name not in self.touched_primitives:
            self.touched_primitives[var_name] = [value]
        elif value not in self.touched_primitives[var_name]:
            self.touched_primitives[var_name].append(value)

    def get_touched_addresses(self) -> dict[Address, object]:
        return self.touched_addresses

    def get_touched_primitives(self) -> dict[str : list[Type]]:
        return self.touched_primitives

    def set_has_recursive_call(self, has_recursive_call: bool) -> None:
        self.has_recursive_call = has_recursive_call

    def add_return_value(self, return_value: Type) -> None:
        if return_value not in self.return_values:
            self.return_values.append(return_value)

    def get_return_values(self) -> list[Type]:
        return self.return_values

    def get_heap_frame_address(self) -> Address:
        return self.heap_frame

    def contains_this_address(self) -> bool:
        return "this" in self.variables

    def get_this_address(self) -> Address:
        return list(self.variables["this"].get_all_values())[0]
