import time
from absint_ai.utils.Util import *
from typing import TYPE_CHECKING
from ordered_set import OrderedSet
from absint_ai.Environment.types.Type import *
from absint_ai.Interpreter.visitors.helpers.function_executor import FunctionExecutor
import esprima
import os
import pythonmonkey as pm
import copy
from dotmap import DotMap
import math

# Have to do this because the initial implementation just returns __class__ for any unregistered keys. Weird.
DotMap.__getitem__ = DotMap.get  # type: ignore

if TYPE_CHECKING:
    from absint_ai.Environment.Environment import Environment
    from absint_ai.Interpreter.AbsIntAI import AbsIntAI
    from .statement_visitor import StatementVisitor


class ExprVisitor:
    """Visitor for expressions."""

    def __init__(self, env: "Environment", absint_ai: "AbsIntAI"):
        self.env = env
        self.statement_visitor = None
        self.absint_ai = absint_ai
        self.function_executor: "FunctionExecutor" = None

    def visit(self, node, file_path) -> OrderedSet[Type]:
        if node is None:
            return OrderedSet()
        if not isinstance(node, dict) or "type" not in node:
            raise ValueError("Invalid AST node format")

        node_type = node["type"]
        method_name = f"visit_{node_type}"
        visitor = getattr(self, method_name, self.visit_default)
        return visitor(node, file_path)

    def visit_default(self, node, file_path):
        raise NotImplementedError(f"No visitor method for {node['type']}")

    def set_statement_visitor(self, statement_visitor: "StatementVisitor"):
        self.statement_visitor = statement_visitor
        self.set_function_executor()

    def set_function_executor(self):
        assert self.statement_visitor is not None
        self.function_executor = FunctionExecutor(
            self.env, self.absint_ai, self, self.statement_visitor
        )

    def visit_Literal(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        if node.value is None:
            return OrderedSet([baseType.NULL])
        if "regex" in node:
            return OrderedSet([Primitive(node.raw, is_regex=True)])
        if isinstance(node.value, DotMap):
            return OrderedSet([Primitive(node.raw)])
        return OrderedSet([Primitive(node.value)])

    def visit_MemberExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        if baseType.TOP in self.visit(node.object, file_path):
            if node.computed:
                properties = self.visit(node.property, file_path)
            else:
                properties = [Primitive(node.property.name)]

        try:
            possible_values = self.visit(node.object, file_path)
        except Exception as e:
            logger.debug(f"ERROR on line {self.absint_ai.get_loc(node)}")
            raise e
        propTypes = OrderedSet()
        possible_addresses = [
            _
            for _ in possible_values
            if (
                isinstance(_, Address)
                or (isinstance(_, Primitive) and isinstance(_.get_value(), str))
            )
        ]
        if not len(possible_addresses):
            self.absint_ai.buggy_lines.append(
                f"line {self.absint_ai.get_loc(node)}: {get_identifiers_from_expr(node.object)} not found for file {file_path} MemberExpression object is {possible_values}"
            )
            self.absint_ai.buggy_line_numbers.append(self.absint_ai.get_loc_id(node))
        for address in possible_values:
            if address in self.env.builtin_addresses:
                continue
            if node.computed:
                properties = self.visit(node.property, file_path)
            else:
                properties = [Primitive(node.property.name)]
            if not isinstance(address, Address):
                # want to handle string lengths
                if isinstance(address, Primitive):
                    if isinstance(address.get_value(), str):
                        for prop in properties:
                            if prop == Primitive("length"):
                                propTypes.add(Primitive(len(address.get_value())))
                            elif isinstance(prop, Primitive):
                                if isinstance(prop.get_value(), (int, float)):
                                    logger.info(
                                        f"getting {prop.get_value()} from {address.get_value()}"
                                    )
                                    propTypes.add(
                                        Primitive(
                                            address.get_value()[int(prop.get_value())]
                                        )
                                    )
                            else:
                                self.absint_ai.buggy_lines.append(
                                    f"line {self.absint_ai.get_loc(node)}: {prop.get_value()}: {type(prop.get_value())} not found in string"
                                )
                                self.absint_ai.buggy_line_numbers.append(
                                    self.absint_ai.get_loc_id(node)
                                )
                continue
            obj = self.env.lookup_address(address)  # type: ignore
            for prop in properties:
                if prop == Primitive("length"):
                    if "enumerable_values" in obj["__meta__"]:
                        if baseType.NUMBER in obj["__meta__"]["enumerable_values"]:
                            propTypes.add(baseType.NUMBER)
                        else:
                            propTypes.add(
                                Primitive(len(obj["__meta__"]["enumerable_values"]))
                            )
                    else:
                        propTypes.add(baseType.NUMBER)
                    continue
                prop_lookup_results = self.env.lookup_field(
                    address, prop, line_number=self.absint_ai.get_loc(node)
                )

                possible_bug = False
                if isinstance(prop, AbstractType):
                    possible_bug = True
                if prop_lookup_results or prop_in_obj(prop, obj):  # and len(
                    # prop_lookup_results.get_all_values()
                    # ):
                    propTypes.update(prop_lookup_results.get_all_values())  # type: ignore
                else:
                    possible_bug = True
                    abstract_prop_types = self._get_abstract_property(
                        obj, prop, expr=node, file_path=file_path
                    )
                    propTypes.update(abstract_prop_types)
                if possible_bug:
                    self.absint_ai.buggy_lines.append(
                        f"line {self.absint_ai.get_loc(node)}: {prop.get_value()}: {type(prop.get_value())} not found in object {node.object.name} for file {file_path} MemberExpression object is {obj.keys()}"
                    )
                    self.absint_ai.buggy_line_numbers.append(
                        self.absint_ai.get_loc_id(node)
                    )
        return propTypes

    def visit_ThisExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        return self.env.lookup("this").get_all_values()

    def visit_Identifier(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        if node.name == "undefined" or node.name == "null":
            return OrderedSet([baseType.NULL])

        possible_values = self.env.lookup(node.name).get_all_values()
        if possible_values == []:
            # self.buggy_lines.append(
            #    f"line {self.get_loc(node)}: {node.name} not found at  for file {file_path}"
            # )
            # self.buggy_line_numbers.append(self.get_loc_id(node))
            self.absint_ai.identifier_info[get_identifier_id(node)] = ["TOP"]
            return OrderedSet([baseType.TOP])
        self.absint_ai.identifier_info[get_identifier_id(node)] = (
            self.env.lookup_and_derive(node.name)
        )
        return possible_values

    def visit_Super(self, node: esprima.nodes.Node, file_path: str) -> OrderedSet[Type]:
        return self.env.lookup("super").get_all_values()

    def visit_ObjectExpression(self, node, file_path: str) -> OrderedSet[Type]:
        return self._get_possible_values_for_object(node, file_path)

    def visit_ArrayExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        return self._get_possible_values_for_object(node, file_path)

    def visit_AssignmentExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        if node.left.type == "Identifier":
            if node.right.type == "AssignmentExpression":
                self.visit(node.right, file_path)
                possible_values = self.visit(node.right.left, file_path)
            else:
                possible_values = self.visit(node.right, file_path)
            self.env.add(
                node.left.name,
                possible_values,
                overwrite=True,
                kind=node["kind"],
            )

        if node.left.type == "ArrayPattern" or node.left.type == "ObjectPattern":
            possible_addrs = self.visit(node.right, file_path)
            self.statement_visitor.add_destructuring(
                node.left, possible_addrs, file_path
            )

        if node.left.type == "MemberExpression":
            if not node.left.computed:
                possibleProps: list[Type] = [Primitive(node.left.property.name)]
            else:
                possibleProps = self.visit(node.left.property, file_path)
            if node.right.type == "AssignmentExpression":
                self.visit(node.right, file_path)
                possible_right_values = self.visit(node.right.left, file_path)
            else:
                possible_right_values = self.visit(node.right, file_path)
            if node.left.object.type == "ThisExpression":
                self.env.update_this(
                    possibleProps,
                    possible_right_values,
                    overwrite=True,
                )
            else:
                possible_values = self.visit(node.left.object, file_path)
                for value in possible_values:
                    if isinstance(value, Address):
                        self.env.update(
                            value,
                            possibleProps,
                            possible_right_values,
                            overwrite=True,
                            objName=node.left.object.name,
                        )
        if node.left.type == "Expression":
            return self.visit(node.left, file_path)
        elif node.left.type == "Identifier":
            return self.visit(node.left, file_path)
        # TODO Need to handle Patterns on the left here.

    def visit_SequenceExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        for stmt in node.expressions[:-1]:
            self.statement_visitor.visit(stmt, file_path)
        return self.visit(node.expressions[-1], file_path)

    def visit_ConditionalExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        consequent = self.visit(node.consequent, file_path)
        alternate = self.visit(node.alternate, file_path)
        return consequent | alternate

    def visit_UnaryExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        result = OrderedSet()
        arg_values = self.visit(node.argument, file_path)
        for arg_value in arg_values:
            if isinstance(arg_value, AbstractType):
                if arg_value == baseType.NULL:
                    eval_result = Primitive(pm.eval(f"{node.operator} null"))
                    result.add(eval_result)
                else:
                    result.add(arg_value)
            else:
                arg_value = arg_value.get_value()
                if arg_value == True:
                    arg_value = "true"
                elif arg_value == False:
                    arg_value = "false"
                elif arg_value == None:
                    arg_value = "undefined"
                elif isinstance(arg_value, str):
                    arg_value = f'"{arg_value}"'
                if arg_value == float("inf"):
                    arg_value = "Infinity"
                if arg_value == float("-inf"):
                    arg_value = "-Infinity"
                if isinstance(arg_value, float) and math.isnan(arg_value):
                    arg_value = "NaN"
                # try:
                # logger.info(f"evaluating {node.operator} {arg_value}")
                eval_result = Primitive(pm.eval(f"{node.operator} {arg_value}"))
                result.add(eval_result)
        return result

    def visit_BinaryExpression(  # TODO I really need to clean up the logic here. I'm not convinced it's correct.
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        result = OrderedSet()
        left = self.visit(node.left, file_path)
        right = self.visit(node.right, file_path)
        binary_operators_not_equal = [
            "<<",
            ">>",
            ">>>",
            "+",
            "-",
            "*",
            "/",
            "%",
            "&",
            "|",
            "^",
            "instanceof",
        ]
        binary_operators_equal = [
            "==",
            "!=",
            "===",
            "!==",
            "<",
            ">",
            "<=",
            ">=",
        ]
        if node.operator in binary_operators_not_equal:
            for l in left:
                if (
                    node.right.type == "Identifier"
                    and node.right.name == "RegExp"
                    and node.operator == "instanceof"
                ):
                    result.add(Primitive(isinstance(l, Primitive) and l.is_regex))
                    logger.info("INSTANCE OF REGEXP FOUND")
                    continue
                if isinstance(l, Primitive):
                    l = l.get_value()  # type: ignore
                elif isinstance(l, Address):
                    l = self.env.lookup_and_derive_address(l)
                elif l == baseType.NUMBER:
                    right_is_string = False
                    for r in right:
                        if r == baseType.STRING:
                            right_is_string = True
                        if isinstance(r, Primitive):
                            r = r.get_value()
                            if type(r) == str:
                                right_is_string = True
                    if right_is_string:
                        result.add(baseType.STRING)
                    else:
                        result.add(baseType.NUMBER)
                    continue
                elif l == baseType.TOP:
                    result.add(baseType.TOP)
                    continue
                elif l == baseType.STRING:
                    result.add(baseType.STRING)
                    continue
                if isinstance(l, str):
                    if l == '"':
                        l = f"'{l}'"
                    else:
                        l = f'"{l}"'  # type: ignore
                elif isinstance(l, float) and math.isnan(l):
                    l = "NaN"
                if l == baseType.NULL:
                    l = "null"
                if l == float("inf"):
                    l = "Infinity"
                if l == float("-inf"):
                    l = "-Infinity"
                for r in right:
                    if isinstance(r, Primitive):
                        r = r.get_value()  # type: ignore
                    elif isinstance(r, Address):
                        r = self.env.lookup_and_derive_address(r)
                    elif r == baseType.NUMBER:
                        left_is_string = False
                        if isinstance(l, str):
                            left_is_string = True
                        if left_is_string:
                            result.add(baseType.STRING)
                        else:
                            result.add(baseType.NUMBER)
                        continue
                    elif r == baseType.TOP:
                        result.add(baseType.TOP)
                        continue
                    elif r == baseType.STRING:
                        result.add(baseType.STRING)
                        continue
                    if isinstance(r, str):
                        if r == '"':
                            r = f"'{r}'"
                        else:
                            r = f'"{r}"'  # type: ignore
                    elif isinstance(r, float) and math.isnan(r):
                        r = "NaN"
                    if r == baseType.NULL:
                        r = "null"
                    if r == float("inf"):
                        r = "Infinity"
                    if r == float("-inf"):
                        r = "-Infinity"
                    # logger.info(f"l: {l} r: {r}")
                    # try:
                    # logger.info(
                    #    f"evaluating {l} {node.operator} {r} on line {node.loc.start.line}"
                    # )
                    if node.operator == "instanceof":
                        logger.info(f"INSTANCE OF {l} {r}")
                        if not isinstance(r, dict):
                            result.add(Primitive(False))
                        else:
                            if "__clazz_name__" in r:
                                if l in r["__clazz_name__"]:
                                    result.add(Primitive(True))
                                else:
                                    result.add(Primitive(False))
                            else:
                                result.add(Primitive(False))
                        continue
                    if isinstance(r, str):
                        r = r.replace("\n", "\\n")
                    if isinstance(l, str):
                        l = l.replace("\n", "\\n")
                    if node.operator == "in":
                        eval_result = l in r
                    else:
                        string_to_evaluate = (
                            f"var l={l}; var r = {r}; l {node.operator} r"
                        )
                        string_to_evaluate = string_to_evaluate.replace("True", "true")
                        string_to_evaluate = string_to_evaluate.replace(
                            "False", "false"
                        )
                        try:
                            eval_result = Primitive(pm.eval(string_to_evaluate))
                        except Exception as e:
                            eval_result = baseType.STRING
                    result.add(eval_result)
                    # except Exception:
                    #    result.add(baseType.TOP)
            return result
        elif node.operator in binary_operators_equal:
            for l in left:
                if isinstance(l, Primitive):
                    l = l.get_value()
                    if isinstance(l, bool) and l == True:
                        l = "true"
                    elif isinstance(l, bool) and l == False:
                        l = "false"
                elif isinstance(l, Address):
                    l = self.env.lookup_and_derive_address(l)
                elif l == baseType.NULL:
                    l = "null"
                elif isinstance(l, AbstractType):
                    result.add(baseType.BOOLEAN)
                    continue
                if (
                    isinstance(l, str)
                    and l != "nan"
                    and l != "null"
                    and l != "true"
                    and l != "false"
                ):
                    l = f'"{l}"'  # type: ignore
                elif isinstance(l, float) and math.isnan(l):
                    l = "NaN"
                for r in right:
                    if isinstance(r, Primitive):
                        r = r.get_value()
                        if isinstance(r, bool) and r == False:
                            r = "false"
                        elif isinstance(r, bool) and r == True:
                            r = "true"
                    elif isinstance(r, Address):
                        r = self.env.lookup_and_derive_address(r)
                    elif r == baseType.NULL:  # TODO this is gross
                        #logger.info(f"NULL FOUND TESTING {l} {r}")
                        if l != baseType.NULL and l != "null":
                            eval_result = pm.eval(f"'test' {node.operator} undefined")
                        else:
                            eval_result = pm.eval(
                                f"undefined {node.operator} undefined"
                            )
                        result.add(Primitive(eval_result))
                        continue
                        r = "null"
                    elif isinstance(r, AbstractType):
                        result.add(baseType.BOOLEAN)
                        continue
                    if (
                        isinstance(r, str)
                        and r != "null"
                        and r != "false"
                        and r != "true"
                        and r != "nan"
                    ):
                        r = f'"{r}"'
                    elif isinstance(r, float) and math.isnan(r):
                        r = "NaN"
                    # try:
                    if isinstance(r, str):
                        r = r.replace("\n", "\\n")
                    if isinstance(l, str):
                        l = l.replace("\n", "\\n")
                    # logger.info(f"evaluating {l} {node.operator} {r}")
                    string_to_eval = f"var l={l}; var r = {r}; l {node.operator} r"
                    string_to_eval = string_to_eval.replace("True", "true")
                    string_to_eval = string_to_eval.replace("False", "false")
                    try:
                        eval_result = Primitive(pm.eval(string_to_eval))
                        result.add(eval_result)
                    except Exception:
                        result.add(baseType.BOOLEAN)
            return result
        else:
            return OrderedSet()

    def visit_LogicalExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        left: OrderedSet[Type] = self.visit(node.left, file_path)
        right: OrderedSet[Type] = self.visit(node.right, file_path)
        if node.operator == "&&":
            if only_contains_truthy_values(left):
                return right
            elif not contains_truthy_value(left):
                return left
            else:
                return left | right

        elif node.operator == "||":
            if only_contains_truthy_values(left):
                return left
            elif not contains_truthy_value(left):
                return right
            else:
                return left | right
        else:
            return left | right

    def visit_UpdateExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        if (
            node.argument.type == "Identifier"
            or node.argument.type == "MemberExpression"
        ):
            possible_results = OrderedSet()
            possible_values = self.visit(node.argument, file_path)
            for possible_value in possible_values:
                if isinstance(possible_value, Primitive):
                    possible_value = possible_value.get_value()
                    try:
                        possible_results.add(
                            Primitive(
                                pm.eval(
                                    f"var tmp={possible_value}; {node.operator}tmp;tmp"
                                )
                            )
                        )
                    except Exception:
                        logger.info(
                            f"error: var tmp={possible_value}; {node.operator}tmp;tmp"
                        )
                elif isinstance(possible_value, AbstractType):
                    possible_results.add(possible_value)

            if node.argument.type == "Identifier":
                self.env.add(node.argument.name, possible_results, overwrite=True)
            elif node.argument.type == "MemberExpression":
                if not node.argument.computed:
                    possible_props = [Primitive(node.argument.property.name)]
                else:
                    possible_props = self.visit(node.argument.property, file_path)

                if node.argument.object.type == "ThisExpression":
                    self.env.update_this(
                        possible_props,
                        possible_results,
                        overwrite=True,
                    )
                else:
                    possible_object_addresses = self.visit(
                        node.argument.object, file_path
                    )
                    for address in possible_object_addresses:
                        if isinstance(address, Address):
                            self.env.update(
                                address,
                                possible_props,
                                possible_results,
                                overwrite=True,
                                objName=node.argument.object.name,
                            )

            if node.prefix:
                return possible_results
            else:
                return possible_values

    def visit_CallExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        # Need to handle require calls separately
        if node.callee.type == "Identifier" and node.callee.name == "require":
            source_path = node.arguments[0].value
            source_path = self.load_path_for_require(
                source_path, file_path
            )  # os.path.join(os.path.dirname(file_path), source_path)
            logger.info(f"SOURCE PATH: {source_path}")
            self.absint_ai.run(
                source_path, reset_heap_ids=False, recurse=False
            )  # Analyze the module that's imported by require
            exports = self.env.get_exports_for_module(
                source_path
            )  # Get the exports from the module
            self.env.change_module(file_path)  # Change back to the original module
            logger.info(f"EXPORTS: {exports}")
            return exports.get_all_values()
        else:
            return_value = self.function_executor._execute_function(node, file_path)
            return return_value

    def visit_NewExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        if node.callee.type == "Identifier":
            if node.callee.name == "RegExp":
                arg_values = self.visit(node.arguments[0], file_path)
                result = OrderedSet()
                for arg_value in arg_values:
                    if isinstance(arg_value, Primitive) and isinstance(
                        arg_value.get_value(), str
                    ):
                        arg_value = Primitive(
                            "/" + arg_value.get_value() + "/", is_regex=True
                        )
                        result.add(arg_value)
                return result

            class_name = node.callee.name
            funcs: list[Type] = self.env.lookup(class_name).get_all_values()
            return self.function_executor._execute_constructors(
                funcs, node.arguments, file_path, node, class_name=class_name
            )
        elif node.callee.type == "MemberExpression":
            funcs = []
            possible_object_addresses = self.visit(node.callee.object, file_path)
            if not node.callee.computed:
                possible_props = [Primitive(node.callee.property.name)]
            else:
                possible_props = self.visit(node.callee.property, file_path)
            logger.info(
                f"possible_props: {possible_props}, possible_object_addresses: {possible_object_addresses}"
            )
            for possible_object_address in possible_object_addresses:
                if not isinstance(possible_object_address, Address):
                    continue
                obj = self.env.lookup_address(possible_object_address)
                for prop in possible_props:
                    possible_prop_values = self.env.lookup_field(
                        possible_object_address, prop
                    )
                    if not possible_prop_values:
                        self.absint_ai.buggy_lines.append(
                            f"line {self.absint_ai.get_loc(node)}: {prop.get_value()} not found in object {node.callee.object.name} for file {file_path} NewExpression"
                        )
                        self.absint_ai.buggy_line_numbers.append(
                            self.absint_ai.get_loc_id(node)
                        )
                    for possible_prop_value in possible_prop_values.get_all_values():
                        if isinstance(possible_prop_value, Address) and (
                            self.env.get_object_type(possible_prop_value) == "function"
                            or self.env.get_object_type(possible_prop_value) == "class"
                        ):
                            funcs.append(possible_prop_value)
                    if not len(possible_prop_values.get_all_values()):
                        self.absint_ai.buggy_lines.append(
                            f"line {self.absint_ai.get_loc(node)}: {prop.get_value()} not found in object {node.callee.object.name} for file {file_path} NewExpression"
                        )
                        self.absint_ai.buggy_line_numbers.append(
                            self.absint_ai.get_loc_id(node)
                        )
                        possible_prop_values = self._get_abstract_property(
                            obj, prop, expr=node, file_path=file_path
                        )
                        for possible_prop_value in possible_prop_values:
                            if (
                                isinstance(possible_prop_value, Address)
                                and (
                                    self.env.get_object_type(possible_prop_value)
                                    == "function"
                                    or self.env.get_object_type(possible_prop_value)
                                    == "class"
                                )
                                and possible_prop_value not in funcs
                            ):
                                funcs.append(possible_prop_value)
            return self.function_executor._execute_constructors(
                funcs, node.arguments, file_path, node
            )

    def visit_ClassExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        return self.statement_visitor.add_class(node, file_path)

    def visit_FunctionExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        """
        Just want to store the function name and the parameters in the environment.
        """
        scope_id = get_schema_from_expr(node, file_path, "function")
        function = self.env.initialize_new_function(
            "",
            [param.name for param in node.params],
            node.body,
            scope_id,
            file_path,
            full_function_node=node,
        )
        return OrderedSet([function])

    def visit_ArrowFunctionExpression(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        return self.visit_FunctionExpression(node, file_path)

    def visit_TemplateLiteral(
        self, node: esprima.nodes.Node, file_path: str
    ) -> OrderedSet[Type]:
        return OrderedSet([Primitive(node.quasis[0].value.raw)])  # TODO this is a hack

    # region Helper functions

    def _get_possible_values_for_object(
        self,
        objExpr: esprima.nodes.ExpressionStatement,
        file_path: str,
    ) -> OrderedSet[Type]:
        allocation_site = allocation_site_from_expr(expr=objExpr, file_path=file_path)
        if objExpr.type == "ObjectExpression":
            obj = {}
            for prop in objExpr.properties:
                if prop.type == "Property":
                    if prop.key.type == "Identifier":
                        obj[prop.key.name] = self.visit(prop.value, file_path)
                    if prop.key.type == "Literal":
                        obj[prop.key.value] = self.visit(prop.value, file_path)
            address = self.env.add_object_to_heap(obj, allocation_site=allocation_site)
            return OrderedSet([address])
        elif objExpr.type == "ArrayExpression":
            obj = {}
            for i in range(len(objExpr.elements)):
                prop = objExpr.elements[i]
                obj[i] = self.visit(prop, file_path)
            return OrderedSet(
                [self.env.add_object_to_heap(obj, allocation_site=allocation_site)]
            )

        raise Exception("Object expression not found")

    def _get_abstract_property(
        self,
        obj: object,
        prop: Primitive | AbstractType,
        expr: esprima.nodes.Node,
        file_path: str,
        seen: list = None,
    ) -> OrderedSet[Type]:
        if seen is None:
            seen = []
        original_prop = copy.deepcopy(prop)
        loc_id = self.absint_ai.get_loc_id(expr)
        buggy_line_info = f"line {self.absint_ai.get_loc(expr)}: {prop.get_value()} not found in object for file {file_path} MemberExpression DUE TO ABSTRACTION {obj.keys()} {obj['__meta__']['type']}"
        if isinstance(prop, Primitive):
            if is_property_builtin(prop.get_value()):
                is_builtin = True
            else:
                is_builtin = False
        elif isinstance(prop, str):
            if is_property_builtin(prop):
                is_builtin = True
            else:
                is_builtin = False
        else:
            is_builtin = False
        result = OrderedSet()
        if isinstance(prop, Primitive):
            prop = prop.get_value()
            if isinstance(prop, int) and baseType.NUMBER in obj:
                if not is_builtin:
                    self.absint_ai.buggy_line_numbers.append(loc_id)
                    self.absint_ai.buggy_lines.append(buggy_line_info)
                result.update(obj[baseType.NUMBER])
            elif isinstance(prop, str) and baseType.STRING in obj:
                if not is_builtin:
                    self.absint_ai.buggy_line_numbers.append(loc_id)
                    self.absint_ai.buggy_lines.append(buggy_line_info)
                result.update(obj[baseType.STRING])
            elif baseType.TOP in obj:
                result.update(obj[baseType.TOP])
                if not is_builtin:
                    self.absint_ai.buggy_line_numbers.append(loc_id)
                    self.absint_ai.buggy_lines.append(buggy_line_info)
        elif isinstance(prop, AbstractType):
            if prop == baseType.NUMBER:
                if baseType.NUMBER in obj:
                    result.update(obj[baseType.NUMBER])
                for key in obj:
                    if key == "__meta__":
                        continue
                    if (
                        isinstance(key, Primitive)
                        and isinstance(key.get_value(), int)
                        or isinstance(key, int)
                    ):
                        result.update(obj[key])
            if prop == baseType.STRING:
                if baseType.STRING in obj:
                    result.update(obj[baseType.STRING])
                for key in obj:
                    if key == "__meta__":
                        continue
                    if (
                        isinstance(key, Primitive)
                        and isinstance(key.get_value(), str)
                        or isinstance(key, str)
                    ):
                        result.update(obj[key])
            if prop == baseType.TOP:
                if baseType.TOP in obj:
                    result.update(obj[baseType.TOP])
                for key in obj:
                    if key == "__meta__":
                        continue
                    result.update(obj[key])
            # logger.info(f"Abstract property {prop} from obj {obj}. Result {result}")
            if not is_builtin:
                self.absint_ai.buggy_line_numbers.append(loc_id)
                self.absint_ai.buggy_lines.append(buggy_line_info)
        if "__proto__" in obj:
            for proto in obj["__proto__"]:
                if isinstance(proto, Address) and proto not in seen:
                    seen.append(proto)
                    result.update(
                        self._get_abstract_property(
                            self.env.lookup_address(proto),
                            original_prop,
                            expr,
                            file_path,
                            seen=seen,
                        )
                    )
        else:
            result.add(baseType.NULL)
        return result

    # mimicking the require algorithm in node.js
    def load_path_for_require(self, require_arg: str, file_path: str) -> str:
        logger.info(f"searching for {require_arg} from {file_path}")

        def find_package_scope(dir_name):
            cur_path = dir_name
            while True:
                if os.path.exists(os.path.join(cur_path, "package.json")):
                    return cur_path
                if cur_path == "/":
                    return None
                cur_path = os.path.dirname(cur_path)

        def load_as_file(x: str) -> str:
            if os.path.exists(x):
                return x
            elif os.path.exists(x + ".js"):
                return x + ".js"
            elif os.path.exists(x + ".cjs"):
                return x + ".cjs"
            else:
                return None

        def load_as_directory(x: str) -> str:
            if os.path.exists(os.path.join(x, "package.json")):
                package_json = json.load(open(os.path.join(x, "package.json")))
                if "main" in package_json:
                    return load_as_file(os.path.join(x, package_json["main"]))
                else:
                    return load_as_file(os.path.join(x, "index"))
            else:
                return load_as_file(os.path.join(x, "index"))

        # resolve self referencing packages
        def load_package_self(x: str, dir_name: str) -> str:
            package_scope = find_package_scope(dir_name)
            if not package_scope:
                return None
            package_json = json.load(open(os.path.join(package_scope, "package.json")))
            if "name" not in package_json:
                return None
            package_name = package_json["name"]
            if not x.startswith(package_name):
                return None
            if "exports" not in package_json:
                return None
            exports = package_json["exports"]
            resolved_path = package_exports_resolve(
                package_scope,
                "." + x[len(package_name) :],
                exports,
                ["node", "require"],
            )
            return resolved_path

        def package_exports_resolve(packageURL, subpath, exports, conditions):
            if subpath in exports:
                return os.path.join(packageURL, exports[subpath])

        path = os.path.dirname(file_path)
        core_modules = [
            "assert",
            "buffer",
            "child_process",
            "cluster",
            "crypto",
            "dgram",
            "dns",
            "domain",
            "events",
            "fs",
            "http",
            "https",
            "net",
            "os",
            "path",
            "punycode",
            "querystring",
            "readline",
            "repl",
            "stream",
            "string_decoder",
            "timers",
            "tls",
            "tty",
            "url",
            "util",
            "v8",
            "vm",
            "zlib",
        ]
        cur_path = os.path.dirname(os.path.abspath(__file__))
        if (
            require_arg in core_modules
            or os.path.join(cur_path, "libraries", "lib") in file_path
        ):
            if not require_arg.endswith(".js") and not require_arg.endswith(".cjs"):
                require_arg = require_arg + ".js"
            return os.path.join(cur_path, "libraries", "lib", f"{require_arg}")
        if require_arg.startswith("/"):
            path = "/"
        elif require_arg.startswith("./") or require_arg.startswith("../"):
            load_as_file_result = load_as_file(os.path.join(path, require_arg))
            if load_as_file_result:
                return load_as_file_result
            load_as_directory_result = load_as_directory(
                os.path.join(path, require_arg)
            )
            if load_as_directory_result:
                return load_as_directory_result
            raise Exception(f"Cannot find module {require_arg}")
        elif require_arg.startswith("#"):
            raise Exception("Not implemented")
        else:
            load_package_self_result = load_package_self(require_arg, path)
            if load_package_self_result:
                return load_package_self_result

        if not require_arg.endswith(".js") and not require_arg.endswith(".cjs"):
            require_arg = require_arg + ".js"
        source_path = os.path.join(os.path.dirname(file_path), require_arg)
        return source_path

    # endregion
