import time
from absint_ai.utils.Util import *
from typing import TYPE_CHECKING
from absint_ai.Environment.memory.RecordResult import RecordResult
from ordered_set import OrderedSet
from absint_ai.Environment.types.Type import *
import pythonmonkey as pm
import absint_ai.Interpreter.visitors.helpers.builtins.strings as string_builtins
import esprima
import js2py
import copy
import itertools
from dotmap import DotMap

# Have to do this because the initial implementation just returns __class__ for any unregistered keys. Weird.
DotMap.__getitem__ = DotMap.get  # type: ignore

if TYPE_CHECKING:
    from absint_ai.Environment.Environment import Environment
    from absint_ai.Interpreter.AbsIntAI import AbsIntAI
    from absint_ai.Interpreter.visitors.expr_visitor import ExprVisitor
    from absint_ai.Interpreter.visitors.statement_visitor import StatementVisitor


# Helper class for executing functions
class FunctionExecutor:
    def __init__(
        self,
        env: "Environment",
        absint_ai: "AbsIntAI",
        expr_visitor: "ExprVisitor",
        statement_visitor: "StatementVisitor",
    ):
        self.env = env
        self.statement_visitor = None
        self.absint_ai = absint_ai
        self.expr_visitor = expr_visitor
        self.statement_visitor = statement_visitor

    def _execute_function(
        self, funcExpr: esprima.nodes.CallExpression, file_path: str
    ) -> OrderedSet[Type]:
        # We should be able to just execute it with the normal environment. The only thing is that we need to copy the parameters
        if time.time() - self.absint_ai.start_time > self.absint_ai.timeout:
            raise Exception("Took too long")
        call_loc = self.absint_ai.get_loc(funcExpr)
        if (
            funcExpr.callee.type == "Identifier"
            or funcExpr.callee.type == "ArrowFunctionExpression"
            or funcExpr.callee.type == "FunctionExpression"
        ):
            return self._call_non_member_function(funcExpr, file_path)
        elif funcExpr.callee.type == "MemberExpression":
            return self._call_member_function(funcExpr, file_path)
        logger.info(
            f"Function call not found {funcExpr.callee.type} on line {self.absint_ai.get_loc(funcExpr)}. Identifiers: {get_identifiers_from_expr(funcExpr.callee)}"
        )
        return OrderedSet([baseType.TOP])

    def _call_non_member_function(
        self, funcExpr: esprima.nodes.CallExpression, file_path: str
    ) -> OrderedSet[Type]:
        possibleValues: set[Type] = OrderedSet()
        is_recursive = False
        if time.time() - self.absint_ai.start_time > self.absint_ai.timeout:
            raise Exception("Took too long")
        call_loc = self.absint_ai.get_loc(funcExpr)
        if funcExpr.callee.type == "Identifier":
            if self.is_builtin(funcExpr.callee.name):
                arguments_as_types = [
                    self.expr_visitor.visit(_, file_path) for _ in funcExpr.arguments
                ]
                possibleValues.update(
                    self.execute_builtin(
                        funcExpr.callee.name,
                        arguments_as_types,
                        self.env,
                        expr=funcExpr,
                        file_path=file_path,
                    )
                )
                return possibleValues
            funcs = [
                _
                for _ in self.env.lookup(funcExpr.callee.name, loc=call_loc)
                if isinstance(_, Address) and self.env.get_object_type(_) == "function"
            ]
        else:
            scope_id = get_schema_from_expr(funcExpr.callee, file_path, "function")
            scope_schema = self.env.get_schema_for_scope_id(scope_id)
            params = (
                [param.name for param in funcExpr.callee.params]
                if funcExpr.callee.params
                else []
            )
            function = self.env.initialize_new_function(
                "", params, funcExpr.callee.body, scope_id, file_path
            )
            funcs = [function]
        # if len(funcs) > 1:
        #    logger.info(f"Multiple functions ({len(funcs)}) found: {funcs}")
        for func in funcs:
            func_info = self.env.get_meta(func)
            params = func_info["params"]
            params_to_add = {}
            # logger.info(
            #    f"Function {func} with params {params} called at file {file_path} line {self.get_loc(funcExpr)}"
            # )
            for i in range(len(params)):
                param = params[i]
                if i > len(funcExpr.arguments) - 1:
                    params_to_add[param] = [baseType.NULL]
                else:
                    params_to_add[param] = self.expr_visitor.visit(
                        funcExpr.arguments[i], file_path
                    )
            if func_info["rest"]:
                rest_args = funcExpr.arguments[len(params) :]
                rest_args_evaluated = array_to_object(
                    [self.expr_visitor.visit(arg, file_path) for arg in rest_args]
                )
                rest_args_evaluated_address = self.env.add_object_to_heap(
                    rest_args_evaluated,
                    allocation_site=None,
                )
                params_to_add[func_info["rest"]] = [rest_args_evaluated_address]
            # We can memoize function calls here. If we have already run the function with the same parameters, we can just return the result.
            # Because function ID's are names, functionExpressions and arrowFunctionExpressions shouldn't be memoized
            if funcExpr.callee.type == "Identifier":
                memoized_function_return_values = self.env.get_memoized_function(
                    func, params_to_add
                )
                if memoized_function_return_values != None:
                    # logger.info(
                    #    f"memoized function call {func} on line {self.get_loc(funcExpr)} found"
                    # )
                    possibleValues.update(memoized_function_return_values)  # type: ignore
                    continue

            if self.env.is_recursive_call(func, params_to_add):
                self.env.set_has_recursive_call(True)
                is_recursive = True
                return_values = [baseType.RECURSIVE_PLACEHOLDER]
                possibleValues.update(return_values)
                continue
            if func_info["file_path"]:
                function_file_path = func_info["file_path"]
            else:
                function_file_path = file_path
            arguments = {}
            for i in range(len(funcExpr.arguments)):
                arguments[i] = self.expr_visitor.visit(funcExpr.arguments[i], file_path)
            if func_info.get("__parent__"):
                self.env.initialize_from_schema(
                    func_info["schema"],
                    function_file_path,
                    allocation_site=func_info["schema_id"],
                    parent_address=func_info.get("__parent__"),
                    abstract_parents=True,
                )
            else:
                logger.info(f"Function {func} has no parent heap frame")
                self.env.initialize_from_schema(
                    func_info["schema"],
                    function_file_path,
                    allocation_site=func_info["schema_id"],
                    parent_address=self.env.get_current_heap_frame(),
                    abstract_parents=True,
                )

            params_debug = {}
            for param, values in params_to_add.items():
                for value in values:
                    if isinstance(value, Address):
                        logger.info(
                            f"Moving {value} to abstract heap for {param} on line {self.absint_ai.get_loc(funcExpr)}"
                        )
                        self.env.move_object_to_abstract_heap(value)

                self.env.add(param, values)  # Need to copy arguments from the parent
                params_debug[param] = self.env.lookup_and_derive(param)

            arguments_address = self.env.add_object_to_heap(
                arguments,
                allocation_site=allocation_site_from_expr(funcExpr, file_path),
            )
            self.env.add("arguments", [arguments_address])

            # We need to add the function to the memoization table before we run it, because the function might call itself recursively.
            self.env.add_recursive_call(
                func, params_to_add
            )  # Need to do this for member functions
            if func_info["body"].type == "BlockStatement":
                if not self._guarantees_return(func_info["body"]):
                    self.env.add_return_value(baseType.NULL)
                for statement in func_info["body"].body:
                    self.statement_visitor.visit(
                        statement,
                        function_file_path,
                    )
                result = self.env.get_return_values()
            else:
                result = self.expr_visitor.visit(func.body, function_file_path)
            if isinstance(result, list) or isinstance(result, OrderedSet):
                possibleValues.update(result)
            elif isinstance(result, Type):
                possibleValues.add(result)
            self.env.memoize_and_return_from_function(
                func, params_to_add, function_file_path, file_path
            )
        if not funcs:
            logger.debug(f"Function {funcExpr.callee.name} not found")
            return OrderedSet([baseType.TOP])  # type: ignore
        return possibleValues

    def _call_member_function(
        self, funcExpr: esprima.nodes.CallExpression, file_path: str
    ):
        possibleValues: set[Type] = OrderedSet()
        if time.time() - self.absint_ai.start_time > self.absint_ai.timeout:
            raise Exception("Took too long")
        func_fields: list[
            tuple[
                Address, Address
            ]  # First address is the object, second address is the actual function that's being called
        ] = (
            []
        )  # Tuple of address and function. We need the address in case the function performs a `this` call.
        if is_builtin_object(funcExpr.callee.object):
            # If the object is a builtin, we can just call the builtin directly
            if not funcExpr.callee.computed:
                possibleProps = [Primitive(funcExpr.callee.property.name)]
            else:
                possibleProps = self.expr_visitor.visit(
                    funcExpr.callee.property, file_path
                )
            for prop in possibleProps:
                if isinstance(prop, Primitive) and isinstance(prop.get_value(), str):
                    prop = prop.get_value()
                    possible_arguments = [
                        self.expr_visitor.visit(_, file_path)
                        for _ in funcExpr.arguments
                    ]
                    possibleValues.update(
                        self.execute_static_builtin(
                            funcExpr.callee.object.name,
                            prop,
                            possible_arguments,
                        )
                    )
            return possibleValues
        addresses = [
            address
            for address in self.expr_visitor.visit(funcExpr.callee.object, file_path)
        ]
        found_builtin = False
        if not funcExpr.callee.computed:
            possibleProps = [Primitive(funcExpr.callee.property.name)]
        else:
            possibleProps = self.expr_visitor.visit(funcExpr.callee.property, file_path)
        for address in addresses:
            if address in self.env.builtin_addresses:
                continue
            if isinstance(address, Primitive) or isinstance(address, AbstractType):
                if isinstance(address, Primitive) or address == baseType.STRING:
                    address = address.get_value()
                    if isinstance(address, str):
                        possible_arguments = [
                            self.expr_visitor.visit(_, file_path)
                            for _ in funcExpr.arguments
                        ]
                        found_builtin = True
                        for prop in possibleProps:
                            if isinstance(prop, Primitive) and self.is_string_builtin(
                                address, prop
                            ):
                                possibleValues.update(
                                    string_builtins.execute_string_builtin(
                                        address,
                                        prop.get_value(),
                                        possible_arguments,
                                        self.env,
                                    )
                                )
                continue
            else:
                # Handle strings earlier because that way we don't have to keep looking up obj here
                obj = self.env.lookup_address(address)  # type: ignore
                for prop in possibleProps:
                    possible_prop_values = self.env.lookup_field(address, prop)
                    if self.is_array_builtin(obj, prop):
                        found_builtin = True
                        possibleValues.update(
                            self.execute_array_builtin(
                                address, prop, funcExpr.arguments, file_path=file_path
                            )
                        )
                    elif not possible_prop_values.get_all_values():
                        self.absint_ai.buggy_lines.append(
                            f"line {self.absint_ai.get_loc(funcExpr)}: {prop} not found in object {address} for file {file_path} function call"
                        )
                        logger.info(f"{prop} not found in object {obj} function call")
                        self.absint_ai.buggy_line_numbers.append(
                            self.absint_ai.get_loc_id(funcExpr)
                        )

                    for value in possible_prop_values:  # type: ignore
                        if self.env.is_function(value) or self.env.is_builtin(value):
                            func_fields.append((address, value))  # type: ignore
                    if not possible_prop_values.get_all_values():
                        possible_prop_values = self.expr_visitor._get_abstract_property(
                            obj, prop, expr=funcExpr, file_path=file_path
                        )
                        for value in possible_prop_values:
                            if (
                                self.env.is_function(value)
                                and (address, value) not in func_fields
                            ):
                                func_fields.append((address, value))
        # if len(func_fields) > 1:
        #    logger.info(
        #        f"Multiple functions ({len(func_fields)}) on line {funcExpr.loc}: {func_fields}"
        #    )
        #    logger.info(f"Fields for object {possibleProps}")
        for this_address, func in func_fields:
            func_info = self.env.get_meta(func)
            if self.env.is_builtin(func):
                args = [
                    self.expr_visitor.visit(arg, file_path)
                    for arg in funcExpr.arguments
                ]
                result = self.env.execute_builtin(self.env.lookup_address(func), args)
                possibleValues.update(result)
                continue

            params = func_info["params"]
            params_to_add = {}
            for i in range(len(params)):
                param = params[i]
                if i > len(funcExpr.arguments) - 1:
                    params_to_add[param] = [baseType.NULL]
                else:
                    params_to_add[param] = self.expr_visitor.visit(
                        funcExpr.arguments[i], file_path
                    )

            if func_info.get("rest", None):
                rest_args = funcExpr.arguments[len(params) :]
                rest_args_evaluated = array_to_object(
                    [self.expr_visitor.visit(arg, file_path) for arg in rest_args]
                )
                rest_args_evaluated_address = self.env.add_object_to_heap(
                    rest_args_evaluated,
                    allocation_site=None,
                )
                params_to_add[func_info["rest"]] = [rest_args_evaluated_address]

            memoized_function_return_values = self.env.get_memoized_function(
                func, params_to_add
            )
            if memoized_function_return_values != None:
                logger.info(
                    f"memoized function call {func} on line {self.absint_ai.get_loc(funcExpr)} found"
                )
                possibleValues.update(memoized_function_return_values)  # type: ignore
                continue
            if self.env.is_recursive_call(func, params_to_add):
                self.env.set_has_recursive_call(True)
                is_recursive = True
                return_values = [baseType.RECURSIVE_PLACEHOLDER]
                possibleValues.update(return_values)
                continue
            if func_info.get("file_path"):
                function_file_path = func_info.get("file_path")
            else:
                function_file_path = file_path
            arguments = {}
            for i in range(len(funcExpr.arguments)):
                arguments[i] = self.expr_visitor.visit(funcExpr.arguments[i], file_path)
            if this_address.get_parent_heap_frame():
                self.env.initialize_from_schema(
                    func_info["schema"],
                    function_file_path,
                    allocation_site=func_info["schema_id"],
                    parent_address=this_address.get_parent_heap_frame(),  # type: ignore
                    this_address=this_address,
                    abstract_parents=True,
                )
            elif func_info.get("__parent__"):
                self.env.initialize_from_schema(
                    func_info["schema"],
                    function_file_path,
                    allocation_site=func_info["schema_id"],
                    parent_address=func_info.get("__parent__"),  # type: ignore
                    this_address=this_address,
                    abstract_parents=True,
                )
            else:
                logger.info(f"Function {func} has no parent heap frame")
                self.env.initialize_from_schema(
                    func_info["schema"],
                    function_file_path,
                    allocation_site=func_info["schema_id"],
                    parent_address=self.env.get_current_heap_frame(),
                    this_address=this_address,
                    abstract_parents=True,
                )

            for param, values in params_to_add.items():
                for value in values:
                    if isinstance(value, Address):
                        self.env.move_object_to_abstract_heap(value)
                self.env.add(param, values)  # Need to copy arguments from the parent

            arguments_address = self.env.add_object_to_heap(
                arguments,
                allocation_site=allocation_site_from_expr(funcExpr, file_path),
            )
            self.env.add("arguments", [arguments_address])
            self.env.add_recursive_call(
                func, params_to_add
            )  # Need to do this for member functions

            result = []
            if func_info["body"].type == "BlockStatement":
                if not self._guarantees_return(func_info["body"]):
                    self.env.add_return_value(baseType.NULL)
                for statement in func_info["body"].body:
                    self.statement_visitor.visit(statement, file_path)
                    result = self.env.get_return_values()
            else:
                result = self.expr_visitor.visit(func_info["body"], file_path)
                logger.info(
                    f"Function body not found for {func}, body is {func_info['body']}: result {result}"
                )
            if isinstance(result, list) or isinstance(result, OrderedSet):
                possibleValues.update(result)
            elif isinstance(result, Type):
                possibleValues.add(result)
            self.env.memoize_and_return_from_function(
                func, params_to_add, function_file_path, file_path
            )
        if not func_fields and not found_builtin:
            return OrderedSet([baseType.TOP])
        return possibleValues

    def _execute_constructors(
        self,
        funcs: list[Type],
        arguments: list,
        file_path: str,
        expr: esprima.nodes.Node,
        class_name: str = "",
    ) -> OrderedSet[Type]:
        # For now we will just assume that only field initialization happens inside of the constructor.
        possible_values = OrderedSet()
        allocation_site = allocation_site_from_expr(expr, file_path)
        if len(funcs) > 1:
            logger.info(f"Multiple functions ({len(funcs)}) found: {funcs}")
        for clazz in funcs:
            obj: dict[str, object] = (
                {}
            )  # This will be the object that will be initialized from the constructor and added to the heap
            if isinstance(clazz, Address):
                clazz_obj = self.env.lookup_address(clazz)
                if self.env.get_object_type(clazz) == "class":
                    obj["__clazz_name__"] = Primitive(class_name)
                    if "prototype" not in clazz_obj:
                        continue
                    obj["__proto__"] = clazz_obj["prototype"]
                    for field_name, field_values in clazz_obj.items():
                        if (
                            field_name != "__proto__"
                            and field_name != "__meta__"
                            and field_name != "constructor"
                            and field_name != "prototype"
                        ):
                            obj[field_name] = field_values
                    constructor_addresses = clazz_obj["constructor"]
                elif self.env.get_object_type(clazz) == "function":
                    constructor_addresses = [clazz]
                    obj["__proto__"] = clazz_obj["prototype"]
                else:
                    continue
            else:
                continue
            constructors = [
                _
                for _ in constructor_addresses
                if self.env.get_object_type(_) == "function"
            ]
            for constructor in constructors:
                constructor_info = self.env.get_meta(constructor)
                params = constructor_info["params"]
                params_to_add = {}
                addr = self.env.add_object_to_heap(obj, allocation_site=allocation_site)
                for i in range(len(params)):
                    param = params[i]
                    if i > len(arguments) - 1:
                        params_to_add[param] = [baseType.NULL]
                    else:
                        params_to_add[param] = self.expr_visitor.visit(
                            arguments[i], file_path
                        )
                if constructor.get_parent_heap_frame():
                    if isinstance(constructor.get_parent_heap_frame(), Address):
                        self.env.initialize_from_schema(
                            constructor_info["schema"],
                            file_path,
                            allocation_site=constructor_info["schema_id"],
                            this_address=addr,
                            parent_address=constructor.get_parent_heap_frame(),  # type: ignore
                        )
                    else:
                        raise Exception(
                            f"Parent heap frame is not an address, got {constructor.get_parent_heap_frame()}"
                        )
                else:
                    self.env.initialize_from_schema(
                        constructor_info["schema"],
                        file_path,
                        allocation_site=constructor_info["schema_id"],
                        this_address=addr,
                        parent_address=self.env.get_current_heap_frame(),
                    )
                for param, values in params_to_add.items():
                    self.env.add(
                        param, values
                    )  # Need to copy arguments from the parent
                for statement in constructor_info["body"].body:
                    self.absint_ai.executed_lines += 1
                    if statement.type == "ExpressionStatement":
                        if (
                            statement.expression.type == "CallExpression"
                            and statement.expression.callee.type == "Super"
                        ):
                            proto_addresses = clazz_obj["prototype"]
                            parent_objects = self._execute_constructors(
                                proto_addresses,
                                statement.expression.arguments,
                                file_path,
                                expr,
                                class_name=class_name,
                            )
                            self.env.update(
                                addr,
                                [Primitive("__proto__")],
                                parent_objects,
                                overwrite=True,
                            )
                            # obj["__proto__"] = parent_objects
                        else:
                            self.statement_visitor.visit(statement, file_path)
                    else:
                        self.statement_visitor.visit(statement, file_path)
                return_values = self.env.get_return_values()
                self.env.return_from_function(None, file_path)
                # logger.info(
                #    f"returning {self.env.lookup_and_derive_address(addr)} from constructor"
                # )
                # Sometimes constructors return something, which is kind of wacky
                if (
                    isinstance(return_values, list)
                    or isinstance(return_values, OrderedSet)
                ) and len(return_values) > 0:
                    possible_values.update(return_values)
                else:
                    possible_values.add(addr)

        if len(possible_values) == 0:
            return OrderedSet([baseType.TOP])

        return possible_values  # type: ignore

    # TODO eventually we should move all these builtins to their own classes
    def is_builtin(
        self, function_called: Type | str, obj: object | None = None
    ) -> bool:
        builtins = [
            "eval",
            "parseInt",
            "parseFloat",
            "isNaN",
            "isFinite",
            "decodeURI",
            "decodeURIComponent",
            "encodeURI",
            "encodeURIComponent",
            "Array",
            "String",
            "GetInt",
        ]
        if obj is None:
            if isinstance(function_called, Primitive):
                return function_called.get_value() in builtins
            elif isinstance(function_called, str):
                return function_called in builtins
        else:
            if self.is_array_builtin(obj, function_called):
                return True
            if self.is_string_builtin(obj, function_called):
                return True
        return False

    def execute_builtin(
        self,
        function_called: Type | str,
        arguments_as_types: list[OrderedSet[Type]],
        env: "Environment",
        expr: esprima.nodes.Node,
        file_path: str,
        obj: object | None = None,
    ) -> list[Type]:
        arguments_as_values = [
            [type_to_value(arg, env, ignore_proto=True) for arg in argument]
            for argument in arguments_as_types
        ]
        results = OrderedSet()
        abstract_values = ["NUMBER", "STRING", "BOOLEAN"]
        all_possible_values = list(itertools.product(*arguments_as_values))
        if obj is None:

            if isinstance(function_called, Primitive):
                function_called = function_called.get_value()
            if function_called == "eval":
                pass
            elif function_called == "parseInt":
                for values in all_possible_values:
                    if any(not isinstance(value, (int, str)) for value in values):
                        results.add(baseType.NUMBER)
                        continue
                    values_comma_separated = ", ".join(
                        [
                            f"'{str(value)}'" if isinstance(value, str) else str(value)
                            for value in values
                        ]
                    )
                    
                    string_to_execute = f"parseInt({values_comma_separated})"
                    logger.info(f"Executing {string_to_execute}")
                    result_as_value = pm.eval(string_to_execute)
                    result_as_type = value_to_type(result_as_value, env)
                    results.add(result_as_type)
            elif function_called == "parseFloat":
                pass
            elif function_called == "isNaN":
                pass
            elif function_called == "isFinite":
                pass
            elif function_called == "decodeURI":
                pass
            elif function_called == "decodeURIComponent":
                pass
            elif function_called == "encodeURI":
                pass
            elif function_called == "encodeURIComponent":
                pass
            elif function_called == "Array":
                allocation_site = allocation_site_from_expr(expr, file_path)
                for values in all_possible_values:
                    values_comma_separated = ", ".join(
                        [
                            f"'{str(value)}'" if isinstance(value, str) else str(value)
                            for value in values
                        ]
                    )
                    string_to_execute = f"Array({values_comma_separated})"
                    logger.info(f"Executing {string_to_execute}")
                    result_as_value = pm.eval(string_to_execute)
                    result_as_type = value_to_type(
                        result_as_value, env, allocation_site=allocation_site
                    )
                    logger.info("HERE")

                    results.add(result_as_type)
            elif function_called == "String":
                results.add(baseType.STRING)
            elif function_called == "GetInt":
                results.add(baseType.NUMBER)
        return results

    def is_array_builtin(self, obj: object, function_called: Type | str) -> bool:
        array_builtins = [
            "at",
            "concat",
            "copyWithin",
            "entries",
            "every",
            "fill",
            "filter",
            "find",
            "findIndex",
            "findLast",
            "findLastIndex",
            "flat",
            "flatMap",
            "forEach",
            "includes",
            "indexOf",
            "join",
            "keys",
            "lastIndexOf",
            "map",
            "pop",
            "push",
            "reduce",
            "reduceRight",
            "reverse",
            "shift",
            "slice",
            "some",
            "sort",
            "splice",
            "toLocaleString",
            "toSorted",
            "toSpliced",
            "toString",
            "unshift",
            "values",
        ]
        if all(
            [
                isinstance(_, int) or _ == baseType.NUMBER
                for _ in obj["__meta__"]["enumerable_values"]
            ]
        ) or obj["__meta__"]["enumerable_values"] == [baseType.NUMBER]:
            if isinstance(function_called, Primitive):
                return function_called.get_value() in array_builtins
            elif isinstance(function_called, str):
                return function_called in array_builtins
        return False

    def is_string_builtin(self, obj: Type, function_called: Type | str) -> bool:
        string_builtins = [
            "at",
            "charAt",
            "charCodeAt",
            "codePointAt",
            "concat",
            "endsWith",
            "includes",
            "indexOf",
            "lastIndexOf",
            "localeCompare",
            "match",
            "matchAll",
            "normalize",
            "padEnd",
            "padStart",
            "repeat",
            "replace",
            "search",
            "slice",
            "split",
            "startsWith",
            "substr",
            "toLowerCase",
            "substring",
        ]
        if (
            (isinstance(obj, Primitive) and isinstance(obj.get_value(), str))
            or (isinstance(obj, str))
            or obj == baseType.STRING
        ):
            if isinstance(function_called, Primitive):
                return function_called.get_value() in string_builtins
            elif isinstance(function_called, str):
                return function_called in string_builtins
        return False

    def execute_static_builtin(
        self, identifier: str, method: str, args_as_types: list[OrderedSet[Type]]
    ) -> OrderedSet[Type]:
        """
        Executes a static builtin with the given arguments.
        """
        results = OrderedSet()
        arguments_as_values = [
            [type_to_value(arg, self.env) for arg in argument]
            for argument in args_as_types
        ]
        abstract_values = ["NUMBER", "STRING", "BOOLEAN"]
        all_possible_values = list(itertools.product(*arguments_as_values))
        for values in all_possible_values:
            # check if there's an abstract value in the values
            if any(str(value) in abstract_values for value in values) or any(
                [isinstance(value, (dict, list)) for value in values]
            ):
                if identifier == "String":
                    results.add(string_builtins.return_for_abstract_type(method))
                elif identifier == "Math":
                    results.add(static_return_for_abstract_type(method))
                continue
            values_comma_separated = ", ".join(
                [
                    f"'{str(value)}'" if isinstance(value, str) else str(value)
                    for value in values
                ]
            )
            string_to_execute = f"{identifier}.{method}({values_comma_separated})"
            string_to_execute = string_to_execute.replace("nan", "NaN")
            result_as_value = pm.eval(string_to_execute)
            result_as_type = value_to_type(result_as_value, self.env)
            results.add(result_as_type)
        return results

    # After converting an object in TAS format to an array that's been modified, convert it back to an object
    def array_to_obj(self, original_obj: object, arr: list) -> object:
        obj = {}
        meta = original_obj["__meta__"]
        obj["__meta__"] = meta
        obj["__proto__"] = original_obj["__proto__"]
        enumerable_values = []
        for i, item in enumerate(arr):
            if isinstance(item, RecordResult):
                obj[i] = item
            elif isinstance(item, list) or isinstance(item, OrderedSet):
                record_result_for_item = RecordResult("local", item)
                obj[i] = record_result_for_item
            elif (
                isinstance(item, object)
                and not isinstance(item, list)
                and not isinstance(item, OrderedSet)
            ):
                record_result_for_item = RecordResult("local", [item])
                obj[i] = record_result_for_item
            enumerable_values.append(i)
        obj["__meta__"]["enumerable_values"] = enumerable_values
        return obj

    def obj_to_array(self, obj: object) -> list[RecordResult]:
        enumerable_values = obj["__meta__"]["enumerable_values"]
        arr = []
        for key in enumerable_values:
            arr.append(obj[key])
        return arr

    def update_original_obj(self, original_obj: object, new_obj: object) -> object:
        keys_to_del = list(original_obj.keys())
        for key in keys_to_del:
            del original_obj[key]
        for key in new_obj:
            original_obj[key] = new_obj[key]

    # Each argument could have multiple values, which is why it's a list of lists
    def execute_array_builtin(
        self,
        addr: Address,
        function_called: Type | str,
        func_expr_args: list[str],
        file_path: str,
    ) -> list[Type]:
        possible_values = []
        args = [self.expr_visitor.visit(_, file_path) for _ in func_expr_args]
        if addr.get_addr_type() == "abstract":
            for arg in args:
                for value in arg:
                    if isinstance(value, Address):
                        self.env.move_object_to_abstract_heap(value)
        obj = self.env.lookup_address(addr)
        if isinstance(function_called, Primitive):
            function_called = function_called.get_value()
        if function_called == "at":
            array_obj = self.obj_to_array(obj)
            at = pm.eval(f"(arr, index) => arr.at(index)")
            for i, possible_values_for_arg in enumerate(
                args
            ):  # i isn't used, but it's there to show that I'm iterating over the arguments.
                for arg in possible_values_for_arg:
                    logger.info(f"at arg: {arg}, array_obj {array_obj}")
                    result = at(array_obj, arg)
                    logger.info(f"at result: {result}")
                    possible_values.append(Primitive(at(array_obj, arg)))
        elif function_called == "concat":
            pass
        elif function_called == "copyWithin":
            pass
        elif function_called == "entries":
            pass
        elif function_called == "every":
            pass
        elif function_called == "fill":
            pass
        elif function_called == "filter":
            pass
        elif function_called == "find":
            pass
        elif function_called == "findIndex":
            pass
        elif function_called == "findLast":
            pass
        elif function_called == "findLastIndex":
            pass
        elif function_called == "flat":
            pass
        elif function_called == "flatMap":
            pass
        elif function_called == "forEach":
            pass
        elif function_called == "includes":
            pass
        elif function_called == "indexOf":
            pass
        elif function_called == "join":
            if len(args) == 1:
                separators = list(
                    [_.get_value() for _ in args[0] if isinstance(_, Primitive)]
                )
            else:
                separators = [","]
            array_obj = self.obj_to_array(obj)
            array_obj_values = [_.get_all_values() for _ in array_obj]
            # filter inner lists for primitives
            array_obj_values = [
                [
                    str(item.get_value())
                    for item in inner_list
                    if isinstance(item, Primitive)
                ]
                for inner_list in array_obj_values
            ]
            # interleave possible values with product
            array_obj_values = list(itertools.product(*array_obj_values))
            if len(array_obj_values) == 0:
                return [Primitive("")]
            for separator in separators:
                for inner_list in array_obj_values:
                    logger.info(f"inner list: {inner_list}")
                    result = Primitive(
                        separator.join([str(item) for item in inner_list])
                    )
                    possible_values.append(result)
                    logger.info(f"join result: {result}")
        elif function_called == "keys":
            pass
        elif function_called == "lastIndexOf":
            pass
        elif function_called == "map":
            pass
        elif function_called == "pop":
            # We need to pop the first element from the array and return it
            # We need to convert the object to an array first
            
            if len(obj["__meta__"]["enumerable_values"]) == 0:
                possible_values.append(baseType.NULL)
            elif baseType.NUMBER in obj:
                for value in obj[baseType.NUMBER]:
                    possible_values.append(value)
            else:
                # get minimum object index
                min_index = min(obj["__meta__"]["enumerable_values"])
                possible_values.append(obj[min_index])
                obj["__meta__"]["enumerable_values"].remove(min_index)
                # remove the object from the array
                del obj[min_index]
                #enumerable_values_copy = copy.deepcopy(obj["__meta__"]["enumerable_values"])
                #max_index = max(enumerable_values_copy)
                #obj["__meta__"]["enumerable_values"] = OrderedSet([_ - 1 for _ in enumerable_values_copy])
                #obj_copy = copy.deepcopy(obj)
                #for key in obj_copy:
                #    if isinstance(key,int):
                #        obj[key-1] = obj_copy[key]
                #del obj[max_index]
                # We now need to update the object because the keys need to be shifted
                #new_obj = self.array_to_obj(obj, self.obj_to_array(obj))
                #self.update_original_obj(obj, new_obj)
        elif function_called == "push":
            try:
                if baseType.NUMBER in obj["__meta__"]["enumerable_values"]:
                    for arg in args:
                        self.env.update(addr, [baseType.NUMBER], arg, overwrite=False)
                else:
                    for arg in args:
                        # logger.info(f"push arg: {arg}")
                        if not obj["__meta__"]["enumerable_values"]:
                            self.env.update(addr, [Primitive(0)], arg)
                        else:
                            largest_index = max(obj["__meta__"]["enumerable_values"])
                            self.env.update(addr, [Primitive(largest_index + 1)], arg)
            except Exception as e:
                logger.info(f"Error in push: {e}, {obj}, {args}")
                pass
        elif function_called == "reduce":
            pass
        elif function_called == "reduceRight":
            pass
        elif function_called == "reverse":
            pass
        elif function_called == "shift":
            pass
        elif function_called == "slice":
            pass
        elif function_called == "some":
            pass
        elif function_called == "sort":
            pass
        elif function_called == "splice":
            pass
        elif function_called == "toLocaleString":
            pass
        elif function_called == "toSorted":
            pass
        elif function_called == "toSpliced":
            pass
        elif function_called == "toString":
            pass
        elif function_called == "unshift":
            pass
        elif function_called == "values":
            pass
        elif function_called == "with":
            pass
        else:
            raise ValueError(f"Invalid array builtin: {function_called}")
        return possible_values

    def _guarantees_return(self, node: esprima.nodes.Node):
        if not node:
            return False
        if node.type == "BlockStatement":
            return self._block_guarantees_return(node.body)

    def _block_guarantees_return(self, statements):
        for statement in statements:
            if (
                statement.type == "ReturnStatement"
                or statement.type == "ThrowStatement"
            ):
                return True
            elif statement.type == "IfStatement":
                then_returns = self._guarantees_return(statement.consequent)
                else_returns = statement.alternate and self._guarantees_return(
                    statement.alternate
                )
                if then_returns and else_returns:
                    return True
            elif statement.type == "SwitchStatement":
                for case in statement.cases:
                    if not (
                        case.consequent
                        and self._block_guarantees_return(case.consequent)
                    ):
                        return False
        return False
