import os
from typing import TYPE_CHECKING
from absint_ai.Environment.types.Type import *
from absint_ai.utils.Util import *
from absint_ai.Environment.memory.RecordResult import RecordResult
import absint_ai.Environment.agents.merging as merging
from ordered_set import OrderedSet
import copy
import debugpy
import time
import esprima

if TYPE_CHECKING:
    from absint_ai.Environment.Environment import Environment
    from absint_ai.Interpreter.AbsIntAI import AbsIntAI
    from .expr_visitor import ExprVisitor


class StatementVisitor:
    def __init__(self, env: "Environment", absint_ai: "AbsIntAI", timeout: int):
        self.env = env
        self.absint_ai = absint_ai
        self.debug_lines = [1862]
        self.start_time=time.time()
        self.timeout = timeout

    def visit(self, node, file_path) -> None:
        if node is None:
            return
        if time.time() - self.start_time > self.timeout:
            logger.info(
                f"Timeout on line {self.absint_ai.get_loc(node)} after {time.time() - self.start_time} seconds"
            )
            raise TimeoutError(
                f"Timeout on line {self.absint_ai.get_loc(node)} after {time.time() - self.start_time} seconds"
            )
        if self.absint_ai.debug:
            self.env.update_variable_state()
            if node.loc.start.line in self.debug_lines:
                debugpy.breakpoint()
        if not isinstance(node, dict) or "type" not in node:
            raise ValueError("Invalid AST node format")
        node_type = node["type"]
        method_name = f"visit_{node_type}"
        if self.absint_ai.log_lines_run:
            logger.info(
                f"running on line {self.absint_ai.get_loc(node)}"  # method {method_name} in {file_path}. {len(list(set(self.absint_ai.buggy_line_numbers)))} buggy lines."
            )
        visitor = getattr(self, method_name, self.visit_default)
        result = visitor(node, file_path)
        return result

    def visit_default(self, node, file_path):
        raise NotImplementedError(
            f"No visitor method for {node['type']} on line {self.absint_ai.get_loc(node)}"
        )

    def set_expr_visitor(self, expr_visitor: "ExprVisitor"):
        self.expr_visitor = expr_visitor

    def visit_ExpressionStatement(self, node: esprima.nodes.Node, file_path: str):
        self.expr_visitor.visit(node, file_path)

    def visit_ReturnStatement(self, node: esprima.nodes.Node, file_path: str):
        return_values = self.expr_visitor.visit(node.argument, file_path)

        filtered_return_values = []
        for return_value in return_values:
            if isinstance(return_value, Type):
                filtered_return_values.append(return_value)
            elif (
                isinstance(return_value, list)
                or isinstance(return_value, OrderedSet)
                or isinstance(return_value, set)
            ):
                filtered_return_values.extend(return_value)
        for return_value in filtered_return_values:
            if isinstance(return_value, Address):
                if return_value.get_addr_type() == "abstract":
                    self.env.move_object_to_abstract_heap(
                        self.env.get_current_heap_frame()
                    )
            return_value.set_parent_heap_frame(self.env.get_current_heap_frame())
            self.env.add_return_value(return_value)

    def visit_TryStatement(self, node: esprima.nodes.Node, file_path: str):
        self.visit(node.block, file_path)

    def visit_BlockStatement(self, node: esprima.nodes.Node, file_path: str):
        scope_id = get_schema_from_expr(node, file_path, "block")
        contains_scope_id = self.env.schema_contains_scope_id(scope_id)
        if contains_scope_id:
            scope_schema = self.env.get_schema_for_scope_id(scope_id)
            current_return_values = self.env.get_raw_return_values()
            if self.env.cur_stack_frame.contains_this_address():
                this_address = self.env.cur_stack_frame.get_this_address()
            else:
                this_address = None
            self.env.initialize_from_schema(
                scope_schema,
                file_path,
                allocation_site=scope_id,
                this_address=this_address,
                parent_address=self.env.get_current_heap_frame(),
                abstract_parents=False,
            )
            for return_value in current_return_values:
                self.env.add_return_value(return_value)
        for statement in node.body:
            self.visit(statement, file_path)
        if contains_scope_id:
            self.env.return_from_schema_non_function(file_path)

    # TODO this is where I need to actually split the environment
    def visit_IfStatement(self, node: esprima.nodes.Node, file_path: str):
        test = self.expr_visitor.visit(node.test, file_path)
        # logger.info(f"TEST on line {node.loc.start.line}: {test}, {node.test.type}")
        #concrete_heap_snapshot, abstract_heap_snapshot, stack_snapshot = (
        #    self.env.split_state(file_path)
        #)
        (
            consequent_concrete_heap_snapshot,
            consequent_abstract_heap_snapshot,
            consequent_stack_snapshot,
        ) = (None, None, None)
        (
            alternate_concrete_heap_snapshot,
            alternate_abstract_heap_snapshot,
            alternate_stack_snapshot,
        ) = (
            None,
            None,
            None,
        )
        # if the test contains truthy values, we execute the consequent
        if contains_truthy_value(test):
            if node.consequent:
                # logger.info(
                #    f"Abstract heap addresses in consequent: {self.env.abstract_heap.abstract_heap.keys()}"
                # )
                # self.env.set_state(
                #    concrete_heap=concrete_heap_snapshot,
                #    abstract_heap=abstract_heap_snapshot,
                #    stack_frame=stack_snapshot,
                #    file_path=file_path,
                # )
                self.visit(node.consequent, file_path)
                # (
                #    consequent_concrete_heap_snapshot,
                #    consequent_abstract_heap_snapshot,
                #    consequent_stack_snapshot,
                # ) = self.env.split_state(file_path)
        # We only execute the alternate if the test does not contain ONLY truthy values
        if not only_contains_truthy_values(test):
            # If the test is false, does not contain any truthy values
            if node.alternate:
                # logger.info(
                #    f"Abstract heap addresses in alternate before setting state: {self.env.abstract_heap.abstract_heap.keys()}"
                # )
                # self.env.set_state(
                #    concrete_heap=concrete_heap_snapshot,
                #    abstract_heap=abstract_heap_snapshot,
                #    stack_frame=stack_snapshot,
                #    file_path=file_path,
                # )
                # logger.info(
                #    f"Abstract heap addresses in alternate after setting state: {self.env.abstract_heap.abstract_heap.keys()}"
                # )
                self.visit(node.alternate, file_path)
                # (
                #    alternate_concrete_heap_snapshot,
                #    alternate_abstract_heap_snapshot,
                #    alternate_stack_snapshot,
                # ) = self.env.split_state(file_path)
        # if consequent_abstract_heap_snapshot:
        #    self.env.merge_state(
        #        consequent_concrete_heap_snapshot,
        #        consequent_abstract_heap_snapshot,
        #        consequent_stack_snapshot,
        #        file_path,
        #    )
        # if alternate_abstract_heap_snapshot:
        #    self.env.merge_state(
        #        alternate_concrete_heap_snapshot,
        #        alternate_abstract_heap_snapshot,
        #        alternate_stack_snapshot,
        #        file_path,
        #    )
        # self.env.merge_state(
        #    concrete_heap_snapshot,
        #    abstract_heap_snapshot,
        #    stack_snapshot,
        #    file_path,
        # )

        # TODO the start of path sensitive analysis. We don't want this for now while we compare to WALA and TAJS
        """
        if (
            node.test.type == "BinaryExpression"
            and (node.test.operator == "==" or node.test.operator == "===")
            and node.test.left.type
            == "Identifier"  # temporary for now, we shouldn't need to test for this.
        ):
            if node.test.left.type == "Identifier":
                self._if_identifier_helper(node, file_path, test)
            elif (
                node.test.left.type == "MemberExpression"
                and is_member_expression_static(node.test.left)
            ):

                right = self.expr_visitor.visit(node.test.right, file_path)
                if (
                    node.test.left.object.type == "Identifier"
                ):  # If this is the case, we need to overwrite the variable with only certain addresses
                    pass
                elif (
                    node.test.left.object.type == "MemberExpression"
                ):  # If this is the case, we need to go 1 level deeper and filter the final fields based on the property
                    pass
                left_addresses = self.expr_visitor.visit(
                    node.test.left.object
                )  # This returns a list of addresses
                right_property = self.expr_visitor.visit(
                    node.test.left.property
                )  # We want to filter the above list of addresses based on this property
                # if so, we want to save the current state and then merge it back in after the conditional
                concrete_heap_snapshot, abstract_heap_snapshot, stack_snapshot = (
                    self.env.split_state(file_path)
                )
                (
                    consequent_concrete_heap_snapshot,
                    consequent_abstract_heap_snapshot,
                    consequent_stack_snapshot,
                ) = (None, None, None)
                (
                    alternate_concrete_heap_snapshot,
                    alternate_abstract_heap_snapshot,
                    alternate_stack_snapshot,
                ) = (
                    None,
                    None,
                    None,
                )
                if node.consequent and not (
                    Primitive(False) in test and len(test) == 1
                ):  # If the test is true
                    self.env.set_state(
                        concrete_heap=concrete_heap_snapshot,
                        abstract_heap=abstract_heap_snapshot,
                        stack_frame=stack_snapshot,
                        file_path=file_path,
                    )
                    self.env.overwrite_variable_for_conditional(right, right)
                    self.visit(node.consequent, file_path)
                    (
                        consequent_concrete_heap_snapshot,
                        consequent_abstract_heap_snapshot,
                        consequent_stack_snapshot,
                    ) = self.env.split_state(file_path)

                if (
                    not (Primitive(True) in test and len(test) == 1)
                ) and node.alternate:
                    self.env.set_state(
                        concrete_heap=concrete_heap_snapshot,
                        abstract_heap=abstract_heap_snapshot,
                        stack_frame=stack_snapshot,
                        file_path=file_path,
                    )
                    self.visit(node.alternate, file_path)
                    (
                        alternate_concrete_heap_snapshot,
                        alternate_abstract_heap_snapshot,
                        alternate_stack_snapshot,
                    ) = self.env.split_state(file_path)
                if consequent_abstract_heap_snapshot:
                    self.env.merge_state(
                        consequent_concrete_heap_snapshot,
                        consequent_abstract_heap_snapshot,
                        consequent_stack_snapshot,
                        file_path,
                    )
                if alternate_abstract_heap_snapshot:
                    self.env.merge_state(
                        alternate_concrete_heap_snapshot,
                        alternate_abstract_heap_snapshot,
                        alternate_stack_snapshot,
                        file_path,
                    )
                self.env.merge_state(
                    concrete_heap_snapshot,
                    abstract_heap_snapshot,
                    stack_snapshot,
                    file_path,
                )
        else:
            if node.consequent and not (
                Primitive(False) in test and len(test) == 1
            ):  # If the test is true
                self.visit(node.consequent, file_path)
            if (not (Primitive(True) in test and len(test) == 1)) and node.alternate:
                self.visit(node.alternate, file_path)
        """

    def visit_SwitchStatement(self, node: esprima.nodes.Node, file_path: str):
        for case in node.cases:
            for statement in case.consequent:
                self.visit(statement, file_path)

    def visit_ForInStatement(self, node: esprima.nodes.Node, file_path: str):
        possible_values = self.expr_visitor.visit(node.right, file_path)
        if node.left.type == "VariableDeclaration":
            init_var = node.left.declarations[0].id.name
            init_kind = node.left.kind
        else:
            init_var = node.left.name
            init_kind = None
        for possible_value in possible_values:
            if isinstance(possible_value, Address):
                obj = self.env.lookup_address(possible_value)
                enumerable_values = OrderedSet(obj["__meta__"]["enumerable_values"])
                count = 0
                for key in enumerable_values:
                    count += 1
                    # If it's a let, we need to add a new scope per iteration
                    if init_kind == "let":
                        self._init_non_function_scope(node, file_path)
                    self.env.add(init_var, [Primitive(key)], overwrite=True)
                    self.visit(node.body, file_path)
                    if init_kind == "let":
                        self.env.return_from_schema_non_function(file_path)
    
    def visit_ForOfStatement(self, node: esprima.nodes.Node, file_path: str):
        possible_values = self.expr_visitor.visit(node.right, file_path)
        if node.left.type == "VariableDeclaration":
            init_var = node.left.declarations[0].id.name
            init_kind = node.left.kind
        else:
            init_var = node.left.name
            init_kind = None
        for possible_value in possible_values:
            if isinstance(possible_value, Address):
                obj = self.env.lookup_address(possible_value)
                enumerable_values = OrderedSet(obj["__meta__"]["enumerable_values"])
                count = 0
                for key in enumerable_values:
                    count += 1
                    # If it's a let, we need to add a new scope per iteration
                    if init_kind == "let":
                        self._init_non_function_scope(node, file_path)
                    self.env.add(init_var, [obj[key].get_all_values()], overwrite=True)
                    self.visit(node.body, file_path)
                    if init_kind == "let":
                        self.env.return_from_schema_non_function(file_path)


    def visit_ExpressionStatement(self, node: esprima.nodes.Node, file_path: str):
        return self.expr_visitor.visit(node.expression, file_path)

    def visit_ForStatement(self, node: esprima.nodes.Node, file_path: str):
        self._loop_helper(node, file_path)

    def visit_WhileStatement(self, node: esprima.nodes.Node, file_path: str):
        self._loop_helper(node, file_path)

    def visit_VariableDeclaration(self, node: esprima.nodes.Node, file_path: str):
        self._declare_variables(
            node.declarations,
            node["kind"],
            file_path,
            loc=self.absint_ai.get_loc(node),
        )

    def visit_FunctionDeclaration(self, node: esprima.nodes.Node, file_path: str):
        """
        Just want to store the function name and the parameters in the environment.
        """
        scope_id = get_schema_from_expr(node, file_path, "function")
        params = []
        rest = None
        if node.params:
            for param in node.params:
                if param.type == "Identifier":
                    params.append(param.name)
                elif param.type == "RestElement":
                    rest = param.argument.name
        self.env.add_function(
            node.id.name,
            params,
            node.body,
            scope_id,
            file_path=file_path,
            rest=rest,
            full_function_node=node,
        )

    def visit_ClassDeclaration(self, node: esprima.nodes.Node, file_path: str):
        self.add_class(node, file_path)

    def visit_ImportDeclaration(self, node: esprima.nodes.Node, file_path: str):
        source_path = node.source.value
        source_path = os.path.join(os.path.dirname(file_path), source_path)
        self.absint_ai.run(source_path, reset_heap_ids=False, recurse=False)
        # Need to change back to this module
        self.env.change_module(file_path)
        for specifier in node.specifiers:
            if specifier.type == "ImportSpecifier":
                self.env.add_import(source_path, specifier.local.name)

    def visit_ExportNamedDeclaration(self, node: esprima.nodes.Node, file_path: str):
        if node.declaration:
            self.visit(node.declaration, file_path)
            self.env.add_exported_identifier(file_path, node.declaration.id.name)

    def visit_ExportDefaultDeclaration(self, node: esprima.nodes.Node, file_path: str):
        pass

    def visit_EmptyStatement(self, node: esprima.nodes.Node, file_path: str):
        pass

    def visit_BreakStatement(self, node: esprima.nodes.Node, file_path: str):
        pass

    def visit_ThrowStatement(self, node: esprima.nodes.Node, file_path: str):
        pass

    def visit_ContinueStatement(self, node: esprima.nodes.Node, file_path: str):
        pass

    # region Helper Functions
    def _loop_helper(self, node: esprima.nodes.Node, file_path: str):
        # logger.info(f"LOOP ON {self.absint_ai.get_loc(node)}")
        snapshot = copy.deepcopy(self.env.snapshot())
        allocation_sites_snapshot = copy.deepcopy(self.env.allocation_sites_snapshot())
        keep_looping = self.absint_ai.keep_looping
        loop_count = 0
        all_init_vars = {}
        needs_abstraction = False
        changed_primitives = []
        changed_allocation_sites = []

        general_loop_count = 0
        simplify_method = self.absint_ai.simplify_method
        if node.type == "ForStatement":
            init_kind = node.init.kind if node.init else None
            if init_kind == "let":
                self._init_non_function_scope(node, file_path)
                for init_var in all_init_vars:
                    logger.info(f"ADDING {init_var} to self.env")
                    self.env.add(init_var, all_init_vars[init_var], overwrite=True)
            if node.init and node.init.type == "VariableDeclaration":
                self.visit(node.init, file_path)
            else:
                self.expr_visitor.visit(node.init, file_path)
        else:
            init_kind = None
        test_result = self.expr_visitor.visit(node.test, file_path)
        test_result_false = (
            Primitive(False) in test_result or baseType.NULL in test_result
        ) and len(test_result) == 1
        keep_looping = keep_looping and not test_result_false
        needs_abstraction = not (
            Primitive(True) in test_result and len(test_result) == 1
        )
        loop_id = self.absint_ai.get_loc_id(node)
        should_execute = None
        while keep_looping:
            if loop_count > self.absint_ai.loop_max:
                # If we are abstracting, and we have reached the loop max we raise an exception
                if self.absint_ai.should_abstract and needs_abstraction:
                    logger.info(
                        f"Loop on line {self.absint_ai.get_loc(node)} didn't converge after {loop_count} iterations, max is {self.absint_ai.loop_max}!"
                    )
                    if simplify_method == "llm":
                        loop_count = 0
                        simplify_method = "recency"
                        pass
                        #simplify_method = "allocation_sites"  # if it doesn't converge, we switch to allocation sites
                    elif simplify_method == "recency":
                        loop_count = 0
                        simplify_method = "allocation_sites"  # if it doesn't converge, we switch to allocation sites
                    elif simplify_method == "depth":
                        loop_count = 0
                        simplify_method = "recency"
                    else:
                        raise Exception(
                            f"Loop on line {self.absint_ai.get_loc(node)} didn't converge after {loop_count} iterations, max is {self.absint_ai.loop_max}!"
                        )
                elif (
                    needs_abstraction
                ):  # we only break if we need an abstraction. otherwise we keep executing concretely
                    if init_kind == "let":
                        self.env.return_from_schema_non_function(file_path)
                    break

            self.visit(node.body, file_path)
            if needs_abstraction and self.absint_ai.should_abstract:
                changed_primitives = self.env.changed_from_snapshot(snapshot, log=False)
                code_window = self.absint_ai.get_code_window_around_expr(
                    node, file_path, window_size=50
                )
                loop_body = self.absint_ai.get_code_window_around_expr(
                    node, file_path, window_size=1
                )
                changed_primitives = self.env.changed_primitives_from_snapshot(snapshot)
                changed_allocation_sites = (
                    self.env.changed_object_allocation_sites_from_snapshot(
                        allocation_sites_snapshot, loop_id=loop_id
                    )
                )
                # Generate merging strategies after the first iteration
                # on the first iteration and every 10 iterations after
                if (
                    (not self.env.is_loop_id_summarized(loop_id) and loop_count == 0)
                    or (loop_count > 0 and loop_count % 5 == 0)
                    or should_execute
                ):
                    should_execute = None
                    logger.info(
                        f"Generating merging strategies for loop {loop_id} with changed allocation sites {changed_allocation_sites} and changed primitives {changed_primitives}"
                    )
                    start_merging_time = time.time()
                    should_execute = self.env.generate_merging_strategies(
                        simplify_method,
                        changed_allocation_sites=changed_allocation_sites,
                        loop_id=loop_id,
                        code_window=code_window,
                        loop_iteration=loop_count,
                        loop_body=loop_body,
                        changed_variables=changed_primitives,
                        use_agent=self.absint_ai.use_agent,
                    )
                    end_merging_time = time.time()
                    total_llm_time = end_merging_time - start_merging_time
                    self.absint_ai.llm_time += total_llm_time
                logger.info(
                    f"CALLING SIMPLIFY with method {simplify_method} on loop {loop_id} after {loop_count} iterations"
                )
                self.env.simplify(
                    loop_id=loop_id, changed_allocation_sites=changed_allocation_sites
                )
            if not self.absint_ai.should_abstract:
                if init_kind == "let":
                    self.env.return_from_schema_non_function(file_path)
                break

            changed_primitives = self.env.changed_primitives_from_snapshot(snapshot)
            logger.info(f"CHANGED PRIMITIVES: {changed_primitives}")
            changed_allocation_sites = (
                self.env.changed_object_allocation_sites_from_snapshot(
                    allocation_sites_snapshot, loop_id=loop_id
                )
            )
            logger.info(
                f"CHANGED ALLOCATION SITES AFTER MERGING: {[self.env.get_readable_allocation_site(allocation_site_id) for allocation_site_id in changed_allocation_sites]}"
            )
            keep_looping = (
                len(changed_primitives) > 0 or len(changed_allocation_sites) > 0
            )
            if not self.absint_ai.should_abstract:
                self.env.garbage_collect()  # if we're running without abstraction make sure to GC for comparison
            if keep_looping:
                snapshot = copy.deepcopy(self.env.snapshot())
                allocation_sites_snapshot = copy.deepcopy(
                    self.env.allocation_sites_snapshot()
                )
            if not keep_looping:
                logger.info(
                    f"Loop on line {self.absint_ai.get_loc(node)} converged! {needs_abstraction}"
                )
            else:
                logger.info(
                    f"Loop on line {self.absint_ai.get_loc(node)} didn't converge after {general_loop_count} iterations! {needs_abstraction}"
                )
            if init_kind == "let":
                # First grab variable names from init
                for declaration in node.init.declarations:
                    all_init_vars[declaration.id.name] = self.expr_visitor.visit(
                        declaration.id, file_path
                    )
                self.env.return_from_schema_non_function(file_path)
                self._init_non_function_scope(node, file_path)
                for init_var in all_init_vars:
                    self.env.add(init_var, all_init_vars[init_var], overwrite=True)

            self.expr_visitor.visit(node["update"], file_path)
            test_result = self.expr_visitor.visit(node.test, file_path)
            if not self.absint_ai.keep_looping:
                keep_looping = False
            # If the test result just returns false, break
            if (
                Primitive(False) in test_result or baseType.NULL in test_result
            ) and len(test_result) == 1:
                if init_kind == "let":
                    self.env.return_from_schema_non_function(file_path)
                break

            if (
                not self.absint_ai.should_abstract
                and general_loop_count > self.absint_ai.loop_max
            ):
                if init_kind == "let":
                    self.env.return_from_schema_non_function(file_path)
                break
            if not keep_looping:
                if init_kind == "let":
                    self.env.return_from_schema_non_function(file_path)
            needs_abstraction = not (
                Primitive(True) in test_result and len(test_result) == 1
            )
            if self.absint_ai.should_abstract and needs_abstraction:
                loop_count += 1
            general_loop_count += 1

    def _if_identifier_helper(
        self, node: esprima.nodes.Node, file_path: str, test: OrderedSet[Type]
    ):
        right = self.expr_visitor.visit(node.test.right, file_path)
        left_id = node.test.left.name
        # if so, we want to save the current state and then merge it back in after the conditional
        concrete_heap_snapshot, abstract_heap_snapshot, stack_snapshot = (
            self.env.split_state(file_path)
        )
        (
            consequent_concrete_heap_snapshot,
            consequent_abstract_heap_snapshot,
            consequent_stack_snapshot,
        ) = (None, None, None)
        (
            alternate_concrete_heap_snapshot,
            alternate_abstract_heap_snapshot,
            alternate_stack_snapshot,
        ) = (
            None,
            None,
            None,
        )
        if node.consequent and not (
            Primitive(False) in test and len(test) == 1
        ):  # If the test is true
            self.env.set_state(
                concrete_heap=concrete_heap_snapshot,
                abstract_heap=abstract_heap_snapshot,
                stack_frame=stack_snapshot,
                file_path=file_path,
            )
            self.env.overwrite_variable_for_conditional(left_id, right)
            self.visit(node.consequent, file_path)
            (
                consequent_concrete_heap_snapshot,
                consequent_abstract_heap_snapshot,
                consequent_stack_snapshot,
            ) = self.env.split_state(file_path)

        if (not (Primitive(True) in test and len(test) == 1)) and node.alternate:
            self.env.set_state(
                concrete_heap=concrete_heap_snapshot,
                abstract_heap=abstract_heap_snapshot,
                stack_frame=stack_snapshot,
                file_path=file_path,
            )
            self.visit(node.alternate, file_path)
            (
                alternate_concrete_heap_snapshot,
                alternate_abstract_heap_snapshot,
                alternate_stack_snapshot,
            ) = self.env.split_state(file_path)
        if consequent_abstract_heap_snapshot:
            self.env.merge_state(
                consequent_concrete_heap_snapshot,
                consequent_abstract_heap_snapshot,
                consequent_stack_snapshot,
                file_path,
            )
        if alternate_abstract_heap_snapshot:
            self.env.merge_state(
                alternate_concrete_heap_snapshot,
                alternate_abstract_heap_snapshot,
                alternate_stack_snapshot,
                file_path,
            )
        self.env.merge_state(
            concrete_heap_snapshot,
            abstract_heap_snapshot,
            stack_snapshot,
            file_path,
        )

    # TODO Should double check the logic here since I wrote it a long time ago...
    def _declare_variables(
        self,
        declarators: list[esprima.nodes.VariableDeclarator],
        kind: str,
        file_path: str,
        loc: str = None,
    ) -> None:
        for declarator in declarators:
            if declarator.id.type == "Identifier":
                if declarator.init is None:
                    self.env.add(declarator.id.name, [baseType.NULL], kind=kind)
                else:
                    if declarator.init.type == "AssignmentExpression":
                        self.expr_visitor.visit(declarator.init, file_path)
                        possible_values = self.expr_visitor.visit(
                            declarator.init.left,
                            file_path,
                        )
                    else:
                        possible_values = self.expr_visitor.visit(
                            declarator.init, file_path
                        )
                    if self.env.in_subroutine(file_path):
                        for value in possible_values:
                            if (
                                isinstance(value, Address)
                                and value.get_addr_type() == "concrete"
                            ):
                                if not self.env.is_local(declarator.id.name):
                                    # logger.debug(
                                    #    f"MOVING {declarator.id.name} to abstract heap"
                                    # )
                                    self.env.move_object_to_abstract_heap(value)
                        self.env.add(
                            declarator.id.name,
                            possible_values,
                            kind=kind,
                            overwrite=True,
                        )
                    else:
                        self.env.add(
                            declarator.id.name,
                            possible_values,
                            kind=kind,
                            overwrite=True,
                        )
                self.absint_ai.identifier_info[get_identifier_id(declarator.id)] = (
                    self.env.lookup_and_derive(declarator.id.name)
                )
            if (
                declarator.id.type == "ArrayPattern"
                or declarator.id.type == "ObjectPattern"
            ):
                possible_addrs = self.expr_visitor.visit(declarator.init, file_path)
                self.add_destructuring(declarator.id, possible_addrs, file_path)

    def add_destructuring(
        self,
        declarator,
        possible_addrs: list[Address],
        file_path: str,
    ):
        for possible_addr in possible_addrs:
            if not isinstance(possible_addr, Address):
                continue
            obj: dict[str | AbstractType, RecordResult] = self.env.lookup_address(
                possible_addr
            )
            if declarator.type == "ArrayPattern":
                for i in range(len(declarator.elements)):
                    if declarator.elements[i] == None or i not in obj:
                        continue
                    if declarator.elements[i].type == "Identifier":
                        self.env.add(
                            declarator.elements[i].name,
                            obj[i].get_all_values(),
                            overwrite=True,  # Could this be causing issues with overwriting? TODO
                        )
                    elif (
                        declarator.elements[i].type == "ObjectPattern"
                        or declarator.elements[i].type == "ArrayPattern"
                    ):
                        possible_addrs_for_key = obj[i].get_all_values()
                        self.add_destructuring(
                            property_value, possible_addrs_for_key, file_path
                        )
            elif declarator.type == "ObjectPattern":
                for prop in declarator.properties:
                    property_type = prop.type
                    property_value = prop.value
                    if property_type == "Property":
                        key = prop.key.name
                        if key not in obj:
                            continue
                        if property_value.type == "Identifier":
                            self.env.add(
                                property_value.name,
                                obj[key].get_all_values(),
                                overwrite=True,
                            )
                        elif (
                            property_value.type == "ObjectPattern"
                            or property_value.type == "ArrayPattern"
                        ):
                            possible_addrs_for_key = obj[key].get_all_values()
                            self.add_destructuring(
                                property_value, possible_addrs_for_key, file_path
                            )

    def _init_non_function_scope(self, expr, file_path: str):
        scope_id = get_schema_from_expr(expr, file_path, "for")
        contains_scope_id = self.env.schema_contains_scope_id(scope_id)
        if not contains_scope_id:
            raise Exception("Scope not found for for loop!")
        scope_schema = self.env.get_schema_for_scope_id(scope_id)
        current_return_values = self.env.get_raw_return_values()
        if self.env.cur_stack_frame.contains_this_address():
            this_address = self.env.cur_stack_frame.get_this_address()
        else:
            this_address = None
        self.env.initialize_from_schema(
            scope_schema,
            file_path,
            allocation_site=scope_id,
            this_address=this_address,
            parent_address=self.env.get_current_heap_frame(),
            abstract_parents=False,
            is_function=False,
        )

    # used for ClassDeclarations and ClassExpressions
    def add_class(
        self, expr: esprima.nodes.ClassDeclaration, file_path: str
    ) -> OrderedSet[Type]:
        class_allocation_site = allocation_site_from_expr(expr, file_path)
        if expr.id:
            className = expr.id.name
        else:
            className = ""
        if expr.superClass:
            prototype_address = self.env.lookup(expr.superClass.name).get_all_values()
        else:
            prototype_address = [
                self.env.add_object_to_heap(
                    {},
                    allocation_site=class_allocation_site,
                    add_proto=False,
                    allocation_site_type="class",
                )
            ]
        class_object = {
            "prototype": prototype_address,  # this isn't a list because env.lookup returns a list of addresses.
            "__meta__": {"enumerable_values": [], "type": "class"},
        }
        for method in expr.body.body:
            method_allocation_site = allocation_site_from_expr(method, file_path)
            if method.type == "MethodDefinition":
                if method.kind == "constructor":
                    name = "constructor"
                else:
                    name = method.key.name

                params = [param.name for param in method.value.params]
                body = method.value.body
                scope_id = get_schema_from_expr(method, file_path, "function")
                schema = self.env.get_schema_for_scope_id(scope_id)
                method_function = self.env.initialize_new_function(
                    name,
                    params,
                    body,
                    scope_id,
                    file_path=file_path,
                    full_function_node=method,
                )
                if method.kind == "constructor":
                    class_object["constructor"] = [method_function]
                else:
                    for prototype in prototype_address:
                        if (
                            prototype.get_addr_type() == "abstract"
                            and method_function.get_addr_type() == "concrete"
                        ):
                            self.env.move_object_to_abstract_heap(method_function)
                        # logger.info(
                        #    f"adding {method_function} to {name} at {prototype}"
                        # )
                        self.env.add_value_for_field(prototype, name, method_function)
            elif method.type == "PropertyDefinition":
                if method.key.type == "Identifier":
                    class_object[method.key.name] = self.expr_visitor.visit(
                        method.value, file_path
                    )
        # self.classes[className] = clazz
        result = self.env.add_object_to_heap(
            class_object,
            allocation_site=class_allocation_site,
            allocation_site_type="class",
        )
        self.env.add(className, result)
        return OrderedSet([result])

    # endregion
