import unittest
import unittest.mock
from absint_ai.Interpreter import AbsIntAI
from absint_ai.utils.Logger import logger
from absint_ai.Environment.Environment import Environment
from absint_ai.Environment.types.Type import *
from absint_ai.Environment.memory.ConcreteHeap import *
from absint_ai.Environment.memory.AbstractHeap import *
from absint_ai.utils.Util import *
import os


class testAllocationSiteAbstraction(unittest.TestCase):
    def setUp(self):
        self.testLLM = False
        self.base_class = "js_files/allocation_site_abstraction"

    def test_simple_loop(self):
        check = AbsIntAI.AbsIntAI(simplify_method="allocation_sites")
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "simple_loop.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(env.lookup_and_derive("acc"), ["NUMBER"])

    def test_simple_loop2(self):
        check = AbsIntAI.AbsIntAI(simplify_method="allocation_sites")
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "simple_loop2.js"
        )
        env = check.run(scriptPath)
        logger.info(f"people: {env.lookup_and_derive('people')}")
        people = env.lookup_and_derive("people")
        self.assertTrue(
            is_superset(
                [{"NUMBER": ["NUMBER"]}],
                env.lookup_and_derive("people"),
            )
        )
        # self.assertEqual(
        #    order_dict(env.lookup_and_derive("people")[0]),
        #    {"NUMBER": ["NULL", "NUMBER"]},
        # )

    def test_simple_loop3(self):
        check = AbsIntAI.AbsIntAI(simplify_method="allocation_sites")
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "simple_loop3.js"
        )
        env = check.run(scriptPath)
        """
        {"NUMBER": [{"STRING": ["STRING", "NUMBER"]}]}
        """
        logger.info(f"people: {env.lookup_and_derive('people')}")
        people = env.lookup_and_derive("people")
        logger.info(people)
        # self.assertTrue(
        #    is_superset(
        #        [{"STRING": ["NULL", "NUMBER", "STRING"]}, "NULL"], people[0]["NUMBER"]
        #    )
        # )

    def test_simple_loop4(self):
        check = AbsIntAI.AbsIntAI(simplify_method="allocation_sites")
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "simple_loop4.js"
        )
        env = check.run(scriptPath)

        obj_value = env.lookup_and_derive("obj")
        logger.info(f"obj_value: {obj_value}")

    def test_loop_closure(self):
        check = AbsIntAI.AbsIntAI(simplify_method="allocation_sites")
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "loop_closure.js"
        )
        env = check.run(scriptPath)

    def test_functions_merged(self):
        check = AbsIntAI.AbsIntAI(simplify_method="allocation_sites")
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "functions_merged.js"
        )
        env = check.run(scriptPath)
        self.assertTrue(is_superset([5, "bar"], env.lookup_and_derive("x")))
        self.assertTrue(is_superset([5, "bar"], env.lookup_and_derive("y")))

    def test_loop_with_function(self):
        check = AbsIntAI.AbsIntAI(
            simplify_method="allocation_sites", loop_max=5, check_pointers=True
        )
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "loop_with_function.js"
        )
        env = check.run(scriptPath)
        x_value = env.lookup_and_derive("x")
        logger.info(f"x_value: {x_value}")
        self.assertTrue("NUMBER" in x_value)
        self.assertTrue(len(x_value) == 1)

    def test_loop_with_function_outer(self):
        check = AbsIntAI.AbsIntAI(
            simplify_method="allocation_sites", check_pointers=True
        )
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "loop_with_function_outer.js"
        )  # TODO Visualize this closer...
        env = check.run(scriptPath)
        x_value = env.lookup_and_derive("x")
        logger.info(f"x_value: {x_value}")
        self.assertTrue("STRING" in x_value)
        self.assertTrue("NUMBER" in x_value)
        self.assertTrue(len(x_value) == 2)

    def test_loop_with_function2(self):
        check = AbsIntAI.AbsIntAI(
            simplify_method="allocation_sites", check_pointers=True
        )
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "loop_with_function2.js"
        )
        env = check.run(scriptPath)
        self.assertTrue(is_superset([5], env.lookup_and_derive("x")))

    def test_mixed_value_types(self):
        check = AbsIntAI.AbsIntAI(
            simplify_method="allocation_sites", check_pointers=True
        )
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "mixed_value_types.js"
        )
        env = check.run(scriptPath)
        self.assertTrue(is_superset([5, 10], env.lookup_and_derive("x")))

    def test_no_abstract_heap_overwrites(self):
        check = AbsIntAI.AbsIntAI(
            simplify_method="allocation_sites", check_pointers=True
        )
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "no_abstract_heap_overwrites.js"
        )
        env = check.run(scriptPath)
        x_value = env.lookup_and_derive("x")
        logger.info(f"x_value: {x_value}")
        self.assertTrue("NUMBER" in x_value)
        # self.assertEqual(env.lookup_and_derive("y"), [5])

    def test_abstraction_push(self):
        check = AbsIntAI.AbsIntAI(
            simplify_method="allocation_sites", check_pointers=True
        )
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "abstraction_push.js"
        )
        env = check.run(scriptPath)
        values = env.lookup_and_derive("obj")[0]["NUMBER"]
        logger.info(values)
        # self.assertTrue("NULL" in values)
        self.assertTrue("NUMBER" in values)
        self.assertTrue("STRING" in values)
        self.assertTrue(len(values) >= 2)

    def test_abstraction_push2(self):
        check = AbsIntAI.AbsIntAI(
            simplify_method="allocation_sites", check_pointers=True
        )
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "abstraction_push2.js"
        )
        env = check.run(scriptPath)


class testMultipleRuns(unittest.TestCase):
    def setUp(self):
        self.testLLM = False
        self.base_class = "js_files/multiple_runs"

    def test_multiple_runs(self):
        check = AbsIntAI.AbsIntAI(
            simplify_method="allocation_sites", check_pointers=True
        )
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "multiple_runs1.js"
        )
        env = check.run(scriptPath)
        self.assertTrue(is_superset([{"foo": ["bar"]}, 5], env.lookup_and_derive("x")))
        self.assertTrue(
            is_superset(
                [{"x": [2, 8], "foo": ["not bar", "bar"], "y": [7]}],
                env.lookup_and_derive("obj"),
            )
        )


class testSanityChecksOnRealPrograms(unittest.TestCase):
    def setUp(self):
        self.testLLM = False
        self.base_class = "js_files/allocation_site_abstraction"
        self.run = False

    def test_abstractions_have_fewer_bugs_knockout(self):
        if not self.run:
            return
        check_with_abstractions = AbsIntAI.AbsIntAI(
            simplify_method="allocation_sites",
            should_abstract=True,
            log_lines_run=True,
            check_pointers=False,
        )
        check_without_abstractions = AbsIntAI.AbsIntAI(
            simplify_method="allocation_sites",
            should_abstract=False,
            log_lines_run=True,
            check_pointers=False,
        )
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "todomvc_knockout.js"
        )
        env_with_abstractions = check_with_abstractions.check(scriptPath)
        env_without_abstractions = check_without_abstractions.check(scriptPath)
        with self.subTest():
            self.assertEqual(check_without_abstractions.simplify_calls, 0)
        with self.subTest():
            self.assertTrue(
                len(list(set(check_with_abstractions.trace)))
                >= len(list(set(check_without_abstractions.trace)))
            )
        with self.subTest():
            self.assertTrue(
                len(list(set(check_with_abstractions.buggy_lines)))
                >= len(list(set(check_without_abstractions.buggy_lines)))
            )
        with self.subTest():
            buggy_lines_with_abstractions = set(
                check_with_abstractions.buggy_line_numbers
            )
            buggy_lines_without_abstractions = set(
                check_without_abstractions.buggy_line_numbers
            )
            self.assertTrue(
                len(buggy_lines_without_abstractions)
                < len(buggy_lines_with_abstractions)
            )
            # This would be true if we ran fully until convergence
            # self.assertTrue(
            #    buggy_lines_without_abstractions.issubset(buggy_lines_with_abstractions)
            # )

    def test_abstractions_have_fewer_bugs_vue(self):
        if not self.run:
            return
        check_with_abstractions = AbsIntAI.AbsIntAI(
            simplify_method="allocation_sites",
            should_abstract=True,
            log_lines_run=True,
            check_pointers=False,
        )
        check_without_abstractions = AbsIntAI.AbsIntAI(
            simplify_method="allocation_sites",
            should_abstract=False,
            log_lines_run=True,
            check_pointers=False,
        )
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "todomvc_vue.js"
        )
        env_with_abstractions = check_with_abstractions.check(scriptPath)
        env_without_abstractions = check_without_abstractions.check(scriptPath)
        with self.subTest():
            self.assertEqual(check_without_abstractions.simplify_calls, 0)
        with self.subTest():
            self.assertTrue(
                len(list(set(check_with_abstractions.trace)))
                >= len(list(set(check_without_abstractions.trace)))
            )
        with self.subTest():
            self.assertTrue(
                len(list(set(check_with_abstractions.buggy_lines)))
                >= len(list(set(check_without_abstractions.buggy_lines)))
            )
        with self.subTest():
            buggy_lines_with_abstractions = set(
                check_with_abstractions.buggy_line_numbers
            )
            buggy_lines_without_abstractions = set(
                check_without_abstractions.buggy_line_numbers
            )
            self.assertTrue(
                len(buggy_lines_without_abstractions)
                < len(buggy_lines_with_abstractions)
            )
            # This would be true if we ran fully until convergence
            # self.assertTrue(
            #    buggy_lines_without_abstractions.issubset(buggy_lines_with_abstractions)
            # )


if __name__ == "__main__":
    unittest.main()
