import unittest
import unittest.mock
from absint_ai.Interpreter import AbsIntAI
from absint_ai.utils.Logger import logger
from absint_ai.Environment.Environment import Environment
from absint_ai.Environment.types.Type import *
from absint_ai.Environment.memory.ConcreteHeap import *
from absint_ai.Environment.memory.AbstractHeap import *
from absint_ai.utils.Util import *
import os


class TestBasicFunctionality(unittest.TestCase):
    def setUp(self):
        self.testLLM = False
        self.base_class = "js_files/basic_functionality"

    def test_literal_assignment(self):
        check = AbsIntAI.AbsIntAI()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "literal_assignments.js"
        )
        check.set_test_mode()
        env = check.run(scriptPath)
        # logger.info(env)
        self.assertEqual(["hello"], env.lookup_and_derive("a"))
        self.assertEqual([42], env.lookup_and_derive("b"))
        self.assertEqual([5], env.lookup_and_derive("c"))
        self.assertEqual(["hello"], env.lookup_and_derive("d"))

    def test_object_assignment(self):
        check = AbsIntAI.AbsIntAI()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "object_assignments.js"
        )
        check.set_test_mode()
        env = check.run(scriptPath)
        logger.debug(env.lookup_and_derive("obj"))
        self.assertEqual(
            [
                {
                    "key1": [1],
                    "key2": [2],
                    "foo": [3],
                    "bar": [4],
                    "nested": [{"hello": ["world"]}],
                }
            ],
            env.lookup_and_derive("obj"),
        )

    def test_bools(self):
        check = AbsIntAI.AbsIntAI()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "bools.js"
        )
        check.set_test_mode()
        env = check.run(scriptPath)
        logger.info(env.lookup_and_derive("b"))
        self.assertEqual([True], env.lookup_and_derive("b"))
        self.assertEqual([False], env.lookup_and_derive("c"))
        self.assertEqual(["BOOL"], env.lookup_and_derive("d"))

    def test_update_nested_object(self):
        check = AbsIntAI.AbsIntAI()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "update_nested_object.js"
        )
        check.set_test_mode()
        env = check.run(scriptPath)
        self.assertEqual([{"a": [{"b": [{"c": [1]}]}]}], env.lookup_and_derive("obj"))

    def test_object_function_assignment(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "object_function_assignment.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([5], env.lookup_and_derive("a"))

    def test_object_accesses(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "object_accesses.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(
            [{0: [{"x": [7, 8], "y": [7, 8]}]}], env.lookup_and_derive("snake")
        )

    def test_object_reference(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "object_literal_references.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["foo"], env.lookup_and_derive("a"))
        self.assertEqual([1], env.lookup_and_derive("b"))
        self.assertEqual(
            [{"a": ["hello"], "b": ["world"]}], env.lookup_and_derive("obj")
        )

        self.assertEqual(["hello"], env.lookup_and_derive("c"))
        self.assertEqual(["world"], env.lookup_and_derive("d"))
        self.assertEqual(["world"], env.lookup_and_derive("e"))

    def test_object_update(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "object_updates.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(
            [
                {
                    "foo": ["hello"],
                    "bar": ["world"],
                    "blah": [43],
                    "nestedField": [{"nestedField": ["nested"]}],
                }
            ],
            env.lookup_and_derive("obj"),
        )

    def test_class_expression(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "class_expression.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["blah"], env.lookup_and_derive("result"))

    def test_array_access(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "array_access.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([1], env.lookup_and_derive("b"))

    def test_array_update(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "array_update.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([10], env.lookup_and_derive("b"))

    def test_update_expression(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "update_expression.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([8], env.lookup_and_derive("a"))
        self.assertEqual([6], env.lookup_and_derive("b"))
        self.assertEqual([6], env.lookup_and_derive("c"))

    def test_alias(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "alias.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(
            env.lookup_and_derive("obj"),
            [{"a": ["foo"], "b": [{"bar": ["bar"]}], "test": ["test"]}],
        )

    def test_invalid_access(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "invalid_access.js"
        )
        # with self.assertRaises(Exception) as e:
        env = check.run(scriptPath)
        logger.info(check.buggy_lines)
        self.assertTrue(any(["line 3" in _ for _ in check.buggy_lines]))

    def test_if(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(os.path.dirname(__file__), self.base_class, "if.js")
        env = check.run(scriptPath)
        self.assertEqual([5], env.lookup_and_derive("nullablex"))
        self.assertEqual(
            [{"nullable1": ["foo1"]}], env.lookup_and_derive("nullableObj1")
        )
        self.assertEqual(
            [
                {"hi": [5], "foo": [10], "bar": [20]},
            ],
            env.lookup_and_derive("obj"),
        )

        self.assertEqual([{"test": ["foo"]}], env.lookup_and_derive("testObj"))

    def test_if2(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(os.path.dirname(__file__), self.base_class, "if2.js")
        env = check.run(scriptPath)
        self.assertEqual([{"rotating": [False]}], env.lookup_and_derive("obj"))

    def test_if_reassign(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "if_reassign.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(
            [10], env.lookup_and_derive("x")
        )  # TODO this should be [10,5] but we need to double check some of the control flow stuff.

    def test_nested_if(self):
        return  # TODO: fix this after formalizing control flow
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "if_nested.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(env.lookup_and_derive("nullablex"), ["NULL", 5])

        self.assertEqual(env.lookup_and_derive("nullabley"), ["NULL", 10])

    def test_block_simple(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "block_simple.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([2], env.lookup_and_derive("x"))
        self.assertEqual([1], env.lookup_and_derive("y"))

    def test_block_return(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "block_return.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([5], env.lookup_and_derive("a"))

    def test_blocks(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "blocks.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(env.lookup_and_derive("inner"), [])
        self.assertEqual(
            [{"foo": [5], "blah": ["set from inner"]}],
            env.lookup_and_derive("outer2"),
        )
        self.assertEqual([5], env.lookup_and_derive("outer1"))
        self.assertEqual(["foo"], env.lookup_and_derive("outer"))
        self.assertEqual(["accessible"], env.lookup_and_derive("accessibleFromOuter"))

    def test_function_assignment(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "function_assignment.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["bar"], env.lookup_and_derive("test"))

    def test_reassignment(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "reassignment.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([{"foo": ["bar"]}], env.lookup_and_derive("a"))

    def test_class_update(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "class_update.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["dog"], env.lookup_and_derive("oldDogName"))
        self.assertEqual(["cat"], env.lookup_and_derive("dogName"))

    def test_simple_bug(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), "js_files/bugs/simple_bug.js"
        )
        env = check.run(scriptPath)
        self.assertTrue(any(["line 2" in _ for _ in check.buggy_lines]))

    def test_class(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "class.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["bar"], env.lookup_and_derive("foo"))

    def test_class_field(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "class_field.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([3], env.lookup_and_derive("result"))

    def test_class_object_field(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "class_object_field.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([{"a": [1], "b": [0]}], env.lookup_and_derive("result"))

    def test_class_inheritance(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "class_inheritance.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["Ford"], env.lookup_and_derive("car_name"))
        self.assertEqual(["Mustang"], env.lookup_and_derive("car_model"))
        # self.assertTrue(is_superset(["Super Function"], env.lookup_and_derive("super_function")))

    def test_update_obj_field(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "update_obj_field.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(env.lookup_and_derive("obj"), [{"a": [2]}])


class testFunctions(unittest.TestCase):
    def setUp(self):
        self.testLLM = False
        self.base_class = "js_files/functions"

    def test_simple_function(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "simple_function.js"
        )
        env = check.run(scriptPath)
        logger.info(env.lookup_and_derive("foo"))
        # self.assertTrue(isinstance(env.lookup_and_derive("foo")[0], Function))
        self.assertEqual(["foo"], env.lookup_and_derive("simpleCall"))
        self.assertEqual(env.lookup_and_derive("localVar"), [])
        self.assertEqual(["bar"], env.lookup_and_derive("simpleCall2"))
        self.assertEqual(env.lookup_and_derive("localVar"), [])
        self.assertEqual(env.lookup_and_derive("localVar2"), [])
        self.assertEqual(["test"], env.lookup_and_derive("withArgsCall"))
        self.assertEqual(["test"], env.lookup_and_derive("withArgsCall2"))
        self.assertEqual(["test2"], env.lookup_and_derive("withArgsCall3"))
        self.assertEqual(["nested"], env.lookup_and_derive("nestedCall"))

    def test_local_function(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "local_function.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([1], env.lookup_and_derive("result"))

    def test_conditional_return_object(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "conditional_return_object.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([0, {"a": [1]}], env.lookup_and_derive("result"))

    def test_return_object(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "return_object.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([{"a": ["foo"]}], env.lookup_and_derive("a"))

    def test_conditional_return(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "conditional_return.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([5, 1, 3], env.lookup_and_derive("a"))
        self.assertEqual([7, 2], env.lookup_and_derive("b"))

    def test_conditional_return2(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "conditional_return2.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["NULL", "string"], env.lookup_and_derive("a"))
        self.assertEqual(["NULL"], env.lookup_and_derive("b"))

    def test_function_return(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "function_return.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["foo"], env.lookup_and_derive("result"))

    def test_chained_calls(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "chained_calls.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([1], env.lookup_and_derive("result2"))

    def test_function_field(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "function_field.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([1], env.lookup_and_derive("result"))

    def test_nested_calls(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "nested_calls.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([10], env.lookup_and_derive("result"))

    def test_nested_function(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "nested_function.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["foo"], env.lookup_and_derive("blah"))

    def test_function_scope(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "function_scope.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(env.lookup_and_derive("x"), [])
        self.assertEqual([5], env.lookup_and_derive("t"))

    def test_global_mutation(self):
        check = AbsIntAI.AbsIntAI()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "global_mutation.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(env.lookup_and_derive("x"), ["NUMBER"])
        self.assertEqual(env.lookup_and_derive("a"), ["NUMBER"])
        self.assertEqual(env.lookup_and_derive("c"), ["NUMBER"])

    def test_global_mutation_object(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "global_mutation_object.js"
        )
        env = check.run(scriptPath)
        self.assertTrue(env.lookup_and_derive("obj"), [{"x": [0, 1]}])
        self.assertTrue(env.lookup_and_derive("a"), [0, 1])
        self.assertTrue(env.lookup_and_derive("c"), [0, 1])


class testClosures(unittest.TestCase):
    def test_closure(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), "js_files/closures/closures.js"
        )
        env = check.run(scriptPath)
        logger.info(env.lookup_and_derive("x"))
        self.assertEqual(env.lookup_and_derive("x"), [])
        self.assertEqual([0, 1], env.lookup_and_derive("a"))
        # self.assertTrue(env.lookup_and_derive("c"), [0, 1, 2, 3])
        self.assertEqual([0, 1], env.lookup_and_derive("d"))

    def test_closure_global_object(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), "js_files/closures/closure_global_object.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([0], env.lookup_and_derive("a"))

    def test_closure_objects(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), "js_files/closures/closure_objects.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(env.lookup_and_derive("x"), [])
        self.assertEqual([{"a": [0, 1, 2, 3]}], env.lookup_and_derive("c"))
        self.assertEqual([{"a": [0, 1]}], env.lookup_and_derive("d"))

    def test_closure2(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), "js_files/closures/closures2.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([0], env.lookup_and_derive("a"))
        self.assertEqual([0, 20], env.lookup_and_derive("b"))


class testLoops(unittest.TestCase):
    def setUp(self):
        self.testLLM = False
        self.base_class = "js_files/loops"

    def run_multiple_iterations(self, num_iters, filepath, model="gpt-4o-2024-08-06"):
        buggy_lines = []
        for i in range(num_iters):
            check = AbsIntAI.AbsIntAI(
                simplify_method="llm",
                file_name=os.path.join(os.path.dirname(__file__), filepath),
                model=model,
            )
            scriptName = os.path.join(os.path.dirname(__file__), filepath)
            try:
                env = check.run(scriptName)
                buggy_lines.append(check.buggy_lines)
            except Exception:
                continue
        # get the intersection of the buggy lines
        buggy_lines = list(set.intersection(*map(set, buggy_lines)))
        return buggy_lines

    def test_simple_while_bug_llm(
        self,
    ):  # don't run this test every time since it costs money
        if not self.testLLM:
            return
        check = AbsIntAI.AbsIntAI(
            simplify_method="llm",
            file_name=os.path.join(
                os.path.dirname(__file__), "js_files/bugs/simple_while_loop_bug.js"
            ),
        )
        scriptPath = os.path.join(
            os.path.dirname(__file__), "js_files/bugs/simple_while_loop_bug.js"
        )
        env = check.run(scriptPath)
        print(f"THE BUGGY LINES ARE {check.buggy_lines}")
        self.assertTrue("bar not found in object v at line 13" in check.buggy_lines)

    def test_for_in(self):
        return
        check = AbsIntAI.AbsIntAI()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "for_in.js"
        )
        env = check.run(scriptPath)
        self.assertTrue(env.lookup_and_derive("obj"), [{"a": [1], "b": [2]}])
        # self.assertTrue(env.lookup_and_derive("i"), [0])

    def test_for_in_scope(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "for_in_scope.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([6], env.lookup_and_derive("acc"))
        self.assertEqual(["foo"], env.lookup_and_derive("i"))

    def test_for_in_scope_basic(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "for_in_scope_basic.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([0], env.lookup_and_derive("result1"))
        self.assertEqual([1], env.lookup_and_derive("result2"))
        self.assertEqual([2], env.lookup_and_derive("result3"))

    def test_for_in_closure_let(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "for_in_closure_let.js"
        )
        env = check.run(scriptPath)
        logger.info(env.lookup_and_derive("obj"))
        self.assertEqual([0], env.lookup_and_derive("a1"))
        self.assertEqual([1], env.lookup_and_derive("b"))
        self.assertEqual([2], env.lookup_and_derive("c"))

    def test_for_loop_closure_let(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "for_loop_closure_let.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([0], env.lookup_and_derive("a"))
        self.assertEqual([1], env.lookup_and_derive("b"))
        self.assertEqual([2], env.lookup_and_derive("c"))

    def test_concrete_for_loop(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "concrete_for_loop.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(
            [{0: ["foo"], 1: ["foo"], 2: ["foo"]}],
            env.lookup_and_derive("obj"),
        )

    def test_concrete_for_loop2(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "concrete_for_loop2.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(
            [{0: ["foo"], 1: ["foo"], 2: ["foo"]}],
            env.lookup_and_derive("obj"),
        )

    def test_fibonacci(
        self,
    ):  # don't run this test every time since it costs money
        return  # TODO I think some of the address validation is broken with sets. Fix when we integrate LLMs
        concrete_heap_id_reset()
        abstract_heap_id_reset()
        check = Check.Check(
            simplify_method="file",
            file_name=os.path.join(
                os.path.dirname(__file__), "abstracted_heaps/fibonacci.json"
            ),
        )
        scriptPath = os.path.join(
            os.path.dirname(__file__), "js_files/algorithms/fibonacci.js"
        )
        env = check.run(scriptPath)
        self.assertTrue(env.lookup_and_derive("valid_result"), ["NUMBER"])


class testFunctionAbstractions(unittest.TestCase):
    def test_validate_simple_abstraction(self):
        env = Environment()
        check = AbsIntAI.AbsIntAI()
        scriptPath = os.path.join(
            os.path.dirname(__file__),
            "js_files/function_abstractions/simple_abstraction.js",
        )
        env = check.run(scriptPath)
        obj = env.lookup_and_derive("obj")


class testRecursion(unittest.TestCase):
    def setUp(self):
        self.testLLM = False
        self.base_class = "js_files/recursion"

    def test_tree_recursion(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "tree_recursion.js"
        )
        env = check.run(scriptPath)
        logger.info(env.lookup_and_derive("tree"))
        self.assertEqual(
            env.lookup_and_derive("tree"),
            [
                0,
                {
                    "left": [0, "Abstract Address(19)"],
                    "right": [0, "Abstract Address(19)"],
                    "value": ["TOP"],
                },
            ],
        )

    def test_recursive_member_function(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "recursive_member_function.js"
        )
        env = check.run(scriptPath)
        logger.info(env.lookup_and_derive("result"))
        self.assertEqual(
            env.lookup_and_derive("result"),
            [
                0,
                {
                    "left": [0, "Abstract Address(20)"],
                    "right": [0, "Abstract Address(20)"],
                    "value": ["TOP"],
                },
            ],
        )

    def test_tree_mutual_recursion(self):
        concrete_heap_id_reset()
        abstract_heap_id_reset()
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "tree_mutual_recursion.js"
        )
        env = check.run(scriptPath)
        logger.info(env.lookup_and_derive("tree"))
        self.assertEqual(
            env.lookup_and_derive("tree"),
            [
                0,
                {
                    "value": ["TOP"],
                    "left": [
                        0,
                        {"value": ["TOP"], "right": [0, "Abstract Address(21)"]},
                    ],
                },
            ],
        )

    def test_object_recursion(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "object_recursion.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(
            str(
                [
                    {
                        "a": [1],
                        "obj2_ref": [{"b": [2], "obj1_ref": [Address(16, "abstract")]}],
                    }
                ]
            ),
            str(env.lookup_and_derive("obj1")),
        )
        self.assertEqual(
            str(
                [
                    {
                        "b": [2],
                        "obj1_ref": [{"a": [1], "obj2_ref": [Address(15, "abstract")]}],
                    }
                ]
            ),
            str(env.lookup_and_derive("obj2")),
        )


class testIndividualFunctions(unittest.TestCase):
    def setUp(self):
        self.testLLM = False

    def test_validate_abstraction_simple(self):
        base_schema = {}
        base_schema["global"] = {
            "private_vars": [],
            "shared_vars": ["a"],
            "parent": None,
            "type": "global",
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        env.add("a", Primitive(5))
        abstraction = {"a": [5]}
        second_abstraction = {"a": ["NUMBER"]}
        buggy_abstraction = {"a": ["STRING"]}
        self.assertTrue(env.validate_all_variable_abstractions(abstraction))
        self.assertTrue(env.validate_all_variable_abstractions(second_abstraction))
        self.assertFalse(env.validate_all_variable_abstractions(buggy_abstraction))

    def test_validate_abstraction_simple2(self):
        base_schema = {}
        base_schema["global"] = {
            "private_vars": ["a", "b"],
            "shared_vars": [],
            "parent": None,
            "type": "global",
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        env.add("a", Primitive(5))
        env.add("b", Primitive("hello"))
        abstraction = {"a": [5], "b": ["hello"]}
        second_abstraction = {"a": ["NUMBER"], "b": ["STRING"]}
        buggy_abstraction = {"a": ["STRING"], "b": ["NUMBER"]}
        self.assertTrue(env.validate_all_variable_abstractions(abstraction))
        self.assertTrue(env.validate_all_variable_abstractions(second_abstraction))
        self.assertFalse(env.validate_all_variable_abstractions(buggy_abstraction))

    def test_validate_existing_abstraction(self):
        base_schema = {}
        base_schema["global"] = {
            "private_vars": ["a", "b"],
            "shared_vars": ["n"],
            "parent": None,
            "type": "global",
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        env.add("n", baseType.NUMBER)
        abstraction = {"n": ["NUMBER"]}
        second_abstraction = {"n": ["TOP"]}
        buggy_abstraction = {"n": ["STRING"]}
        buggy_abstraction2 = {"n": [5]}
        # self.assertTrue(env.validate_abstraction(abstraction))
        # self.assertTrue(env.validate_abstraction(second_abstraction))
        self.assertFalse(env.validate_all_variable_abstractions(buggy_abstraction))
        self.assertFalse(env.validate_all_variable_abstractions(buggy_abstraction2))

    def test_incorrect_var_abstraction(self):
        base_schema = {}
        base_schema["global"] = {
            "private_vars": ["a", "b"],
            "shared_vars": ["n"],
            "parent": None,
            "type": "global",
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        env.add("n", baseType.TOP)
        abstraction = {"n": ["NUMBER"]}
        buggy_abstraction = {"n": ["STRING"]}
        buggy_abstraction2 = {"n": [5]}
        self.assertFalse(env.validate_all_variable_abstractions(buggy_abstraction))
        self.assertFalse(env.validate_all_variable_abstractions(buggy_abstraction2))

    def test_incorrect_var_abstraction2(self):
        base_schema = {}
        base_schema["global"] = {
            "private_vars": ["a", "b"],
            "shared_vars": ["n"],
            "parent": None,
            "type": "global",
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        env.add("n", baseType.NULL)
        abstraction = {"n": ["NUMBER"]}
        buggy_abstraction = {"n": ["STRING"]}
        buggy_abstraction2 = {"n": [5]}
        self.assertFalse(env.validate_all_variable_abstractions(abstraction))
        self.assertFalse(env.validate_all_variable_abstractions(buggy_abstraction))
        self.assertFalse(env.validate_all_variable_abstractions(buggy_abstraction2))

    def test_incorrect_object_abstraction(self):
        base_schema = {}
        base_schema["global"] = {
            "private_vars": [],
            "shared_vars": ["n"],
            "parent": None,
            "type": "global",
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        env.abstract_heap.add(
            {},
            Address("1", "abstract"),
        )
        address_type_abstraction = {
            "STRING": ["bar1", "bar2", "bar3", "bar4"],
            "NUMBER": ["bar5", "bar6", "bar7", "bar8"],
        }

        self.assertFalse(
            env.validate_address_abstraction(
                Address("1", "abstract"), address_type_abstraction
            )
        )

    def test_incorrect_object_abstraction2(self):
        base_schema = {}
        base_schema["global"] = {
            "private_vars": [],
            "shared_vars": ["n"],
            "parent": None,
            "type": "global",
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        env.abstract_heap.add(
            {"foo": ["bar"]},
            Address("1", "abstract"),
        )
        address_type_abstraction = {"NUMBER": ["STRING"]}

        self.assertFalse(
            env.validate_address_abstraction(
                Address("1", "abstract"), address_type_abstraction
            )
        )

    def test_invalid_object_abstraction3(self):
        base_schema = {}
        base_schema["global"] = {
            "private_vars": [],
            "shared_vars": ["n"],
            "parent": None,
            "type": "global",
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        env.abstract_heap.add(
            {"0": [], "1": [], "2": []},
            Address("1", "abstract"),
        )
        address_type_abstraction = {"NUMBER": ["STRING"]}

        self.assertFalse(
            env.validate_address_abstraction(
                Address("1", "abstract"), address_type_abstraction
            )
        )

    def test_address_type_abstraction(self):
        base_schema = {}
        base_schema["global"] = {
            "private_vars": [],
            "shared_vars": ["n"],
            "parent": None,
            "type": "global",
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        env.abstract_heap.add(
            {
                "occupation": [baseType.STRING],
                "age": [baseType.NUMBER],
                "location": [baseType.STRING],
            },
            Address("1", "abstract"),
        )
        address_type_abstraction = {
            "occupation": ["STRING"],
            "age": ["NUMBER"],
            "location": ["TOP"],
        }
        self.assertTrue(
            env.validate_address_abstraction(
                Address("1", "abstract"), address_type_abstraction
            )
        )

    def test_join_addresses1(self):
        base_schema = {}
        base_schema["global"] = {
            "private_vars": [],
            "shared_vars": ["n"],
            "parent": None,
            "type": "global",
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        meta_info = {}
        meta_info["type"] = "object"
        meta_info["enumerable_values"] = OrderedSet(["occupation", "age", "location"])
        env.abstract_heap.add(
            {
                "occupation": [Primitive("doctor")],
                "age": [Primitive(30)],
                "location": [Primitive("Boston")],
                "__meta__": meta_info,
            },
            Address("1", "abstract"),
        )
        env.abstract_heap.add(
            {
                "occupation": [Primitive("doctor")],
                "age": [Primitive(30)],
                "location": [Primitive("New York")],
                "__meta__": meta_info,
            },
            Address("2", "abstract"),
        )
        join_result = env.join_addresses(
            Address("1", "abstract"), Address("2", "abstract"), add_null=False
        )
        self.assertTrue(join_result)

    def test_join_addresses2(self):
        base_schema = {}
        base_schema["global"] = {
            "private_vars": [],
            "shared_vars": ["n"],
            "parent": None,
            "type": "global",
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        meta_info = {}
        meta_info["type"] = "object"
        meta_info["enumerable_values"] = OrderedSet(["occupation", "age", "location"])
        env.abstract_heap.add(
            {
                "occupation": [Primitive("doctor")],
                "age": [Primitive(30)],
                "location": [Primitive("Boston")],
                "__meta__": meta_info,
            },
            Address("1", "abstract"),
        )
        env.abstract_heap.add(
            {
                "occupation": [Primitive("doctor")],
                "age": [Primitive(30)],
                "location": [Primitive("Boston")],
                "__meta__": meta_info,
            },
            Address("2", "abstract"),
        )
        join_result = env.join_addresses(
            Address("1", "abstract"), Address("2", "abstract"), add_null=False
        )
        self.assertFalse(join_result)

    def test_address_type_abstraction2(self):
        base_schema = {}
        base_schema["global"] = {
            "private_vars": [],
            "shared_vars": ["n"],
            "parent": None,
            "type": "global",
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        env.abstract_heap.add(
            {
                "occupation": [baseType.STRING],
                "age": [baseType.NUMBER],
                "location": [baseType.STRING],
            },
            Address("1", "abstract"),
        )
        address_type_abstraction = {
            "occupation": ["STRING"],
            "age": ["NUMBER"],
            "location": ["Boston"],
        }
        self.assertFalse(
            env.validate_address_abstraction(
                Address("1", "abstract"), address_type_abstraction
            )
        )

    def test_address_type_abstraction3(self):
        base_schema = {}
        base_schema["global"] = {
            "private_vars": [],
            "shared_vars": ["n"],
            "parent": None,
            "type": "global",
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        env.abstract_heap.add(
            {
                "occupation": [baseType.STRING, baseType.TOP],
                "age": [baseType.NUMBER, baseType.TOP],
                "location": [baseType.STRING, Primitive("Boston")],
            },
            Address("1", "abstract"),
        )
        address_type_abstraction = {
            "occupation": ["STRING"],
            "age": ["NUMBER"],
            "location": ["STRING"],
        }
        self.assertFalse(
            env.validate_address_abstraction(
                Address("1", "abstract"), address_type_abstraction
            )
        )

    def test_value_to_abstract_type(self):
        a = True
        self.assertTrue(value_to_abstract_type(a), baseType.BOOLEAN)

    def test_merge_functions(self):
        return
        base_schema = {}
        base_schema["global"] = {
            "private_vars": ["a", "b"],
            "shared_vars": ["n"],
            "parent": None,
            "type": "global",
        }
        body = {
            "type": "BlockStatement",
            "start": 15,
            "end": 30,
            "body": [
                {
                    "type": "ReturnStatement",
                    "start": 19,
                    "end": 28,
                    "argument": {
                        "type": "Literal",
                        "start": 26,
                        "end": 27,
                        "value": 5,
                        "raw": "5",
                    },
                }
            ],
        }
        env = Environment()
        env.initialize_from_schema(
            base_schema["global"],
            file_path="global",
            allocation_site="global",
            parent_address=None,
        )
        env.abstract_heap.add(
            {"__meta__": {"__parent__": None}}, Address("1", "abstract")
        )
        env.abstract_heap.add(
            {"__meta__": {"__parent__": Address("1", "abstract")}},
            Address("5", "abstract"),
        )
        env.abstract_heap.add(
            {"i": {0}, "__meta__": {"__parent__": Address("5", "abstract")}},
            Address("6", "abstract"),
        )
        env.abstract_heap.add(
            {
                "i": {baseType.NUMBER},
                "__meta__": {"__parent__": Address("5", "abstract")},
            },
            Address("7", "abstract"),
        )
        env.abstract_heap.add(
            {"x": {0}, "__meta__": {"__parent__": Address("6", "abstract")}},
            Address("8", "abstract"),
        )
        env.abstract_heap.add(
            {
                "x": {baseType.NUMBER},
                "__meta__": {"__parent__": Address("7", "abstract")},
            },
            Address("9", "abstract"),
        )
        prototype_address1 = env.add_object_to_heap({}, add_proto=False)
        function_object1 = env.concrete_heap.add(
            {
                "prototype": [prototype_address1],
                "__meta__": {
                    "enumerable_values": [],
                    "__parent__": Address("8", "abstract"),
                },
            }
        )
        function1 = Function(
            "",
            [],
            body,
            "global",
            value=function_object1,
            addr_type="concrete",
            prototype=prototype_address1,
            rest=None,
        )

        prototype_address2 = env.add_object_to_heap({}, add_proto=False)
        function_object2 = env.concrete_heap.add(
            {
                "prototype": [prototype_address2],
                "__meta__": {
                    "enumerable_values": [],
                    "__parent__": Address("9", "abstract"),
                },
            }
        )
        function2 = Function(
            "",
            [],
            body,
            "global",
            value=function_object2,
            addr_type="concrete",
            prototype=prototype_address2,
            rest=None,
        )
        # abstraction = {"n": ["NUMBER"]}
        # second_abstraction = {"n": ["TOP"]}
        # env.merge_abstractions(abstraction)
        # env.merge_abstractions(second_abstraction)
        # self.assertTrue(env.lookup_and_derive("n"), ["NUMBER"])

    def test_object_superset1(self):
        obj = {0: {Primitive("foo")}, 1: {Primitive("foo")}, 2: {Primitive("foo")}}
        abstraction = {
            "0": ["foo"],
            "1": ["foo"],
            "2": ["foo"],
        }  # Sometimes LLM will return strings as keys instead of integers
        self.assertTrue(object_is_superset(obj, abstraction))

    def test_object_superset2(self):
        obj = {0: {Primitive("foo")}, 1: {Primitive("foo")}, 2: {Primitive("foo")}}
        abstraction = {0: ["STRING"], 1: ["STRING"], 2: ["STRING"]}
        self.assertTrue(object_is_superset(obj, abstraction))

    def test_object_superset3(self):
        obj = {
            "key1": [1],
            "key2": [2],
            "foo": [3],
            "bar": [4],
            "nested": [{"hello": ["world"]}],
        }
        obj2 = {"STRING": ["NUMBER", {"STRING": ["NULL", "STRING"]}]}
        self.assertTrue(object_is_superset(obj, obj2))


class testMemoization(unittest.TestCase):
    def setUp(self):
        self.testLLM = False
        self.base_class = "js_files/memoization"

    def test_memoized_calls(self):
        check = AbsIntAI.AbsIntAI()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "memoized_call.js"
        )
        env = check.run(scriptPath)
        self.assertTrue(is_superset([5], env.lookup_and_derive("firstCall")))

    def test_return_objects(self):
        check = AbsIntAI.AbsIntAI()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "return_objects.js"
        )
        env = check.run(scriptPath)
        self.assertTrue(is_superset([{"a": [3], "b": [2]}], env.lookup_and_derive("x")))
        self.assertTrue(is_superset([{"a": [1], "b": [4]}], env.lookup_and_derive("y")))

    def test_return_objects2(self):
        check = AbsIntAI.AbsIntAI()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "return_objects2.js"
        )
        env = check.run(scriptPath)
        self.assertTrue(
            is_superset(
                [{"a": [{"x": [1, 3], "y": [4]}], "b": [2]}], env.lookup_and_derive("x")
            )
        )
        self.assertTrue(
            is_superset(
                [{"a": [{"x": [1, 3], "y": [4]}], "b": [5]}], env.lookup_and_derive("y")
            )
        )


# All the things we need to support to run on real javascript and are annoying.
class testJavascriptFeatures(unittest.TestCase):
    def setUp(self):
        self.testLLM = False
        self.base_class = "js_files/real_world_features"

    def test_iife(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(os.path.dirname(__file__), self.base_class, "iife.js")
        env = check.run(scriptPath)
        self.assertEqual([{"x": [5]}], env.lookup_and_derive("g"))

    def test_iife_field(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "iife_field.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([5], env.lookup_and_derive("blah"))

    def test_function_prototype_call(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "function_prototype_call.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([{"foo": ["blah"]}], env.lookup_and_derive("blah"))

    def test_function_fields(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "function_fields.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([2], env.lookup_and_derive("result"))

    def test_function_prototype_assignment(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__),
            self.base_class,
            "function_prototype_assignment.js",
        )
        env = check.run(scriptPath)
        self.assertEqual([{"field": ["bar"]}], env.lookup_and_derive("test"))

    def test_for_each(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "for_each.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(
            [{"pop": [0], "push": [1], "reverse": [2]}],
            env.lookup_and_derive("obj"),
        )

    def test_imports(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(os.path.dirname(__file__), self.base_class, "base.js")
        env = check.run(scriptPath)
        logger.info(env.lookup_and_derive("x"))
        self.assertEqual(["foo"], env.lookup_and_derive("b"))
        self.assertEqual([1], env.lookup_and_derive("x"))

    def test_require(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "require_base.cjs"
        )
        env = check.run(scriptPath)
        self.assertEqual([10], env.lookup_and_derive("a"))
        self.assertEqual([5], env.lookup_and_derive("b"))

    def test_this_prototype(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "this_prototype.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["bar"], env.lookup_and_derive("result"))

    def test_hashmap_prototype(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "hashmap_prototype.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([42, "NULL"], env.lookup_and_derive("res"))

    def test_destructuring(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "destructuring.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([10], env.lookup_and_derive("a"))
        self.assertEqual([20], env.lookup_and_derive("b"))
        self.assertEqual([10], env.lookup_and_derive("c"))
        self.assertEqual([30], env.lookup_and_derive("d"))
        self.assertEqual([1], env.lookup_and_derive("y"))
        self.assertEqual([2], env.lookup_and_derive("z"))
        self.assertEqual([1], env.lookup_and_derive("key1"))
        self.assertEqual([2], env.lookup_and_derive("key2"))

    def test_arguments_array(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "arguments_array.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([2], env.lookup_and_derive("blah"))

    def test_multiple_assignments(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "multiple_assignments.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([10], env.lookup_and_derive("a"))
        self.assertEqual([10], env.lookup_and_derive("b"))
        self.assertEqual([10], env.lookup_and_derive("c"))
        self.assertEqual([10], env.lookup_and_derive("d"))
        self.assertEqual([{"foo": ["bar"]}], env.lookup_and_derive("e"))
        self.assertEqual([{"foo": ["bar"]}], env.lookup_and_derive("f"))
        self.assertEqual([{"foo": ["bar"]}], env.lookup_and_derive("g"))
        self.assertEqual([{"foo": [5]}], env.lookup_and_derive("blah1"))
        self.assertEqual([{"foo": [5]}], env.lookup_and_derive("blah2"))
        self.assertEqual([5], env.lookup_and_derive("blah3"))

    def test_prototype_inheritance(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "prototype_inheritance.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([3], env.lookup_and_derive("result"))

    def test_prototype_inheritance2(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "prototype_inheritance2.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(
            [{"name": ["White Rabbit"], "isSleeping": [True]}],
            env.lookup_and_derive("rabbit"),
        )

    def test_prototype_inheritance3(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "prototype_inheritance3.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["Thom"], env.lookup_and_derive("personName"))
        self.assertEqual(["Thom"], env.lookup_and_derive("singerName"))

    def test_function_objects(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "function_objects.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["outer result"], env.lookup_and_derive("outer"))
        self.assertEqual(["inner result"], env.lookup_and_derive("inner"))

    def test_class_inheritance(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(os.path.dirname(__file__), self.base_class, "new.js")
        env = check.run(scriptPath)
        self.assertEqual(["bar"], env.lookup_and_derive("c"))
        self.assertEqual(["blah"], env.lookup_and_derive("b"))

    def test_class_object(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "class_object.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["blah"], env.lookup_and_derive("b"))

    def test_this_function(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "this_function.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["bar"], env.lookup_and_derive("b"))
        self.assertEqual(["upper"], env.lookup_and_derive("c"))

    def test_this_function_var(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "this_function_var.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["bar"], env.lookup_and_derive("b"))
        self.assertEqual(["upper"], env.lookup_and_derive("c"))

    def test_basic_dynamic_call(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "dynamic_calls.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(["foo"], env.lookup_and_derive("res"))

    def test_basic_dynamic_addition(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "dynamic_field_addition.js"
        )
        env = check.run(scriptPath)
        self.assertEqual([{"foo": ["bar"]}], env.lookup_and_derive("obj"))

    def test_import_const(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__),
            self.base_class,
            "module_testing",
            "import_const.js",
        )
        env = check.run(scriptPath)
        logger.info(env.lookup_and_derive("CHAR_UPPERCASE_A"))
        self.assertEqual([65], env.lookup_and_derive("CHAR_UPPERCASE_A"))
        self.assertEqual([97], env.lookup_and_derive("CHAR_LOWERCASE_A"))

    def test_nested_destructuring(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__),
            self.base_class,
            "nested_destructuring.js",
        )
        env = check.run(scriptPath)
        self.assertEqual(["foo"], env.lookup_and_derive("simple"))
        self.assertEqual([100], env.lookup_and_derive("width"))
        self.assertEqual([200], env.lookup_and_derive("height"))
        self.assertEqual(["Cake"], env.lookup_and_derive("item1"))
        self.assertEqual(["Donut"], env.lookup_and_derive("item2"))
        self.assertEqual([1], env.lookup_and_derive("nested1"))
        self.assertEqual([2], env.lookup_and_derive("nested2"))
        self.assertEqual([3], env.lookup_and_derive("extra_nested1"))

    def test_import_self_reference(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__),
            self.base_class,
            "module_testing",
            "self_reference_base.js",
        )
        env = check.run(scriptPath)
        self.assertEqual(["bar"], env.lookup_and_derive("result"))

    def test_function_rest(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "function_rest.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(
            [{"arg1": [1], "arg2": [2], "rest": [{0: [3], 1: [4]}]}],
            env.lookup_and_derive("result"),
        )

    def test_require_default_lib(self):
        return
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "require_default_lib.js"
        )
        env = check.run(scriptPath)


class testBuiltins(unittest.TestCase):
    def setUp(self):
        self.testLLM = False
        self.base_class = "js_files/builtins"

    def test_array_push(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__), self.base_class, "arrays", "array_push.js"
        )
        env = check.run(scriptPath)
        self.assertEqual(
            [{0: [1], 1: [2], 2: [3], 3: [4], 4: [5], 5: [6], 6: [7]}],
            env.lookup_and_derive("a"),
        )

    def test_document_get_element_by_id(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__),
            self.base_class,
            "document",
            "document_get_element_by_id.js",
        )
        env = check.run(scriptPath)
        logger.info(env.lookup_and_derive("test"))
        # self.assertEqual(["STRING"], env.lookup_and_derive("test"))

    def test_event_listener(self):
        check = AbsIntAI.AbsIntAI()
        check.set_test_mode()
        scriptPath = os.path.join(
            os.path.dirname(__file__),
            self.base_class,
            "document",
            "event_listener.js",
        )
        env = check.run(scriptPath)
        logger.info(f"test is {env.lookup_and_derive('global_obj')}")
        # self.assertEqual([{"clicked": [True]}], env.lookup_and_derive("global_obj"))


if __name__ == "__main__":
    unittest.main()
