import os
from PIL import Image

def resize_images_in_folder(input_folder, output_folder, target_size):
    """
    调整指定文件夹内所有图片的尺寸并保存到新文件夹。

    :param input_folder: 包含原始图片的文件夹路径。
    :param output_folder: 用于保存调整后图片的新文件夹路径。
    :param target_size: 一个元组，表示目标尺寸 (宽度, 高度)。
    """
    # 检查输入文件夹是否存在
    if not os.path.isdir(input_folder):
        print(f"错误：文件夹 '{input_folder}' 不存在。")
        return

    # 如果输出文件夹不存在，则创建它
    if not os.path.exists(output_folder):
        print(f"创建输出文件夹: {output_folder}")
        os.makedirs(output_folder)

    # 支持的图片文件扩展名
    supported_formats = ('.png', '.jpg', '.jpeg', '.bmp', '.gif')
    
    # 获取文件夹内所有文件名
    files = os.listdir(input_folder)
    
    print(f"\n开始处理 {len(files)} 个文件...")

    for filename in files:
        # 检查文件是否为支持的图片格式
        if filename.lower().endswith(supported_formats):
            try:
                # 构建完整的文件路径
                input_path = os.path.join(input_folder, filename)
                output_path = os.path.join(output_folder, filename)

                # 打开图片
                with Image.open(input_path) as img:
                    # 检查原始尺寸是否为 720x1280 (宽度x高度)
                    if img.size == (1280,720):
                        print(f"正在处理: {filename}，尺寸: {img.size} -> {target_size}")
                        # 调整图片尺寸
                        # 注意：ANTIALIAS 在新版本Pillow中建议用LANCZOS
                        resized_img = img.resize(target_size, Image.Resampling.LANCZOS)
                        # 保存调整后的图片
                        resized_img.save(output_path)
                    else:
                        print(f"跳过: {filename}，尺寸 {img.size} 不符合 (720, 1280)。")
            except Exception as e:
                print(f"处理文件 {filename} 时发生错误: {e}")

    print("\n所有符合条件的图片处理完成！")
    print(f"调整后的图片已保存至: {output_folder}")


if __name__ == "__main__":
    # --- 用户配置 ---
    # 目标尺寸 (宽度, 高度)
    TARGET_WIDTH = 1212
    TARGET_HEIGHT = 677
    new_size = (TARGET_WIDTH, TARGET_HEIGHT)

    # --- 运行脚本 ---
    # 从用户处获取输入文件夹路径
    input_folder_path = input("请输入包含图片的文件夹路径: ")
    
    # 自动设置输出文件夹路径（在输入文件夹的父目录下）
    output_folder_path = os.path.join(os.path.dirname(input_folder_path), "resized_images")
    
    # 执行主函数
    resize_images_in_folder(input_folder_path, output_folder_path, new_size)