from scene.cameras import Camera
import numpy as np
from utils.graphics_utils import fov2focal
from PIL import Image
import cv2

WARNED = False

def loadCam(args, id, cam_info, resolution_scale):
    #调整分辨率，打包信息，创建camera类
    image = Image.open(cam_info.image_path)
    orig_w, orig_h = image.size
    if args.resolution in [1, 2, 4, 8]:
        resolution = round(orig_w/(resolution_scale * args.resolution)), round(orig_h/(resolution_scale * args.resolution))
    else:  # should be a type that converts to float
        if args.resolution == -1:
            if orig_w > 1600:
                global WARNED
                if not WARNED:
                    print("[ INFO ] Encountered quite large input images (>1.6K pixels width), rescaling to 1.6K.\n "
                        "If this is not desired, please explicitly specify '--resolution/-r' as 1")
                    WARNED = True
                global_down = orig_w / 1600
            else:
                global_down = 1
        else:
            global_down = orig_w / args.resolution
    

        scale = float(global_down) * float(resolution_scale)
        resolution = (int(orig_w / scale), int(orig_h / scale))
    # 创建 Camera 类的实例，完成了近平面，远平面，投影矩阵的计算等操作
    return Camera(
        # 图像的分辨率，是一个包含宽度和高度的元组
        resolution,
        # 相机的 colmap ID，用于唯一标识相机
        colmap_id=cam_info.uid,
        # 相机的旋转矩阵，描述相机的朝向
        R=cam_info.R,
        # 相机的平移向量，描述相机的位置
        T=cam_info.T,
        # 相机视场角
        FoVx=cam_info.FovX,FoVy=cam_info.FovY,cx=cam_info.cx/args.resolution, cy=cam_info.cy/args.resolution,
        # 相机拍摄的图像
        image=image,
        # 图像的名称
        image_name=cam_info.image_name,
        # 相机的唯一标识符
        uid=id,
        # 数据所在的设备
        data_device=args.data_device
    )

def cameraList_from_camInfos(cam_infos, resolution_scale, args):
    camera_list = []

    for id, c in enumerate(cam_infos):
        camera_list.append(loadCam(args, id, c, resolution_scale))

    return camera_list

def camera_to_JSON(id, camera : Camera):
    Rt = np.zeros((4, 4))
    Rt[:3, :3] = camera.R.transpose()
    Rt[:3, 3] = camera.T
    Rt[3, 3] = 1.0

    W2C = np.linalg.inv(Rt)
    pos = W2C[:3, 3]
    rot = W2C[:3, :3]
    serializable_array_2d = [x.tolist() for x in rot]
    camera_entry = {
        'id' : id,
        'img_name' : camera.image_name,
        'width' : camera.width,
        'height' : camera.height,
        'position': pos.tolist(),
        'rotation': serializable_array_2d,
        'fy' : fov2focal(camera.FovY, camera.height),
        'fx' : fov2focal(camera.FovX, camera.width)
    }
    return camera_entry