import numpy as np
from PIL import Image

def calculate_metrics(mask1_path, mask2_path):
    """
    计算两个掩码图片之间的MIoU和准确率
    
    参数:
        mask1_path: 第一个掩码图片的路径
        mask2_path: 第二个掩码图片的路径
    
    返回:
        miou: 平均交并比
        accuracy: 准确率
    """
    # 读取掩码图片
    mask1 = Image.open(mask1_path)
    mask2 = Image.open(mask2_path)
    
    # 如果尺寸不同，将mask2调整为mask1的尺寸
    if mask1.size != mask2.size:
        print(f"警告：两个掩码尺寸不同，正在调整尺寸...")
        print(f"原始尺寸 - mask1: {mask1.size}, mask2: {mask2.size}")
        mask2 = mask2.resize(mask1.size, Image.NEAREST)
        print(f"调整后尺寸 - mask1: {mask1.size}, mask2: {mask2.size}")
    
    # 转换为numpy数组
    mask1 = np.array(mask1)
    mask2 = np.array(mask2)
    
    # 将图片二值化（如果不是二值图像）
    mask1 = (mask1 > 0).astype(np.uint8)
    mask2 = (mask2 > 0).astype(np.uint8)
    
    # 计算交集和并集
    intersection = np.logical_and(mask1, mask2).sum()
    union = np.logical_or(mask1, mask2).sum()
    
    # 计算IoU
    iou = intersection / (union + 1e-10)  # 添加小量避免除零
    
    # 计算准确率
    accuracy = (mask1 == mask2).mean()
    
    return iou, accuracy

if __name__ == "__main__":
    import argparse
    
    parser = argparse.ArgumentParser(description='计算两个掩码图片之间的MIoU和准确率')
    parser.add_argument('mask1_path', type=str, help='第一个掩码图片的路径')
    parser.add_argument('mask2_path', type=str, help='第二个掩码图片的路径')
    
    args = parser.parse_args()
    
    iou, acc = calculate_metrics(args.mask1_path, args.mask2_path)
    print(f"MIoU: {iou:.4f}")
    print(f"准确率: {acc:.4f}")
