

import os
import os.path
import shutil
import json

# Function that removes a file or directory in case it exists
# Sources: https://stackoverflow.com/questions/10840533/most-pythonic-way-to-delete-a-file-which-may-not-exist
#          https://stackoverflow.com/questions/6996603/how-can-i-delete-a-file-or-folder-in-python
def ensure_file_removed(path):
    # File
    if os.path.isfile(path) or os.path.islink(path):
        try:
            os.remove(path)
        except OSError as e: # this would be "except OSError, e:" before Python 2.6
            if e.errno != errno.ENOENT: # errno.ENOENT = no such file or directory
                raise # re-raise exception if a different error occurred
    # Folder
    elif os.path.isdir(path):
        shutil.rmtree(path) # Remove folder and contents


def ensure_directory_pathname(path):
    if path != '' and path[-1] != '/':
        return path + '/'
    else:
        return path


def append_to_name(path, string):
    root, ext = os.path.splitext(path)
    return root + string + ext


def basename_noext(path):
    return os.path.splitext(os.path.basename(path))[0]


def save_json(data:dict, path:str):
    os.makedirs(os.path.dirname(path), exist_ok=True)
    with open(path, "w+") as f:
        json.dump(data, f, indent=2)


def target_required(path, *others, force=False, verbose=True, dominated=None):

    def not_required():
        if dominated is not None:
            return os.path.isfile(path) and dominated(path)
        else:
            return os.path.isfile(path)

    if verbose:
        print(f"> Checking if the target requires regeneration: {path}")
    if not_required():
        if not force:
            if verbose:
                print(f"> Target exists and dominated, skipping: {path}")
            return False

        else:
            if verbose:
                print(f"> Target exists and dominated, but forcing regeneration: {path}")
            for _path in [path] + list(others):
                ensure_file_removed(_path)
            return True
    else:
        for _path in others:
            ensure_file_removed(_path)
        return True


def idcache(fn):
    """
    Custom decorator for caching functions.
    It stores the cache based on the built-in object id of the arguments.
    I'm not using functool.cache because Namespace is not hashable (which is stupid) and
    I didn't want to import frozendict just for this
    """
    cache = dict()
    def fn2(*args):
        key = tuple(id(arg) for arg in args)
        if key not in cache:
            cache[key] = fn(*args)
        return cache[key]

    fn2.fn = fn
    fn2.cache = cache

    return fn2

