from abc import ABC, abstractmethod
from string import ascii_uppercase
from typing import Any, Callable, Mapping, Optional, Union


class Question(ABC):
    text: str

    @abstractmethod
    def is_correct(self, answer: Any) -> bool:
        ...

    def __repr__(self):
        return f'{self.__class__.__name__}: {self.text}'


class MultipleChoiceQuestion(Question):
    option_chars = ascii_uppercase

    def __init__(self, text: str, options: Mapping[str, str], label: Optional[str] = None) -> None:
        if label is not None and label not in options:
            raise ValueError('Label must be in options')
        self.question_text = text
        self.options = options
        self.label = label

    @property
    def text(self):
        return self.question_text + '\n' + '\n'.join(f'{c}: {a}' for c, a in self.options.items())

    def is_correct(self, answer: str) -> bool:
        if self.label is None:
            raise AttributeError('Label is None')
        return answer.lower() == self.label.lower().strip()


class YesNoQuestion(MultipleChoiceQuestion):
    option_chars = ['Yes', 'No']

    def __init__(self, text: str, label: Optional[str] = None) -> None:
        if label is not None and label not in self.option_chars:
            raise ValueError('Label must be in options')
        self.question_text = text
        self.options = {o: o for o in self.option_chars}
        self.label = label

    @property
    def text(self):
        return self.question_text

    def is_correct(self, answer: Union[str, float, bool]) -> bool:
        if self.label is None:
            raise AttributeError('Label is None')
        if answer is None:
            return False
        if isinstance(answer, float):
            answer = 'Yes' if answer > 0.5 else 'No'
        elif isinstance(answer, bool):
            answer = 'Yes' if answer else 'No'
        return answer.lower() == self.label.lower().strip()


class ScalarQuestion(Question):
    def __init__(
            self,
            text: str,
            label: Optional[float] = None,
            is_correct_fn: Optional[Callable[[Union[float, None], float], bool]] = None
            ) -> None:
        self.text = text
        self.label = label
        self.is_correct_fn = is_correct_fn

    def is_correct(self, answer: float) -> bool:
        assert self.label is not None and self.is_correct_fn is not None
        return self.is_correct_fn(answer, self.label)


class ExactMatchQuestion(Question):
    def __init__(self, text: str, label: str) -> None:
        self.text = text
        self.label = label

    def is_correct(self, answer: str) -> bool:
        assert self.label is not None
        return answer == self.label


class FreeResponseQuestion(Question):
    def __init__(
            self,
            text: str,
            reference_answer: Optional[str] = None,
            is_correct_fn: Optional[Callable[[str, Optional[str]], bool]] = None
            ) -> None:
        self.text = text
        self.reference_answer = reference_answer
        self.is_correct_fn = is_correct_fn

    def is_correct(self, answer: str) -> bool:
        if self.is_correct_fn is None:
            return True
        return self.is_correct_fn(answer, self.reference_answer)

