import json
import subprocess
from dataclasses import dataclass
from pathlib import Path
from typing import Callable, Dict, List, Union

import numpy as np
import pandas as pd

from lm_understanding.question_template import TemplateModelBehavior

BASELINES = ['predict_average', 'nearest_neighbor', 'nearest_neighbor_three', 'linear_regression']

test_functions = []

def test_function(func):
    test_functions.append(func)
    return func

class FailedTest(Exception):
    pass

@dataclass
class Constraint:
    baseline: str
    passes_constraint: Callable[[float], bool]


@dataclass
class Test:
    name: str
    constraints: List[Constraint]
    model_behavior: List[TemplateModelBehavior]

    @property
    def model_behavior_path(self):
        return Path(f'model_behavior_results/baseline_tests/{self.name}/new_combinations/final_behavior/')

    def baseline_results_path(self, baseline: str) -> Path:
        return Path(f'baseline_results/baseline_tests/{self.name}/new_combinations/{baseline}.json')

    def baseline_results(self, baseline: str) -> Dict[str, float]:
        return json.loads(self.baseline_results_path(baseline).read_text()) ['results']

    @property
    def baselines(self) -> List[str]:
        return [c.baseline for c in self.constraints]

    def create_config(self):
        path = Path(f'config/task/{self.name}.yaml')
        path.write_text(f'name: {self.name}')

    def run_baselines(self) -> None:
        for baseline in self.baselines:
            if self.baseline_results_path(baseline).exists():
                continue
            cmd = [
                'python',
                'run_baseline.py',
                f'baseline={baseline}',
                'model=baseline_tests',
                f'task={self.name}',
                'dataset_construction=base'
            ]
            subprocess.run(cmd)

    def check_constraint(self, constraint: Constraint) -> None:
        results = self.baseline_results(constraint.baseline)
        tv_distance = results['tv_distance']
        if not constraint.passes_constraint(tv_distance):
            raise FailedTest(f'{self.name}, {constraint.baseline}, tv_distance={tv_distance}')
        print('Test Passed:', self.name, constraint.baseline)

    def check_constraints(self):
        for constraint in self.constraints:
            self.check_constraint(constraint)

    def run_tests(self):
        self.create_config()
        for i, model_behavior in enumerate(self.model_behavior, start=1):
            model_behavior.save(self.model_behavior_path / f'test_set_{i}')
        self.run_baselines()
        self.check_constraints()


def existing_model_behavior() -> TemplateModelBehavior:
    return TemplateModelBehavior.load(Path('model_behavior_results/synthetic_linear_model/moral_dilemmas/new_combinations/final_behavior/moral_dilemmas_35'))


@test_function
def test_same_answer():
    mb = existing_model_behavior()
    mb.df['answer_prob'] = 0.2
    constraints = [
        Constraint('predict_average', lambda tv: tv == 0),
        Constraint('nearest_neighbor', lambda tv: tv == 0),
        Constraint('nearest_neighbor_three', lambda tv: tv <= 0.0001),
        Constraint('linear_regression', lambda tv: tv <= 0.01),
    ]
    test = Test('test_same_answer', constraints, [mb])
    test.run_tests()


@test_function
def test_answer_is_average():
    mb = existing_model_behavior()
    test_questions = mb.df.split == 'test'
    train_average = mb.df[~test_questions].answer_prob.to_numpy().mean()
    mb.df.loc[test_questions, 'answer_prob'] = train_average
    constraints = [
        Constraint('predict_average', lambda tv: tv <= .01),
        Constraint('nearest_neighbor', lambda tv: tv >= 0.1),
    ]
    test = Test('test_answer_is_average', constraints, [mb])
    test.run_tests()


@test_function
def test_answer_is_average_offset():
    mb = existing_model_behavior()
    test_questions = mb.df.split == 'test'
    train_average = mb.df[~test_questions].answer_prob.to_numpy().mean() + 0.2
    mb.df.loc[test_questions, 'answer_prob'] = train_average
    constraints = [
        Constraint('predict_average', lambda tv: (tv >= 0.19) and (tv <= .21)),
        Constraint('nearest_neighbor', lambda tv: tv >= 0.1),
    ]
    test = Test('test_answer_is_average_offset', constraints, [mb])
    test.run_tests()


@test_function
def test_answer_in_train():
    mb = existing_model_behavior()
    test_questions = mb.df.split == 'test'
    train = mb.df[~test_questions].copy()
    test = mb.df[~test_questions].copy()
    test['split'] = 'test'
    mb.df = pd.concat([train, test])
    constraints = [
        Constraint('predict_average', lambda tv: tv >= .1),
        Constraint('nearest_neighbor', lambda tv: tv <= 0.001),
    ]
    test = Test('test_answer_in_train', constraints, [mb])
    test.run_tests()

@test_function
def test_offset():
    mb = existing_model_behavior()
    test_questions = mb.df.split == 'test'
    train = mb.df[~test_questions].copy()
    test = mb.df[~test_questions].copy()
    test['split'] = 'test'
    test['answer_prob'] = test.answer_prob.to_numpy() + 0.2
    mb.df = pd.concat([train, test])
    constraints = [
        Constraint('nearest_neighbor', lambda tv: (tv >= 0.19) and (tv <= .21)),
    ]
    test = Test('test_offset', constraints, [mb])
    test.run_tests()

@test_function
def test_misleading():
    mb = existing_model_behavior()
    test_questions = mb.df.split == 'test'
    train = mb.df[~test_questions].copy()
    test = mb.df[~test_questions].copy()
    test['split'] = 'test'
    test['answer_prob'] = 1
    mb.df = pd.concat([train, test])
    constraints = [
        Constraint('predict_average', lambda tv: tv >= .3),
        Constraint('nearest_neighbor', lambda tv: tv >= 0.3),
        Constraint('linear_regression', lambda tv: tv >= 0.3),
    ]
    test = Test('test_misleading', constraints, [mb])
    test.run_tests()


@test_function
def test_multiple():
    mb = existing_model_behavior()
    test_questions = mb.df.split == 'test'
    train = mb.df[~test_questions].copy()
    test = mb.df[~test_questions].copy()
    test['split'] = 'test'
    test['answer_prob'] = test.answer_prob.to_numpy() + 0.2
    mb.df = pd.concat([train, test])
    mb2 = existing_model_behavior()
    test_questions = mb2.df.split == 'test'
    train = mb2.df[~test_questions].copy()
    test = mb2.df[~test_questions].copy()
    test['split'] = 'test'
    test['answer_prob'] = test.answer_prob.to_numpy()
    mb2.df = pd.concat([train, test])

    constraints = [
        Constraint('nearest_neighbor', lambda tv: (tv >= 0.09) and (tv <= .11)),
    ]
    test = Test('test_multiple', constraints, [mb, mb2])
    test.run_tests()


def main():
    for test in test_functions:
        test()


if __name__ == '__main__':
    main()
