import tensorflow as tf
from tensorflow.keras import layers, models, activations

__all__ = ['dense4cnn', 'dense6cnn', 'dense10cnn', 'lenet5']

class Dense4CNN(tf.keras.Model):
    def __init__(self, num_classes=10):
        super(Dense4CNN, self).__init__()
        self.conv1 = layers.Conv2D(64, kernel_size=3, strides=1, padding='same', use_bias=False)
        self.conv2 = layers.Conv2D(64, kernel_size=3, strides=1, padding='same', use_bias=False)
        self.conv3 = layers.Conv2D(128, kernel_size=3, strides=1, padding='same', use_bias=False)
        self.conv4 = layers.Conv2D(128, kernel_size=3, strides=1, padding='same', use_bias=False)

        self.flatten = layers.Flatten()
        self.dense1 = layers.Dense(256)
        self.dense2 = layers.Dense(256)
        self.dense3 = layers.Dense(num_classes)
        self.softmax = layers.Activation('softmax')

    def call(self, x):
        x = activations.relu(self.conv1(x))
        x = layers.MaxPooling2D(pool_size=2, strides=2)(activations.relu(self.conv2(x)))
        x = activations.relu(self.conv3(x))
        x = layers.MaxPooling2D(pool_size=2, strides=2)(activations.relu(self.conv4(x)))

        x = self.flatten(x)
        x = activations.relu(self.dense1(x))
        x = activations.relu(self.dense2(x))
        x = self.dense3(x)
        return self.softmax(x)

class Dense6CNN(tf.keras.Model):
    def __init__(self, num_classes=10):
        super(Dense6CNN, self).__init__()
        self.conv1 = layers.Conv2D(64, kernel_size=3, strides=1, padding='same')
        self.conv2 = layers.Conv2D(64, kernel_size=3, strides=1, padding='same')
        self.conv3 = layers.Conv2D(128, kernel_size=3, strides=1, padding='same')
        self.conv4 = layers.Conv2D(128, kernel_size=3, strides=1, padding='same')
        self.conv5 = layers.Conv2D(256, kernel_size=3, strides=1, padding='same')
        self.conv6 = layers.Conv2D(256, kernel_size=3, strides=1, padding='same')

        self.flatten = layers.Flatten()
        self.dense1 = layers.Dense(256)
        self.dense2 = layers.Dense(256)
        self.dense3 = layers.Dense(num_classes)
        self.softmax = layers.Activation('softmax')

    def call(self, x):
        x = activations.relu(self.conv1(x))
        x = layers.MaxPooling2D(pool_size=2, strides=2)(activations.relu(self.conv2(x)))
        x = activations.relu(self.conv3(x))
        x = layers.MaxPooling2D(pool_size=2, strides=2)(activations.relu(self.conv4(x)))
        x = activations.relu(self.conv5(x))
        x = layers.MaxPooling2D(pool_size=2, strides=2)(activations.relu(self.conv6(x)))

        x = self.flatten(x)
        x = activations.relu(self.dense1(x))
        x = activations.relu(self.dense2(x))
        x = self.dense3(x)
        return self.softmax(x)

class Dense10CNN(tf.keras.Model):
    def __init__(self, num_classes=10):
        super(Dense10CNN, self).__init__()
        self.conv1 = layers.Conv2D(64, kernel_size=3, strides=1, padding='same')
        self.conv2 = layers.Conv2D(64, kernel_size=3, strides=1, padding='same')
        self.conv3 = layers.Conv2D(128, kernel_size=3, strides=1, padding='same')
        self.conv4 = layers.Conv2D(128, kernel_size=3, strides=1, padding='same')
        self.conv5 = layers.Conv2D(256, kernel_size=3, strides=1, padding='same')
        self.conv6 = layers.Conv2D(256, kernel_size=3, strides=1, padding='same')
        self.conv7 = layers.Conv2D(512, kernel_size=3, strides=1, padding='same')
        self.conv8 = layers.Conv2D(512, kernel_size=3, strides=1, padding='same')
        self.conv9 = layers.Conv2D(1024, kernel_size=3, strides=1, padding='same')
        self.conv10 = layers.Conv2D(1024, kernel_size=3, strides=1, padding='same')

        self.flatten = layers.Flatten()
        self.dense1 = layers.Dense(256)
        self.dense2 = layers.Dense(256)
        self.dense3 = layers.Dense(num_classes)
        self.softmax = layers.Activation('softmax')

    def call(self, x):
        x = activations.relu(self.conv1(x))
        x = layers.MaxPooling2D(pool_size=2, strides=2)(activations.relu(self.conv2(x)))
        x = activations.relu(self.conv3(x))
        x = layers.MaxPooling2D(pool_size=2, strides=2)(activations.relu(self.conv4(x)))
        x = activations.relu(self.conv5(x))
        x = layers.MaxPooling2D(pool_size=2, strides=2)(activations.relu(self.conv6(x)))
        x = activations.relu(self.conv7(x))
        x = layers.MaxPooling2D(pool_size=2, strides=2)(activations.relu(self.conv8(x)))
        x = activations.relu(self.conv9(x))
        x = layers.MaxPooling2D(pool_size=2, strides=2)(activations.relu(self.conv10(x)))

        x = self.flatten(x)
        x = activations.relu(self.dense1(x))
        x = activations.relu(self.dense2(x))
        x = self.dense3(x)
        return self.softmax(x)

class LeNet5(tf.keras.Model):
    def __init__(self, num_classes=10):
        super(LeNet5, self).__init__()
        self.conv1 = layers.Conv2D(6, kernel_size=5, padding='same')
        self.pool1 = layers.AvgPool2D(pool_size=2, strides=2)
        self.conv2 = layers.Conv2D(16, kernel_size=5)
        self.pool2 = layers.AvgPool2D(pool_size=2, strides=2)

        self.flatten = layers.Flatten()
        self.fc1 = layers.Dense(120)
        self.fc2 = layers.Dense(84)
        self.fc3 = layers.Dense(num_classes)
        self.softmax = layers.Activation('softmax')

    def call(self, x):
        x = activations.relu(self.conv1(x))
        x = self.pool1(x)
        x = activations.relu(self.conv2(x))
        x = self.pool2(x)
        x = self.flatten(x)
        x = activations.relu(self.fc1(x))
        x = activations.relu(self.fc2(x))
        x = self.fc3(x)
        return self.softmax(x)

def dense4cnn(num_classes=10):
    return Dense4CNN(num_classes=num_classes)

def dense6cnn(num_classes=10):
    return Dense6CNN(num_classes=num_classes)

def dense10cnn(num_classes=10):
    return Dense10CNN(num_classes=num_classes)

def lenet5(num_classes=10):
    return LeNet5(num_classes=num_classes)
