# This training script is adapted from the TRL library: https://github.com/huggingface/trl/

# flake8: noqa
# Copyright 2023 The HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""
python GPT2_large_DPO_tldr.py \
    --model_name_or_path="vistagi/gpt2-large-tldr-sum" \
    --per_device_train_batch_size 2 \
    --learning_rate 5e-5 \
    --gradient_accumulation_steps 16 \
    --logging_steps 10 \
    --output_dir="GPT2_large_tldr_DPO" \
    --report_to wandb \
    --fp16 \
    --logging_first_step \
    --no_remove_unused_columns
    --use_peft 
    --lora_r=16 
    --lora_alpha=16 
    --max_length=512 
    --max_prompt_length=512
    --optim="rmsprop"
"""

import logging
import multiprocessing
import os
from contextlib import nullcontext
import wandb
import numpy as np
from datasets import load_dataset, load_from_disk, Dataset # huggingface datasets

TRL_USE_RICH = os.environ.get("TRL_USE_RICH", False)

from trl.commands.cli_utils import DPOScriptArguments, init_zero_verbose, TrlParser

if TRL_USE_RICH:
    init_zero_verbose()
    FORMAT = "%(message)s"

    from rich.console import Console
    from rich.logging import RichHandler

import torch
from datasets import load_dataset
from transformers import AutoModelForCausalLM, AutoTokenizer

from trl import DPOConfig, DPOTrainer, ModelConfig, RichProgressCallback, get_kbit_device_map, get_peft_config, get_quantization_config


if TRL_USE_RICH:
    logging.basicConfig(format=FORMAT, datefmt="[%X]", handlers=[RichHandler()], level=logging.INFO)


if __name__ == "__main__":
    parser = TrlParser((DPOScriptArguments, DPOConfig, ModelConfig))
    args, training_args, model_config = parser.parse_args_and_config()
    wandb.init(project="DPO_Models")

    # Force use our print callback
    if TRL_USE_RICH:
        training_args.disable_tqdm = True
        console = Console()

    ################
    # Model & Tokenizer
    ################
    torch_dtype = (
        model_config.torch_dtype
        if model_config.torch_dtype in ["auto", None]
        else getattr(torch, model_config.torch_dtype)
    )
    quantization_config = get_quantization_config(model_config)
    model_kwargs = dict(
        revision=model_config.model_revision,
        trust_remote_code=model_config.trust_remote_code,
        attn_implementation=model_config.attn_implementation,
        torch_dtype=torch_dtype,
        use_cache=False if training_args.gradient_checkpointing else True,
        device_map=get_kbit_device_map() if quantization_config is not None else None,
        quantization_config=quantization_config,
    )
    model = AutoModelForCausalLM.from_pretrained(model_config.model_name_or_path, **model_kwargs)
    peft_config = get_peft_config(model_config)
    if peft_config is None:
        model_ref = AutoModelForCausalLM.from_pretrained(model_config.model_name_or_path, **model_kwargs)
    else:
        model_ref = None
    tokenizer = AutoTokenizer.from_pretrained(model_config.model_name_or_path)
    if tokenizer.pad_token is None:
        tokenizer.pad_token = tokenizer.eos_token
    if tokenizer.chat_template is None:
        tokenizer.chat_template = "{% for message in messages %}{{message['role'] + ': ' + message['content'] + '\n\n'}}{% endfor %}{{ eos_token }}"
    if args.ignore_bias_buffers:
        # torch distributed hack
        model._ddp_params_and_buffers_to_ignore = [
            name for name, buffer in model.named_buffers() if buffer.dtype == torch.bool
        ]

    ################
    # Optional rich context managers
    ###############
    init_context = nullcontext() if not TRL_USE_RICH else console.status("[bold green]Initializing the DPOTrainer...")
    save_context = (
        nullcontext()
        if not TRL_USE_RICH
        else console.status(f"[bold green]Training completed! Saving the model to {training_args.output_dir}")
    )

    ################
    # Dataset
    ################
    
    raw_datasets = load_dataset('openai/summarize_from_feedback', 'comparisons')
    raw_train_dataset = raw_datasets['train']
    raw_eval_dataset = raw_datasets['validation']

    transformed_rows_train = []
    for row in raw_train_dataset:
        prompt = row['info']['post']
        if prompt == None:
            continue
        summaries = row['summaries']
        choice = int(row['choice'])
        transformed_rows_train.append({
            'chosen': f'Summarize: "{prompt}"\n{summaries[choice]["text"]}"',
            'rejected': f'Summarize: "{prompt}"\n{summaries[1 - choice]["text"]}"',
            'prompt': prompt,
        })
    transformed_rows_eval = []
    for row in raw_eval_dataset:
        prompt = row['info']['post']
        if prompt == None:
            continue
        summaries = row['summaries']
        choice = int(row['choice'])
        transformed_rows_eval.append({
            'chosen': f'Summarize: "{prompt}"\n{summaries[choice]["text"]}"',
            'rejected': f'Summarize: "{prompt}"\n{summaries[1 - choice]["text"]}"',
            'prompt': prompt,
        })
    train_dataset = Dataset.from_list(transformed_rows_train)
    eval_dataset = Dataset.from_list(transformed_rows_eval)
    
    train_dataset = train_dataset
    idxs = np.random.choice(len(eval_dataset), size=256, replace=False)
    eval_dataset = eval_dataset.select(idxs)

    ################
    # Training
    ################
    trainer = DPOTrainer(
        model,
        model_ref,
        args=training_args,
        train_dataset=train_dataset,
        eval_dataset=eval_dataset,
        tokenizer=tokenizer,
        peft_config=get_peft_config(model_config),
        callbacks=[RichProgressCallback] if TRL_USE_RICH else None,
    )

    trainer.train()
    trainer.save_model(training_args.output_dir)