from collections import namedtuple
from typing import Optional
from typing import Tuple

import sys

import gym
import gym_hybrid
import numpy as np
import cv2
import os
from gym import spaces
from gym.utils import seeding

from shapely.geometry import Point, Polygon
import math
import matplotlib.pyplot as plt
import pandas as pd
# import random
from easydict import EasyDict
from shapely.geometry import Point, Polygon, LineString

from .parking_environment_safe_core import plot_arrow, calculate_overlap_area
from .parking_environment_safe_core import distance_point_to_segment, point_to_polygon_distance
from .parking_environment_safe_core import ceter_to_rear, rear_to_ceter
from .parking_environment_safe_core import is_point_in_rectangle, position_vehicle_vertex
from .parking_environment_safe_core import env_stop_condition, compute_env_reward
from .parking_environment_safe_core import compute_vehicle_pos, compute_vehicle_pos_hybrid_com
from .parking_environment_safe_core import Perpendicular_parking, Perpendicular_parking_hybrid, Perpendicular_parking_hybrid_boundary
from .parking_environment_safe_core import Perpendicular_parking_hybrid_boundary
from .parking_environment_safe_core import generate_init_vehicle, get_init_vehicle_pos
from .parking_environment_safe_core import generate_target_parking,  get_target_parking_pos
from .parking_environment_safe_core import get_init_vehicle_area, vehicle_wall_obstacle
from .parking_environment_safe_core import ego_vehicle_vertex_midpoints, lidar_detection_distance
from .parking_environment_safe_core import Parallel_parking_hybrid_boundary, get_target_parking_pos_parallel
from .parking_environment_safe_core import vehicle_wall_obstacle_parallel, get_init_vehicle_area_parallel
from .parking_environment_safe_core import get_init_vehicle_pos_parallel, compute_env_reward_parallel


import warnings
warnings.filterwarnings("ignore")
from .agents import BaseAgent, MovingAgent, SlidingAgent

ACTION_Ratio = 1
# Discrete Action
ACTION ={'rightfront':0,
         'straightfront':1,
         'leftfront':2,
         'leftrear':3,
         'straightrear':4,
         'rightrear':5}

ACTION_COM ={'right':0,
            'straight':1,
            'left':2}

ACTION = EasyDict(ACTION)
ACTION_COM = EasyDict(ACTION_COM)
# The minimum radius
RADIUS = 5

def generate_discrete_points(start_point, heading_angle, radius, arc_length, interval, turn_direction=None):
    """
    Generate discrete points at fixed intervals on an arc or straight line
    :param start_point: start point coordinate (x, y)
    :param heading_angle: Heading Angle (in degrees,0 degrees is the positive direction of the X-axis)
    :param radius: indicates the radius of an arc or straight line
    :param arc_length: indicates the arc length
    :param interval: indicates the fixed interval of discrete points
    :param turn_direction: 0 indicates a right turn,1 indicates straight ahead, and 2 indicates a left turn
    :return: list of coordinates of discrete points [(x1, y1), (x2, y2),...]
    """
    points = []
    x, y = start_point

    heading_rad = heading_angle

    if turn_direction == 2 or turn_direction == 0:
        
        if turn_direction == 2:  
            center_x = x - radius * np.sin(heading_rad)
            center_y = y + radius * np.cos(heading_rad)
            arc_angle = arc_length / radius
            start_angle = np.arctan2(y - center_y, x - center_x)
            theta = np.linspace(start_angle, start_angle + arc_angle, int(abs(arc_length) // interval) + 1)
            heading_theta = np.linspace(heading_rad, heading_rad + arc_angle, int(abs(arc_length) // interval) + 1)
        else:  
            center_x = x + radius * np.sin(heading_rad)
            center_y = y - radius * np.cos(heading_rad)
            arc_angle = arc_length / radius
            start_angle = np.arctan2(y - center_y, x - center_x)
            theta = np.linspace(start_angle, start_angle - arc_angle, int(abs(arc_length) // interval) + 1)
            heading_theta = np.linspace(heading_rad, heading_rad - arc_angle, int(abs(arc_length) // interval) + 1)
        

        for i in range(len(theta)):
            px = center_x + radius * np.cos(theta[i])
            py = center_y + radius * np.sin(theta[i])
            points.append((px, py, heading_theta[i]))
    
    else:  
        num_points = int(abs(arc_length) // interval) + 1
        direction = 1 if arc_length >= 0 else -1
        dx = direction * interval * np.cos(heading_rad)
        dy = direction * interval * np.sin(heading_rad)
        
        for i in range(num_points):
            px = x + i * dx
            py = y + i * dy
            points.append((px, py, heading_rad))

    return points

def get_path_with_actions(start_points, instructions, interval=0.1):
    """
    Generate and draw discrete points on a path according to instructions
    :param start_points: list of initial point coordinates and heading [(x1, y1, heading1), (x2, y2, heading2),...]
    :param instructions: list of instructions for each point [(turn_direction1, arc_length1),...]
    :param interval: indicates the interval between discrete points
    """
    all_points = []  

    for (x, y, heading), instruction_set in zip(start_points, instructions):
        direction, arc_length = instruction_set
        radius = 5  
        points = generate_discrete_points((x, y), heading, radius, arc_length, interval, turn_direction=direction)
        all_points.extend(points)  
    
    all_points = np.array(all_points)
    return all_points

class Action:
    """"
    Action class to store and standardize the action for the environment.
    """

    def __init__(self, id_: int, parameters: list):
        """"
        Initialization of an action.

        Args:
            id_: The id of the selected action.
            parameters: The parameters of an action.
        """
        self.id = id_
        self.parameters = parameters * ACTION_Ratio

    @property
    def parameter(self) -> float:
        """"
        Property method to return the parameter related to the action selected.

        Returns:
            The parameter related to this action_id
        """
        # if len(self.parameters) == 2:
        #     return self.parameters[self.id]
        # else:
        #     if self.parameters[0] < 0:
        #         return 0 
        #     elif self.parameter[0]>2:
        #         return 2
        #     else:
        #         return self.parameters[0]
            
        if len(self.parameters) == 2:
            return self.parameters[self.id]
        else:
            return self.parameters[0]

class PerpendicularParkEnv_Safe(gym.Env):
    '''
        Create the RL environment for perpendicular parking environment
    '''
    def __init__(self):
        '''
        Init the environment
        '''
        super(PerpendicularParkEnv_Safe, self).__init__()
        self.width_park = 2.5    # the width of the parking spaces
        self.length_park = 5.3   # the legth of the parking spaces
        self.width_lane = 3.5    # the width of the lane
        self.width_vehicle = 2.0     
        self.length_vehicle = 4.3    

        self.parking_num = 4        
        self.wheelbase = 2.7         
             
        self.action_dim = 3     
        self.state_dim = 17        

        self.max_steps = 50

        self.current_step = 0     
        self.park_bounary = 0.5   

        vehicle_config = {
            'vehicle_width': 2.0,
            'vehicle_length': 4.3,
            'wheelbase': 2.7
        }

        parking_config = {
            'park_width': 2.5,
            'park_length': 5.3,
            'lane_width': 3.5,
            'parking_num': 4
        }
        self.vehicle_config = EasyDict(vehicle_config)
        self.parking_config = EasyDict(parking_config)

        self.ego_vehicle_area =  self.vehicle_config.vehicle_width* self.vehicle_config.vehicle_length


        self.parking_spaces, self.lane_marks, self.park_vertex = Perpendicular_parking_hybrid_boundary(
            width_park = self.width_park,
            length_park = self.length_park,
            width_lane = self.width_lane,
            parking_num = 4,
            boundary_width = self.park_bounary
        )
        

        parameters_min = -1
        parameters_max = 1
        self.action_space = spaces.Tuple((spaces.Discrete(self.action_dim), 
                                          spaces.Box(low=parameters_min, high=parameters_max)))
        self.observation_space = spaces.Box(low=-20, high=20, 
                                            shape=(self.state_dim,), dtype=np.float32)

    def seed(self, seed = 0):
        '''
        Set the seed for the environment
        '''
        np.random.seed(seed)
        return True
    
    def reset(self, train_stage=1, eval_stage=False):
        '''
        Reset the environment
        '''       
        reset_flag = True 
        
        while reset_flag:
            # self.target_parking_id = random.randint(1,2)
            self.target_parking_id = 1
            self.target_pos, self.target_area = get_target_parking_pos(
                target_parking_id = self.target_parking_id,
                parking_spaces = self.parking_spaces
            )

            self.park_vehicle_polygons, self.park_vertex_polygon = vehicle_wall_obstacle(
                parking_spaces = self.parking_spaces,
                park_vertex = self.park_vertex,
                vehicle_config = self.vehicle_config,
                target_parking_id = self.target_parking_id,
            )

            # reset the position of the ego vehicle 
            # area_id = random.randint(1, 2)
            if eval_stage == True:
                area_id = np.random.randint(3, 4)
            else:
                area_id = np.random.randint(1, 4)
            # area_id = 2
            # area_id = 1
            # area_id = np.random.randint(1, 3)
            # area_id = np.random.randint(2, 3)
            # area_id = np.random.randint(1, 2)
            # area_id = np.random.randint(1, 4)
        
            vehicle_area = get_init_vehicle_area(
                target_parking_id = self.target_parking_id,
                area_id = area_id,
                parking_config = self.parking_config,
                vehicle_config = self.vehicle_config,
                park_boundary = self.park_bounary
            )

            # vehicle_area1 = get_init_vehicle_area(
            #     target_parking_id = self.target_parking_id,
            #     area_id = 2,
            #     parking_config = self.parking_config,
            #     vehicle_config = self.vehicle_config,
            #     park_boundary = self.park_bounary
            # )

            # vehicle_area2 = get_init_vehicle_area(
            #     target_parking_id = self.target_parking_id,
            #     area_id = 3,
            #     parking_config = self.parking_config,
            #     vehicle_config = self.vehicle_config,
            #     park_boundary = self.park_bounary
            # )

            # self.render_parking_area(
            #     parking_spaces = self.parking_spaces,
            #     lane_marks = self.lane_marks,
            #     park_vertex = self.park_vertex,
            #     vehicle_area = vehicle_area,
            #     vehicle_area1 = vehicle_area1,
            #     vehicle_area2 = vehicle_area2
            # )

            self.vehicle_pos = get_init_vehicle_pos(
                vehicle_area = vehicle_area,
                vehicle_config = self.vehicle_config,
                area_id = area_id,
            )

            self.init_ego_target_dis = math.sqrt(
                (self.vehicle_pos[0]-self.target_pos[0])**2+
                (self.vehicle_pos[1]-self.target_pos[1])**2
            )    

            self.vehicle_vert = position_vehicle_vertex(
                width_vehicle = self.width_vehicle,
                length_vehicle = self.length_vehicle,
                vehicle_pos = self.vehicle_pos
            )

            self.vehicle_vert_polygons, self.vehicle_midpoints_polygons = ego_vehicle_vertex_midpoints(
                vehicle_vert = self.vehicle_vert
            )

            self.vehicle_ray_start_points = self.vehicle_vert_polygons + self.vehicle_midpoints_polygons

            self.obs_min_dis = lidar_detection_distance(
                vehicle_ray_start_points = self.vehicle_ray_start_points,
                park_vehicle_polygons = self.park_vehicle_polygons,
                park_vertex_polygon = self.park_vertex_polygon,
                vehicle_pos = self.vehicle_pos
            )

            self.vehicle_pos_r = ceter_to_rear(
                ceter_point = self.vehicle_pos, 
                wheelbase = self.wheelbase
            )

            need_reset = any(x < 0 for x in self.obs_min_dis)

            if need_reset:
                reset_flag = True
            else:
                reset_flag = False

        # self.render_parking_vehicle(
        #     parking_spaces = self.parking_spaces,
        #     lane_marks = self.lane_marks,
        #     vehicle_pos = self.vehicle_pos,
        #     vehicle_vert = self.vehicle_vert,
        #     target_parking_id = self.target_parking_id,
        #     target_pos = self.target_pos,
        #     vehicle_area = vehicle_area,
        #     park_vertex = self.park_vertex,
        #     park_vehicle_polygons = self.park_vehicle_polygons,
        #     vehicle_ray_start_points = self.vehicle_ray_start_points,
        #     obs_min_dis = self.obs_min_dis,
        # )

        # self.render_park_trajectory(
        #     parking_spaces = self.parking_spaces,
        #     lane_marks = self.lane_marks,
        #     vehicle_pos = self.vehicle_pos,
        #     vehicle_vert = self.vehicle_vert,
        #     target_parking_id = self.target_parking_id,
        #     target_pos = self.target_pos,
        #     vehicle_area = vehicle_area,
        #     park_vertex = self.park_vertex,
        #     park_vehicle_polygons = self.park_vehicle_polygons,
        #     vehicle_ray_start_points = self.vehicle_ray_start_points,
        #     obs_min_dis = self.obs_min_dis,
        # )


        # vert_mindis = []
        # for i in range(len(self.vehicle_vert[0])):
        #     px = self.vehicle_vert[0][i][0]
        #     py = self.vehicle_vert[0][i][1]
        #     # Internal_points = is_point_in_rectangle(self.vehicle_vert[0][i], self.park_vertex)
        #     if is_point_in_rectangle(self.vehicle_vert[0][i], self.park_vertex):
        #         min_distance = point_to_polygon_distance(px, py, self.park_vertex)
        #     else:
        #         min_distance = - point_to_polygon_distance(px, py, self.park_vertex)
        #     vert_mindis.append(min_distance)
            
        # print('vert_mindis:', vert_mindis)
        self.bias_pos = [self.vehicle_pos[i] - self.target_pos[i] for i in range(len(self.vehicle_pos))]

        # Get the state after reset the environment
        # self.state = np.array(self.vehicle_pos + self.target_pos + self.bias_pos)
        # state = np.array(self.vehicle_pos + self.target_pos + self.bias_pos + vert_mindis).tolist()
        # state = np.array(self.vehicle_pos + self.target_pos + self.bias_pos + vert_mindis)
        state = np.array(self.vehicle_pos + self.target_pos + self.bias_pos + self.obs_min_dis)
        self.current_step = 0
 
        return state
    
    def step(self, raw_action):
        '''
        Interact with the environment
        '''
        # action = action.tolist()
        self.current_step +=1     
        action = Action(*raw_action)

        # print('self.vehicle_pos_r:', self.vehicle_pos_r)
        # Compute the position of the ego vehicle
        # self.vehicle_pos_r = compute_vehicle_pos_hybrid(
        #     vehicle_pos = self.vehicle_pos_r,
        #     action = action
        # )


        # self.vehicle_pos_r = compute_vehicle_pos_hybrid_com(
        #     vehicle_pos = self.vehicle_pos_r,
        #     action = action
        # )


        # self.vehicle_pos = rear_to_ceter(
        #     rear_point = self.vehicle_pos_r,
        #     wheelbase = self.wheelbase
        # )

        self.vehicle_pos = compute_vehicle_pos_hybrid_com(
            vehicle_pos = self.vehicle_pos,
            action = action
        )

        # Transform the vertex of the ego vehicle
        self.vehicle_vert = position_vehicle_vertex(
            width_vehicle = self.width_vehicle,
            length_vehicle = self.length_vehicle,
            vehicle_pos = self.vehicle_pos
        )


        self.vehicle_vert_polygons, self.vehicle_midpoints_polygons = ego_vehicle_vertex_midpoints(
            vehicle_vert = self.vehicle_vert
        )


        self.vehicle_ray_start_points = self.vehicle_vert_polygons + self.vehicle_midpoints_polygons


        self.obs_min_dis = lidar_detection_distance(
            vehicle_ray_start_points = self.vehicle_ray_start_points,
            park_vehicle_polygons = self.park_vehicle_polygons,
            park_vertex_polygon = self.park_vertex_polygon,
            vehicle_pos = self.vehicle_pos
        )


        # vert_mindis = []
        # for i in range(len(self.vehicle_vert[0])):
        #     px = self.vehicle_vert[0][i][0]
        #     py = self.vehicle_vert[0][i][1]
        #     # min_distance = point_to_polygon_distance(px, py, self.park_vertex)
        #     if is_point_in_rectangle(self.vehicle_vert[0][i], self.park_vertex):
        #         min_distance = point_to_polygon_distance(px, py, self.park_vertex)
        #     else:
        #         min_distance = - point_to_polygon_distance(px, py, self.park_vertex)
        #     vert_mindis.append(min_distance)
        
        # The bias of the position and directory between the ego vehicle and the target parking space
        self.bias_pos = [self.vehicle_pos[i] - self.target_pos[i] for i in range(len(self.vehicle_pos))]

        # Get the new state
        # self.state = np.array(self.vehicle_pos + self.target_pos + self.bias_pos)
        # state = np.array(self.vehicle_pos + self.target_pos + self.bias_pos + vert_mindis).tolist()
        # state = np.array(self.vehicle_pos + self.target_pos + self.bias_pos + vert_mindis)
        state = np.array(self.vehicle_pos + self.target_pos + self.bias_pos + self.obs_min_dis)


        # Compute the reward and done
        info = compute_env_reward(
            vehicle_pos = self.vehicle_pos,
            target_pos = self.target_pos,
            parking_spaces = self.parking_spaces,
            vehicle_vertex = self.vehicle_vert,
            park_vertex = self.park_vertex,
            target_area = self.target_area,
            ego_vehicle_area = self.ego_vehicle_area,
            init_ego_target_dis = self.init_ego_target_dis,
            current_step = self.current_step,
            max_step = self.max_steps,
            obs_min_dis = self.obs_min_dis,
        )


        if self.current_step >= self.max_steps:
            truncated = True
        else:
            truncated = False


        info['truncated'] = truncated
        done = info['truncated'] or info['terminated']


        # if done:
        #     print('finished step:', self.current_step)

        # return self.state, info['reward'], done, info
        return state, info['reward'], info['cost'], done, info 

    def close(self):
        '''
        Close the environment
        '''
        return True

    def render_parking(self,parking_spaces,lane_marks,park_vertex):
        '''
        Render the parking scenarios
        '''
        save_dir = './fig/init_parking_v01'


        plt.figure(figsize=(10,17.5))
        for i in range(len(parking_spaces)):
            for j in range(len(parking_spaces[i])):
                if j == len(parking_spaces[i])-1:
                    x = [parking_spaces[i][j][0],parking_spaces[i][0][0]]
                    y = [parking_spaces[i][j][1],parking_spaces[i][0][1]]
                else:
                    x = [parking_spaces[i][j][0],parking_spaces[i][j+1][0]]
                    y = [parking_spaces[i][j][1],parking_spaces[i][j+1][1]]
                plt.plot(x, y, label='linear', linestyle='-', color='black', linewidth=3.0)
        

        x = [lane_marks[0][0][0],lane_marks[0][1][0]]
        y = [lane_marks[0][0][1],lane_marks[0][1][1]]
        plt.plot(x, y, label='linear', linestyle='--', color='black', linewidth=3.0)


        for i in range(len(park_vertex)):
            if i == len(park_vertex)-1:
                x = [park_vertex[i][0],park_vertex[0][0]]
                y = [park_vertex[i][1],park_vertex[0][1]]
            else:
                x = [park_vertex[i][0],park_vertex[i+1][0]]
                y = [park_vertex[i][1],park_vertex[i+1][1]]
            plt.plot(x, y, label='linear', linestyle='-', color='black', linewidth=4.0)

     
        plt.axis('off')
        plt.savefig(save_dir)
        plt.savefig(save_dir+'.pdf')

    def render_parking_area(self,parking_spaces,lane_marks,park_vertex,vehicle_area, vehicle_area1, vehicle_area2):
        '''
        Render the parking scenarios
        '''
        save_dir = './fig/init_parking_area_v01'


        plt.figure(figsize=(10,17.5))
        for i in range(len(parking_spaces)):
            for j in range(len(parking_spaces[i])):
                if j == len(parking_spaces[i])-1:
                    x = [parking_spaces[i][j][0],parking_spaces[i][0][0]]
                    y = [parking_spaces[i][j][1],parking_spaces[i][0][1]]
                else:
                    x = [parking_spaces[i][j][0],parking_spaces[i][j+1][0]]
                    y = [parking_spaces[i][j][1],parking_spaces[i][j+1][1]]
                plt.plot(x, y, label='linear', linestyle='-', color='black', linewidth=3.0)
        

        x = [lane_marks[0][0][0],lane_marks[0][1][0]]
        y = [lane_marks[0][0][1],lane_marks[0][1][1]]
        plt.plot(x, y, label='linear', linestyle='--', color='black', linewidth=3.0)

        for i in range(len(vehicle_area[0])):

            if i == len(vehicle_area[0])-1:
                x = [vehicle_area[0][i][0], vehicle_area[0][0][0]]
                y = [vehicle_area[0][i][1], vehicle_area[0][0][1]]
            else:
                x = [vehicle_area[0][i][0], vehicle_area[0][i+1][0]]
                y = [vehicle_area[0][i][1], vehicle_area[0][i+1][1]]
            plt.plot(x, y, label='linear', linestyle='-.', color='black', linewidth=3.0)

            x = [vehicle_area[0][0][0], vehicle_area[0][1][0]]
            y1 = [vehicle_area[0][0][1], vehicle_area[0][1][1]]
            y2 = [vehicle_area[0][2][1], vehicle_area[0][3][1]]
            plt.fill_between(x,y1,y2,color='grey')



        for i in range(len(vehicle_area1[0])):
            if i == len(vehicle_area1[0])-1:
                x = [vehicle_area1[0][i][0], vehicle_area1[0][0][0]]
                y = [vehicle_area1[0][i][1], vehicle_area1[0][0][1]]
            else:
                x = [vehicle_area1[0][i][0], vehicle_area1[0][i+1][0]]
                y = [vehicle_area1[0][i][1], vehicle_area1[0][i+1][1]]
            plt.plot(x, y, label='linear', linestyle='-.', color='black', linewidth=3.0)




        for i in range(len(vehicle_area2[0])):
            if i == len(vehicle_area2[0])-1:
                x = [vehicle_area2[0][i][0], vehicle_area2[0][0][0]]
                y = [vehicle_area2[0][i][1], vehicle_area2[0][0][1]]
            else:
                x = [vehicle_area2[0][i][0], vehicle_area2[0][i+1][0]]
                y = [vehicle_area2[0][i][1], vehicle_area2[0][i+1][1]]
            plt.plot(x, y, label='linear', linestyle='-.', color='black', linewidth=3.0)
            x = [vehicle_area2[0][0][0], vehicle_area2[0][1][0]]
            y1 = [vehicle_area2[0][0][1], vehicle_area2[0][1][1]]
            y2 = [vehicle_area2[0][2][1], vehicle_area2[0][3][1]]
            plt.fill_between(x,y1,y2,color='grey')


        for i in range(len(park_vertex)):
            if i == len(park_vertex)-1:
                x = [park_vertex[i][0],park_vertex[0][0]]
                y = [park_vertex[i][1],park_vertex[0][1]]
            else:
                x = [park_vertex[i][0],park_vertex[i+1][0]]
                y = [park_vertex[i][1],park_vertex[i+1][1]]
            plt.plot(x, y, label='linear', linestyle='-', color='black', linewidth=4.0)

     
        plt.axis('off')
        plt.savefig(save_dir)
        plt.savefig(save_dir+'.pdf')

    def render_parking_vehicle(
            self,
            parking_spaces,
            lane_marks,
            vehicle_pos, 
            vehicle_vert, 
            target_parking_id,
            target_pos,
            vehicle_area,
            park_vertex,
            park_vehicle_polygons,
            vehicle_ray_start_points,
            obs_min_dis,
    ):
        '''
        Render the init parking scenarios and vehicle.
        '''
        save_dir = './fig/parking_area_vehicle_v04'
        plt.figure(figsize=(10,12.5))


        vehicle_center = Point(vehicle_pos[0], vehicle_pos[1])
        ray_length = 10


        park_vertex_polygons = Polygon(park_vertex)
        park_x, park_y = park_vertex_polygons.exterior.xy
        plt.fill(park_x, park_y, label='parking lot', color='lightgrey')
        
        parking_spaces_polygons = []
        for i in range(len(parking_spaces)):
            parking_polygon = Polygon(
                [(parking_spaces[i][0][0], parking_spaces[i][0][1]),
                (parking_spaces[i][1][0], parking_spaces[i][1][1]),
                (parking_spaces[i][2][0], parking_spaces[i][2][1]),
                (parking_spaces[i][3][0], parking_spaces[i][3][1])]
            )
            parking_spaces_polygons.append(parking_polygon)

        for parking_spaces_polygon in parking_spaces_polygons:
            if isinstance(parking_spaces_polygon, Polygon):
                parking_area_x, parking_area_y = parking_spaces_polygon.exterior.xy
                plt.fill(parking_area_x, parking_area_y, alpha = 0.5, label='parking area', color ='lightblue')

        for i in range(len(parking_spaces)):
            for j in range(len(parking_spaces[i])):
                if j == len(parking_spaces[i])-1:
                    x = [parking_spaces[i][j][0],parking_spaces[i][0][0]]
                    y = [parking_spaces[i][j][1],parking_spaces[i][0][1]]
                else:
                    x = [parking_spaces[i][j][0],parking_spaces[i][j+1][0]]
                    y = [parking_spaces[i][j][1],parking_spaces[i][j+1][1]]
                plt.plot(x, y, label='linear', linestyle='-', color='black', linewidth=1.0)


        x = [lane_marks[0][0][0],lane_marks[0][1][0]]
        y = [lane_marks[0][0][1],lane_marks[0][1][1]]
        plt.plot(x, y, label='linear', linestyle='--', color='black', linewidth=2.0)

        for park_vehicle_polygon in park_vehicle_polygons:
            if isinstance(parking_spaces_polygon, Polygon):
                park_vehicle_x, park_vehicle_y = park_vehicle_polygon.exterior.xy
                plt.fill(park_vehicle_x, park_vehicle_y, alpha = 0.5, label='park vehicle', color ='salmon')

        vehicle_vert_polygon = Polygon(
            [(vehicle_vert[0][0][0], vehicle_vert[0][0][1]),
            (vehicle_vert[0][1][0], vehicle_vert[0][1][1]),
            (vehicle_vert[0][2][0], vehicle_vert[0][2][1]),
            (vehicle_vert[0][3][0], vehicle_vert[0][3][1])]
        )
        vehicle_x, vehicle_y = vehicle_vert_polygon.exterior.xy
        plt.fill(vehicle_x, vehicle_y, label='parking lot', color='lightgreen')

        plt.plot(vehicle_pos[0], vehicle_pos[1], marker='o', c='k',markersize=10)

        plot_arrow(vehicle_pos[0], vehicle_pos[1], vehicle_pos[2], length=2.0, color='green')


        id = 0
        for start_point in vehicle_ray_start_points:


            angle = np.arctan2(start_point.y - vehicle_center.y, start_point.x - vehicle_center.x)

            end_x = start_point.x + obs_min_dis[id]*np.cos(angle)
            end_y = start_point.y + obs_min_dis[id]*np.sin(angle)

            plt.plot([start_point.x, end_x], [start_point.y, end_y], color='blue', alpha=0.5)
            id += 1

        plt.grid()
        plt.axis('off')
        plt.savefig(save_dir)
        plt.savefig(save_dir+'.pdf')

    def render_park_trajectory(
            self,
            parking_spaces,
            lane_marks,
            vehicle_pos, 
            vehicle_vert, 
            target_parking_id,
            target_pos,
            vehicle_area,
            park_vertex,
            park_vehicle_polygons,
            vehicle_ray_start_points,
            obs_min_dis,
    ):
        '''
        Render the parking scenarios and parking trajectory.
        '''
        save_dir = './perpendicular_fig/eval_tra/trajectory_20241115_234904'

        plt.figure(figsize=(10,12.5))



        park_vertex_polygons = Polygon(park_vertex)
        park_x, park_y = park_vertex_polygons.exterior.xy
        plt.fill(park_x, park_y, label='parking lot', color='lightgrey')
        

        parking_spaces_polygons = []
        for i in range(len(parking_spaces)):
            parking_polygon = Polygon(
                [(parking_spaces[i][0][0], parking_spaces[i][0][1]),
                (parking_spaces[i][1][0], parking_spaces[i][1][1]),
                (parking_spaces[i][2][0], parking_spaces[i][2][1]),
                (parking_spaces[i][3][0], parking_spaces[i][3][1])]
            )
            parking_spaces_polygons.append(parking_polygon)

        for parking_spaces_polygon in parking_spaces_polygons:
            if isinstance(parking_spaces_polygon, Polygon):
                parking_area_x, parking_area_y = parking_spaces_polygon.exterior.xy
                plt.fill(parking_area_x, parking_area_y, alpha = 0.5, label='parking area', color ='lightblue')


        for i in range(len(parking_spaces)):
            for j in range(len(parking_spaces[i])):
                if j == len(parking_spaces[i])-1:
                    x = [parking_spaces[i][j][0],parking_spaces[i][0][0]]
                    y = [parking_spaces[i][j][1],parking_spaces[i][0][1]]
                else:
                    x = [parking_spaces[i][j][0],parking_spaces[i][j+1][0]]
                    y = [parking_spaces[i][j][1],parking_spaces[i][j+1][1]]
                plt.plot(x, y, label='linear', linestyle='-', color='black', linewidth=1.0)



        x = [lane_marks[0][0][0],lane_marks[0][1][0]]
        y = [lane_marks[0][0][1],lane_marks[0][1][1]]
        plt.plot(x, y, label='linear', linestyle='--', color='black', linewidth=2.0)


        for park_vehicle_polygon in park_vehicle_polygons:
            if isinstance(parking_spaces_polygon, Polygon):
                park_vehicle_x, park_vehicle_y = park_vehicle_polygon.exterior.xy
                plt.fill(park_vehicle_x, park_vehicle_y, alpha = 0.5, label='park vehicle', color ='salmon')


        data_dir = './eval_log_3/obs_act_rew_20241115_234904.csv' 
        data = pd.read_csv(data_dir)         
        traj_x = data['pos_x'].tolist()
        traj_y = data['pos_y'].tolist()
        traj_dir = data['pos_dir'].tolist()
        action_type = data['action_type'].tolist()
        action_args = data['action_args'].tolist()

        start_points_traj = []
        actions = []
        for i in range(len(traj_x)-1):
            start_points_traj.append((traj_x[i], traj_y[i], traj_dir[i]))
            actions.append((action_type[i], action_args[i]))

        all_points = get_path_with_actions(start_points_traj, actions)
        plt.scatter(all_points[:, 0], all_points[:, 1], s= 5)

        path_vehicle_verts = []
        for i in range(len(all_points)):
            point_vehicle_vert = position_vehicle_vertex(
                width_vehicle = self.vehicle_config.vehicle_width,
                length_vehicle = self.vehicle_config.vehicle_length,
                vehicle_pos = all_points[i]
            )
            path_vehicle_verts.append(point_vehicle_vert[0])

        # print('path_vehicle_verts:', len(path_vehicle_verts))
        print('path_vehicle_verts:', len(path_vehicle_verts[0]))

        for i in range(len(path_vehicle_verts)):
            for j in range(len(path_vehicle_verts[i])):
                if j == len(path_vehicle_verts[i])-1:
                    x = [path_vehicle_verts[i][j][0],path_vehicle_verts[i][0][0]]
                    y = [path_vehicle_verts[i][j][1],path_vehicle_verts[i][0][1]]
                else:
                    x = [path_vehicle_verts[i][j][0],path_vehicle_verts[i][j+1][0]]
                    y = [path_vehicle_verts[i][j][1],path_vehicle_verts[i][j+1][1]]
                plt.plot(x, y, label='linear', linestyle='-', color='green', linewidth=0.5)


        vehicle_pos = [traj_x[0], traj_y[0], traj_dir[0]]
        vehicle_vert = position_vehicle_vertex(width_vehicle = self.width_vehicle, 
                                                 length_vehicle = self.length_vehicle, 
                                                 vehicle_pos = vehicle_pos )
        for i in range(len(vehicle_vert[0])):
            if i == len(vehicle_vert[0])-1:
                x = [vehicle_vert[0][i][0], vehicle_vert[0][0][0]]
                y = [vehicle_vert[0][i][1], vehicle_vert[0][0][1]]
            else:
                x = [vehicle_vert[0][i][0], vehicle_vert[0][i+1][0]]
                y = [vehicle_vert[0][i][1], vehicle_vert[0][i+1][1]]
            plt.plot(x, y, label='linear', linestyle='-', color='green', linewidth=3.0)
        plt.plot(vehicle_pos[0], vehicle_pos[1], marker='o', c='k',markersize=10)
        plot_arrow(vehicle_pos[0], vehicle_pos[1], vehicle_pos[2], length=2.0, color='green')


        final_vehicle_pos = [traj_x[-1], traj_y[-1], traj_dir[-1]]
        final_vehicle_vert = position_vehicle_vertex(width_vehicle = self.width_vehicle, 
                                                     length_vehicle = self.length_vehicle, 
                                                     vehicle_pos = final_vehicle_pos)
        for i in range(len(final_vehicle_vert[0])):
            if i == len(final_vehicle_vert[0])-1:
                x = [final_vehicle_vert[0][i][0], final_vehicle_vert[0][0][0]]
                y = [final_vehicle_vert[0][i][1], final_vehicle_vert[0][0][1]]
            else:
                x = [final_vehicle_vert[0][i][0], final_vehicle_vert[0][i+1][0]]
                y = [final_vehicle_vert[0][i][1], final_vehicle_vert[0][i+1][1]]
            plt.plot(x, y, label='linear', linestyle='-', color='green', linewidth=3.0)
        plt.plot(final_vehicle_pos[0], final_vehicle_pos[1], marker='o', c='k',markersize=10)  
        plot_arrow(final_vehicle_pos[0], final_vehicle_pos[1], final_vehicle_pos[2], length=2.0, color='green')  

        plt.grid()
        plt.axis('off')
        plt.savefig(save_dir)
        plt.savefig(save_dir+'.pdf')
        quit()

    def render_init_parking(self,parking_spaces, lane_marks, vehicle_pos, vehicle_vert, target_parking_id, target_pos, vehicle_area):
        '''
        Render the init parking scenarios
        '''
        save_dir = './fig/parking_vehicle_pos_02'

        plt.figure(figsize=(10,17.5))
        for i in range(len(parking_spaces)):
            for j in range(len(parking_spaces[i])):
                if j == len(parking_spaces[i])-1:
                    x = [parking_spaces[i][j][0],parking_spaces[i][0][0]]
                    y = [parking_spaces[i][j][1],parking_spaces[i][0][1]]
                else:
                    x = [parking_spaces[i][j][0],parking_spaces[i][j+1][0]]
                    y = [parking_spaces[i][j][1],parking_spaces[i][j+1][1]]
                plt.plot(x, y, label='linear', linestyle='-', color='black', linewidth=3.0)
        

        x = [lane_marks[0][0][0],lane_marks[0][1][0]]
        y = [lane_marks[0][0][1],lane_marks[0][1][1]]
        plt.plot(x, y, label='linear', linestyle='--', color='black', linewidth=3.0)


        for i in range(len(vehicle_area[0])):
            if i == len(vehicle_area[0])-1:
                x = [vehicle_area[0][i][0], vehicle_area[0][0][0]]
                y = [vehicle_area[0][i][1], vehicle_area[0][0][1]]
            else:
                x = [vehicle_area[0][i][0], vehicle_area[0][i+1][0]]
                y = [vehicle_area[0][i][1], vehicle_area[0][i+1][1]]
            plt.plot(x, y, label='linear', linestyle='-.', color='black', linewidth=3.0)
            x = [vehicle_area[0][0][0], vehicle_area[0][1][0]]
            y1 = [vehicle_area[0][0][1], vehicle_area[0][1][1]]
            y2 = [vehicle_area[0][2][1], vehicle_area[0][3][1]]
            plt.fill_between(x,y1,y2,color='grey')  

        for i in range(len(vehicle_vert[0])):
            if i == len(vehicle_vert[0])-1:
                x = [vehicle_vert[0][i][0], vehicle_vert[0][0][0]]
                y = [vehicle_vert[0][i][1], vehicle_vert[0][0][1]]
            else:
                x = [vehicle_vert[0][i][0], vehicle_vert[0][i+1][0]]
                y = [vehicle_vert[0][i][1], vehicle_vert[0][i+1][1]]
            plt.plot(x, y, label='linear', linestyle='-', color='black', linewidth=3.0)

            plt.plot(vehicle_pos[0], vehicle_pos[1], marker='o', c='k',markersize=10)


        for j in range(len(parking_spaces[target_parking_id])):
            if j == len(parking_spaces[target_parking_id])-1:
                x = [parking_spaces[target_parking_id][j][0],parking_spaces[target_parking_id][0][0]]
                y = [parking_spaces[target_parking_id][j][1],parking_spaces[target_parking_id][0][1]]
            else:
                x = [parking_spaces[target_parking_id][j][0],parking_spaces[target_parking_id][j+1][0]]
                y = [parking_spaces[target_parking_id][j][1],parking_spaces[target_parking_id][j+1][1]]
            plt.plot(x, y, label='linear', linestyle='-', color='black', linewidth=3.0)

            x = [parking_spaces[target_parking_id][0][0], parking_spaces[target_parking_id][1][0]]
            y1 = [parking_spaces[target_parking_id][0][1], parking_spaces[target_parking_id][1][1]]
            y2 = [parking_spaces[target_parking_id][2][1], parking_spaces[target_parking_id][3][1]]
            plt.fill_between(x,y1,y2,color='limegreen')

            plt.plot(target_pos[0], target_pos[1], marker='*', c='k', markersize=15) 

        plt.axis('off')
        plt.savefig(save_dir)
        plt.savefig(save_dir+'.pdf')

        return True
        
    def render_parking_trajectory(self,parking_spaces,lane_marks, park_vertex):
        '''
        Render the parking scenarios
        '''
        save_dir = './fig_3/eval_fig_tra/trajectory_20241019_121240'


        plt.figure(figsize=(10,17.5))
        for i in range(len(parking_spaces)):
            for j in range(len(parking_spaces[i])):
                if j == len(parking_spaces[i])-1:
                    x = [parking_spaces[i][j][0],parking_spaces[i][0][0]]
                    y = [parking_spaces[i][j][1],parking_spaces[i][0][1]]
                else:
                    x = [parking_spaces[i][j][0],parking_spaces[i][j+1][0]]
                    y = [parking_spaces[i][j][1],parking_spaces[i][j+1][1]]
                plt.plot(x, y, label='linear', linestyle='-', color='black', linewidth=3.0)


        for i in range(len(park_vertex)):
            if i == len(park_vertex)-1:
                x = [park_vertex[i][0],park_vertex[0][0]]
                y = [park_vertex[i][1],park_vertex[0][1]]
            else:
                x = [park_vertex[i][0],park_vertex[i+1][0]]
                y = [park_vertex[i][1],park_vertex[i+1][1]]
            plt.plot(x, y, label='linear', linestyle='-', color='black', linewidth=4.0)

        x = [lane_marks[0][0][0],lane_marks[0][1][0]]
        y = [lane_marks[0][0][1],lane_marks[0][1][1]]
        plt.plot(x, y, label='linear', linestyle='--', color='black', linewidth=3.0)

        data_dir = './eval_log_3/obs_act_rew_20241019_121240.csv' 
        data = pd.read_csv(data_dir)            
        traj_x = data['pos_x'].tolist()
        traj_y = data['pos_y'].tolist()
        traj_dir = data['pos_dir'].tolist()
        plt.scatter(traj_x, traj_y)    



        vehicle_pos = [traj_x[0], traj_y[0], traj_dir[0]]
        vehicle_vert = position_vehicle_vertex(width_vehicle = self.width_vehicle, 
                                                 length_vehicle = self.length_vehicle, 
                                                 vehicle_pos = vehicle_pos )
        for i in range(len(vehicle_vert[0])):
            if i == len(vehicle_vert[0])-1:
                x = [vehicle_vert[0][i][0], vehicle_vert[0][0][0]]
                y = [vehicle_vert[0][i][1], vehicle_vert[0][0][1]]
            else:
                x = [vehicle_vert[0][i][0], vehicle_vert[0][i+1][0]]
                y = [vehicle_vert[0][i][1], vehicle_vert[0][i+1][1]]
            plt.plot(x, y, label='linear', linestyle='-', color='green', linewidth=3.0)
        plt.plot(vehicle_pos[0], vehicle_pos[1], marker='o', c='k',markersize=10)
        plot_arrow(vehicle_pos[0], vehicle_pos[1], vehicle_pos[2], length=2.0, color='green')


        final_vehicle_pos = [traj_x[-1], traj_y[-1], traj_dir[-1]]
        final_vehicle_vert = position_vehicle_vertex(width_vehicle = self.width_vehicle, 
                                                     length_vehicle = self.length_vehicle, 
                                                     vehicle_pos = final_vehicle_pos)
        for i in range(len(final_vehicle_vert[0])):
            if i == len(final_vehicle_vert[0])-1:
                x = [final_vehicle_vert[0][i][0], final_vehicle_vert[0][0][0]]
                y = [final_vehicle_vert[0][i][1], final_vehicle_vert[0][0][1]]
            else:
                x = [final_vehicle_vert[0][i][0], final_vehicle_vert[0][i+1][0]]
                y = [final_vehicle_vert[0][i][1], final_vehicle_vert[0][i+1][1]]
            plt.plot(x, y, label='linear', linestyle='-', color='green', linewidth=3.0)
        plt.plot(final_vehicle_pos[0], final_vehicle_pos[1], marker='o', c='k',markersize=10)  
        plot_arrow(final_vehicle_pos[0], final_vehicle_pos[1], final_vehicle_pos[2], length=2.0, color='green')  
             
        plt.axis('off')
        plt.savefig(save_dir)
        plt.savefig(save_dir+'.pdf')
