import torch.nn as nn
import torch
import torch.nn.functional as F
from model.base_module.cross_attention import TransformerCrossAttnLayer
from model.base_module.flash_attention.transformer import FlashSelfAttnLayer, FlashCrossAttnLayer
from einops import rearrange
import math

class CrossViewEncoder(nn.Module):
    def __init__(self, config, in_dim, in_res) -> None:
        super(CrossViewEncoder, self).__init__()
        self.config = config

        self.transformers_cross, self.transformers_self = make_encoder_transformer_layers(config, in_dim)
        
        
    def forward(self, x):
        '''
        x in shape [b,t,c,h,w]
        '''
        b,t,c,h,w = x.shape

        # get canonical view
        x_canonical = x[:, 0]                                           # [b,c,h,w]
        x = x[:, 1:]                                                    # [b,t-1,c,h,w]
        x_canonical = rearrange(x_canonical, 'b c h w -> b (h w) c')
        x = rearrange(x, 'b t c h w -> b (t h w) c')

        # get through transformer encoder
        for (cross_attn, self_attn) in zip(self.transformers_cross, self.transformers_self):   # requires [b,n,c] inputs
            # cross-attention between canonical-other frames
            x = cross_attn(x, memory=x_canonical)       # [b,(t-1)*h*w,c]
            # concat all frame features
            x = rearrange(x, 'b (t h w) c -> b t c h w', t=t-1, h=h, w=w)
            x_canonical = rearrange(x_canonical, 'b (t h w) c -> b t c h w', t=1, h=h, w=w)
            x = torch.cat([x_canonical, x], dim=1)          # [b,t,c,h,w]
            x = rearrange(x, 'b t c h w -> b (t h w) c')    # [b,n=t*h*w,c]
            # self-attention refinement for all frames
            x = self_attn(x)
            # split the canonical and other frame features
            x = rearrange(x, 'b (t h w) c -> b t c h w', t=t, h=h, w=w)
            x_canonical = x[:, 0]
            x = x[:, 1:]
            x_canonical = rearrange(x_canonical, 'b c h w -> b (h w) c')
            x = rearrange(x, 'b t c h w -> b (t h w) c')

        x_canonical = rearrange(x_canonical, 'b (t h w) c -> b t c h w', t=1, h=h, w=w)
        x = rearrange(x, 'b (t h w) c -> b t c h w', t=t-1, h=h, w=w)
        x = torch.cat([x_canonical, x], dim=1)          # [b,t,c,h,w]

        return x


def make_encoder_transformer_layers(config, in_dim):
    transformers_cross, transformers_self = [], []
    num_layers = config.model.encoder_layers
    mlp_ratio = 4.0
    norm_first = config.model.norm_first

    if not config.model.use_flash_attn:
        latent_dim = int(mlp_ratio * in_dim)
        for _ in range(num_layers):
            cross_attn = TransformerCrossAttnLayer(d_model=in_dim, nhead=8, dim_feedforward=latent_dim, 
                                                   dropout=0.0, activation='gelu', batch_first=True, norm_first=norm_first)
            self_attn = torch.nn.TransformerEncoderLayer(d_model=in_dim, nhead=8, dim_feedforward=latent_dim,
                                                         dropout=0.0, activation='gelu', batch_first=True, norm_first=norm_first)
            transformers_cross.append(cross_attn)
            transformers_self.append(self_attn)
    else:
        for _ in range(num_layers):
            cross_attn = FlashCrossAttnLayer(d_model=in_dim, n_head=12, mlp_ratio=mlp_ratio, norm_first=norm_first)
            self_attn = FlashSelfAttnLayer(d_model=in_dim, n_head=12, mlp_ratio=mlp_ratio, norm_first=norm_first)
            transformers_cross.append(cross_attn)
            transformers_self.append(self_attn)

    transformers_cross = nn.ModuleList(transformers_cross)
    transformers_self = nn.ModuleList(transformers_self)
    return transformers_cross, transformers_self
