"""
Autodock Vina Docking evaluation pipeline.

Adapted from Therapeutic Data Commons (TDC).
Huang et al. (2021) https://arxiv.org/abs/2102.09548
"""
import os
import subprocess
from typing import List, Tuple, Optional
from pathlib import Path
from tqdm import tqdm
import pickle
import time

import numpy as np
from rdkit import Chem
from rdkit.Chem import AllChem

from utils.convert_data import get_smiles_from_atom_pos

try:
    from vina import Vina
except:
    raise ImportError(
        "Please install vina following guidance in https://github.com/ccsb-scripps/AutoDock-Vina/tree/develop/build/python"
    )

if 'TMPDIR' in os.environ:
    TMPDIR = Path(os.environ['TMPDIR'])
else:
    TMPDIR = Path('./')

# Ligands from PDB
docking_target_info = {
    "1iep": {
        "center": (15.61389189189189, 53.38013513513513, 15.454837837837842),
        "size": (15, 15, 15),
        "ligand": "Cc1ccc(cc1Nc2nccc(n2)c3cccnc3)NC(=O)c4ccc(cc4)CN5CCN(CC5)C", # STI, Imatinib
        "pH": 6.5
    },
    "3eml": {
        "center": (-9.063639999999998, -7.1446, 55.86259999999999),
        "size": (15, 15, 15),
        "ligand": "c1cc(oc1)c2nc3nc(nc(n3n2)N)NCCc4ccc(cc4)O", # ZMA
        "pH": 6.5
    },
    "3ny8": {
        "center": (2.2488, 4.68495, 51.39820000000001),
        "size": (15, 15, 15),
        "ligand": "Cc1ccc(c2c1CCC2)OC[C@H]([C@H](C)NC(C)C)O", # JRZ
        "pH": 7.5
    },
    "4rlu": {
        "center": (-0.7359999999999999, 22.75547368421052, -31.2368947368421),
        "size": (15, 15, 15),
        "ligand": "c1cc(ccc1C=CC(=O)c2ccc(cc2O)O)O", # HCC
        "pH": 7.5
    },
    "4unn": {
        "center": (5.684346153846153, 18.191769230769232, -7.37157692307692),
        "size": (15, 15, 15),
        "ligand": "COc1cccc(c1)C2=CC(=C(C(=O)N2)C#N)c3ccc(cc3)C(=O)O", # QZZ
        "pH": 7.4 # non reported
    },
    "5mo4": {
        "center": (-44.901709677419355, 20.490354838709674, 8.483354838709678),
        "size": (15, 15, 15),
        "ligand": "c1cc(ccc1NC(=O)c2cc(c(nc2)N3CC[C@H](C3)O)c4ccn[nH]4)OC(F)(F)Cl", # AY7, asciminib
        "pH": 7.5
    },
    "7l11": {
        "center": (-21.814812500000006, -4.216062499999999, -27.983781250000),
        "size": (15, 15, 15),
        "ligand": "CCCOc1cc(cc(c1)Cl)C2=CC(=CN(C2=O)c3cccnc3)c4ccccc4C#N", # XF1
        "pH": 6.0
    },
}


def protonate_smiles(smiles: str,
                     pH: float = 7.4,
                     path_to_bin: str = ''
                     ) -> str:
    """
    Protonate SMILES string with OpenBabel at given pH.

    Adapted from DockString:
    https://github.com/dockstring/dockstring/blob/main/dockstring/utils.py#L330

    Arguments
    ---------
    smiles : str SMILES string of molecule to be protonated
    pH : float (default = 7.4) pH at which the molecule should be protonated
    path_to_bin : str (default = '') path to environment bin containing `mk_prepare_ligand.py`
        from `meeko` and `openbabel` if `protonate`=True

    Returns
    -------
    SMILES string of protonated structure
    """

    # cmd list format raises errors, therefore one string
    cmd = f'{path_to_bin}obabel -:"{smiles}" -ismi -ocan -p {pH}'
    cmd_return = subprocess.run(cmd, capture_output=True, shell=True)
    output = cmd_return.stdout.decode('utf-8')

    if cmd_return.returncode != 0:
        raise ValueError(f'Could not protonate SMILES: {smiles}')

    return output.strip()


class VinaSmiles:
    """
    Perform docking search from a SMILES string.

    Adapted from TDC.
    """
    def __init__(self,
                 receptor_pdbqt_file: str,
                 center: Tuple[float],
                 box_size: Tuple[float],
                 pH: float = 7.4,
                 scorefunction: str = "vina",
                 num_processes: int = 4,
                 verbose: int = 0):
        """
        Constructs Vinva scoring function with receptor.

        Arguments
        ---------
        receptor_pdbqt_file : str path to .pdbqt file of receptor.
        center : Tuple[float] (len=3) coordinates for the center of the pocket.
        box_size : Tuple[float](len=3) box edge lengths of pocket.
        pH : float Experimental pH used for crystal structure elucidation.
        scorefunction : str (default=vina) name of scoring function to use with Vina. 'vina' or 'ad4'
        num_processes : int (default=2) Number of cpus to use for scoring
        verbose : int (default = 0) Level of verbosity from vina.Vina (0 is silent)
        """
        self.v = Vina(sf_name=scorefunction, seed=987654321, verbosity=verbose, cpu=num_processes)
        self.receptor_pdbqt_file = receptor_pdbqt_file
        self.center = center
        self.box_size = box_size
        self.pH = pH
        self.v.set_receptor(rigid_pdbqt_filename=receptor_pdbqt_file)
        try:
            self.v.compute_vina_maps(center=self.center, box_size=self.box_size)
        except:
            raise ValueError(
                "Cannot compute the affinity map, please check center and box_size"
            )


    def __call__(self,
                 ligand_smiles: str,
                 output_file: Optional[str] = None,
                 exhaustiveness: int = 8,
                 n_poses: int = 5,
                 protonate: bool = False,
                 path_to_bin: str = ''
                 ) -> float:
        """
        Score ligand by docking in receptor.

        Arguments
        ---------
        ligand_smiles : str SMILES of ligand to dock.
        output_file : Optional[str] path to save docked poses.
        exhaustiveness : int (default = 8) Number of Monte Carlo simulations to run per pose.
        n_poses : int (default = 5) Number of poses to save.
        protonate : bool (default = False) (de-)protonate ligand with OpenBabel at pH=7.4
        path_to_bin : str (default = '') path to environment bin containing `mk_prepare_ligand.py`
            from `meeko` and `openbabel` if `protonate`=True

        Returns
        -------
        float : energy (affinity) in kcal/mol
        """
        try:
            if protonate:
                ligand_smiles = protonate_smiles(ligand_smiles, pH=self.pH, path_to_bin=path_to_bin)
            m = Chem.MolFromSmiles(ligand_smiles)
            m = Chem.AddHs(m)
            AllChem.EmbedMolecule(m, randomSeed=123456789)
            AllChem.MMFFOptimizeMolecule(m)
            rand = str((os.getpid() * int(time.time())) % 123456789)
            temp_mol_file = TMPDIR / f'__temp_{rand}.mol'
            print(Chem.MolToMolBlock(m), file=open(str(temp_mol_file), "w+"))
            temp_pdbqt = TMPDIR / f'__temp_{rand}.pdbqt'
            os.system(f"{path_to_bin}mk_prepare_ligand.py -i {temp_mol_file} -o {temp_pdbqt}")
            self.v.set_ligand_from_file(str(temp_pdbqt))
            self.v.dock(exhaustiveness=exhaustiveness, n_poses=n_poses)
            if output_file is not None:
                self.v.write_poses(str(output_file), n_poses=n_poses, overwrite=True)
            energy = self.v.score()[0]
            os.system(f"rm {temp_mol_file} {temp_pdbqt}")
        except Exception as e:
            print(e)
            return np.nan
        return energy


class DockingEvalPipeline:

    def __init__(self,
                 pdb_id: str,
                 num_processes: int = 4,
                 verbose: int = 0,
                 path_to_bin: str = ''):
        f"""
        Constructor for docking evaluation pipeline. Initializes VinaSmiles with receptor pdbqt.

        Arguments
        ---------
        pdb_id : str PDB ID of receptor. Currently only support: {list(docking_target_info.keys())}
        path_to_bin : str (default = '') path to environment bin containing `mk_prepare_ligand.py`
            from `meeko` and `openbabel` if `protonate`=Truepath_to_bin : str (default = '') path to `mk_prepare_ligand.py` from `meeko`.
        """
        self.pdb_id = pdb_id
        self.path_to_bin = path_to_bin
        self.vina_smiles = None
        self.smiles = []
        self.energies = []
        self.buffer = {}
        self.num_failed = 0
        self.repeats = 0

        if path_to_bin == '':
            result = subprocess.run(['which', 'mk_prepare_ligand.py'], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            stdout = result.stdout.decode().strip()

            if not stdout:
                raise FileNotFoundError(
                    f"`mk_prepare_ligand.py` that should have been installed with with meeko was not found."+\
                    "Install and run `which mk_prepare_ligand.py` and supply to `path_to_bin` argument."
                )

        if pdb_id not in list(docking_target_info.keys()):
            raise ValueError(
                f"Provided `pdb_id` ({pdb_id}) not supported. Please choose from: {list(docking_target_info.keys())}."
            )

        path_to_receptor_pdbqt = f'./pdbs/{pdb_id}.pdbqt'

        self.vina_smiles = VinaSmiles(
            receptor_pdbqt_file=path_to_receptor_pdbqt,
            center=docking_target_info[self.pdb_id]['center'],
            box_size=docking_target_info[self.pdb_id]['size'],
            pH=docking_target_info[self.pdb_id]['pH'],
            scorefunction='vina',
            num_processes=num_processes,
            verbose=verbose
        )


    def evaluate(self,
                 smiles_ls: List[str],
                 exhaustiveness: int = 32,
                 n_poses: int = 1,
                 protonate: bool = False,
                 save_poses_dir_path: Optional[str] = None,
                 verbose = False
                 ) -> List[float]:
        """
        Loop through supplied list of SMILES strings, dock, and collect energies.

        Arguments
        ---------
        smiles_ls : List[str] list of SMILES to dock
        exhaustiveness : int (default = 32) Number of Monte Carlo simulations to run per pose
        n_poses : int (default = 1) Number of poses to save
        protonate : bool (default = False) (de-)protonate ligand with OpenBabel at pH=7.4
        path_to_bin : str (default = '') path to environment bin containing `mk_prepare_ligand.py`
            from `meeko` and `openbabel` if `protonate`=True
        save_poses_dir_path : Optional[str] (default = None) Path to directory to save docked poses.

        Returns
        -------
        List of energies (affinities) in kcal/mol
        """
        save_poses_path = None
        self.smiles = smiles_ls

        if save_poses_dir_path is not None:
            dir_path = Path(save_poses_dir_path)

        energies = []
        if verbose:
            pbar = tqdm(enumerate(smiles_ls), desc=f'Docking {self.pdb_id}', total=len(smiles_ls))
        else:
            pbar = enumerate(smiles_ls)
        for i, smiles in pbar:
            if smiles in self.buffer:
                self.num_failed += 1
                self.repeats += 1
                energies.append(self.buffer[smiles])
                continue
            if smiles is None:
                energies.append(np.nan)
                self.num_failed += 1
                continue

            if save_poses_dir_path is not None:
                save_poses_path = dir_path / f'{self.pdb_id}_docked{"_prot" if protonate else ""}_{i}.pdbqt'
            try:
                energies.append(
                    self.vina_smiles(
                        ligand_smiles=smiles,
                        output_file=save_poses_path,
                        exhaustiveness=exhaustiveness,
                        n_poses=n_poses,
                        protonate=protonate,
                        path_to_bin=self.path_to_bin,
                    )
                )
                self.buffer[smiles] = float(energies[-1])
            except:
                energies.append(np.nan)
                self.buffer[smiles] = float(energies[-1])

        self.energies = energies
        return energies


    def benchmark(self,
                  exhaustiveness: int = 32,
                  n_poses: int = 5,
                  protonate: bool = False,
                  save_poses_dir_path: Optional[str] = None
                  ) -> float:
        """
        Run benchmark with experimental ligands.

        Arguments
        ---------
        exhaustiveness : int (default = 32) Number of Monte Carlo simulations to run per pose
        n_poses : int (default = 5) Number of poses to save
        protonate : bool (default = False) (de-)protonate ligand with OpenBabel at pH=7.4
        save_poses_dir_path : Optional[str] (default = None) Path to directory to save docked poses.

        Returns
        -------
        float : Energies (affinities) in kcal/mol
        """
        save_poses_path = None
        if save_poses_dir_path is not None:
            dir_path = Path(save_poses_dir_path)
            save_poses_path = dir_path / f"{self.pdb_id}_docked{'_prot' if protonate else ''}.pdbqt"

        best_energy = self.vina_smiles(
            docking_target_info[self.pdb_id]['ligand'],
            output_file=str(save_poses_path),
            exhaustiveness=exhaustiveness,
            n_poses=n_poses,
            protonate=protonate,
            path_to_bin=self.path_to_bin,
        )
        return best_energy


def run_benchmark():
    """ Run benchmark."""
    receptor_pdbs = list(docking_target_info.keys())
    for pdb in receptor_pdbs:
        print(f'{pdb}')
        print('\tProtonated')
        dep = DockingEvalPipeline(pdb_id=pdb, path_to_bin='')
        dep.benchmark(exhaustiveness=32, n_poses=5, save_poses_dir_path='./pdbs/benchmarks_1_2_5/', protonate=True)

        print('\tNot protonated')
        dep = DockingEvalPipeline(pdb_id=pdb, path_to_bin='')
        dep.benchmark(exhaustiveness=32, n_poses=5, save_poses_dir_path='./pdbs/benchmarks_1_2_5/', protonate=False)

if __name__=="__main__":
    run_benchmark()
