from __future__ import annotations

from dataclasses import dataclass
from enum import Enum, IntEnum

from PIL import Image, ImageDraw

HSV = 'tuple[int, int, int]'


class Shape(Enum):
    ELLIPSE = 0
    RECTANGLE = 1
    TRIANGLE = 2


class Color(IntEnum):
    RED = 0
    GREEN = 85
    BLUE = 170


class Position(float, Enum):
    START = 1/4
    MIDDLE = 2/4
    END = 3/4


class Size(float, Enum):
    SMALL = 1/4
    MEDIUM = 2/4
    LARGE = 3/4


@dataclass
class SpriteWorld:
    size: int
    margin: int
    background: HSV

    hue_variance: int
    saturation_range: tuple[int, int]
    value_range: tuple[int, int]

    width_variance: int
    height_variance: int

    x_variance: int
    y_variance: int

    def draw(self,
             shape: Shape,
             fill: HSV,
             width: int,
             height: int,
             x: int,
             y: int):
        image = Image.new('HSV', (self.size, self.size), self.background)
        draw = ImageDraw.Draw(image)

        left = max(self.margin, x - width // 2)
        right = min(self.size - self.margin, left + width)

        top = max(self.margin, y - height // 2)
        bottom = min(self.size - self.margin, top + height)

        if shape == Shape.ELLIPSE:
            draw.ellipse((left, top, right, bottom), fill)
        elif shape == Shape.RECTANGLE:
            draw.rectangle((left, top, right, bottom), fill)
        elif shape == Shape.TRIANGLE:
            draw.polygon(
                (((left + right) // 2, top), (left, bottom), (right, bottom)),
                fill
            )
        else:
            raise ValueError('Unknown shape')
        return image

    def sample(self, rng):
        shape = rng.choice(list(Shape))
        color = rng.choice(list(Color))
        width = rng.choice(list(Size))
        height = rng.choice(list(Size))
        x = rng.choice(list(Position))
        y = rng.choice(list(Position))

        attributes = (shape, color, width, height, x, y)
        properties = (
            shape,
            ((color+rng.randint(-self.hue_variance, self.hue_variance)) % 256,
             rng.randint(*self.saturation_range),
             rng.randint(*self.value_range)),
            int(self.size * width) + rng.randint(-self.width_variance,
                                                 self.width_variance),
            int(self.size * height) + rng.randint(-self.height_variance,
                                                  self.height_variance),
            int(self.size * x) + rng.randint(-self.x_variance,
                                             self.x_variance),
            int(self.size * y) + rng.randint(-self.y_variance,
                                             self.y_variance)
        )
        return {'attributes': attributes,
                'properties': properties,
                'image': self.draw(*properties)}
