import os
import torch
from pathlib import Path
from diffusers import StableDiffusionPipeline, UNet2DConditionModel
from tqdm import tqdm

def setup_pipeline():
    """Set up and initialize Stable Diffusion pipeline with custom UNet model."""
    # Configure CUDA device
    os.environ["CUDA_VISIBLE_DEVICES"] = "0"
    device = torch.device("cuda:0" if torch.cuda.is_available() else "cpu")
    
    # Load the model
    unet = UNet2DConditionModel.from_pretrained(
        "DwanZhang/SePPO", 
        torch_dtype=torch.float16
    ).to(device)

    inference_dtype = torch.float16
    pipeline = StableDiffusionPipeline.from_pretrained(
        "pt-sk/stable-diffusion-1.5", 
        torch_dtype=inference_dtype
    )
    
    pipeline.safety_checker = None
    pipeline = pipeline.to(device)
    pipeline.unet = unet
    
    return pipeline

def generate_image(prompt, pipeline, seed=42):
    """Generate a single image."""
    generator = torch.Generator(device='cuda:0').manual_seed(seed)
    image = pipeline(
        prompt=prompt,
        generator=generator,
        guidance_scale=7.5
    ).images[0]
    return image

def load_prompts(file_path):
    """Load prompts from file"""
    with open(file_path, 'r', encoding='utf-8') as f:
        return [line.strip() for line in f if line.strip()]

def main():
    # Set the path
    input_file = './GenEval.txt'
    save_dir = Path('./Generation/SePPO')
    
    # Create save directory
    save_dir.mkdir(parents=True, exist_ok=True)
    
    # Initialize the pipeline
    pipeline = setup_pipeline()
    
    prompts = load_prompts(input_file)
    print(f"Loaded {len(prompts)} prompts")
    
    # Generation
    for idx, prompt in enumerate(tqdm(prompts, desc="Generating images")):
        image_path = save_dir / f"img{idx}.jpg"
        try:
            image = generate_image(prompt, pipeline)
            image.save(image_path)
        except Exception as e:
            print(f"Error generating image {idx}: {e}")

if __name__ == "__main__":
    main()