import os
import torch
import numpy as np
from pathlib import Path
from PIL import Image
from statistics import mean
import ImageReward as RM

def load_prompts(file_path):
    """Load prompts from text file"""
    with open(file_path, 'r', encoding='utf-8') as f:
        return [line.strip() for line in f]

def calculate_scores(prompts, image_dir, model):
    """Calculate ImageReward"""
    scores = []
    for idx, prompt in enumerate(prompts):
        image_path = os.path.join(image_dir, f"img{idx}.jpg")
        try:
            image = Image.open(image_path)
            score = model.score(prompt, image)
            scores.append(score)
            print(f"Score for image {idx}: {score}")
        except Exception as e:
            print(f"Error processing image {idx}: {e}")
    return scores

def print_statistics(scores):
    """Print statistical analysis of scores"""
    mean_score = mean(scores)
    std_score = np.std(scores, ddof=1)
    separator = '*' * 50

    print(f"Complete score list: {scores}")
    print(f"Mean score: {mean_score:.4f}")
    print(f"Standard deviation: {std_score:.4f}")
    print(f"\n{separator}Results Summary{separator}")
    print(f"Mean score: {mean_score:.4f}")
    print(f"Standard deviation: {std_score:.4f}")

def main():
    torch.set_grad_enabled(False)
    inference_dtype = torch.float16
    
    # Setup the path
    base_dir = Path("./evaluating")
    prompt_file = base_dir / "image_generation/DPG-Bench/DPG-Bench.txt"
    image_dir = base_dir / "image_generation/DPG-Bench/Generation/RainbowPA"
    
    # Load prompts
    prompts = load_prompts(prompt_file)
    print(f"Loaded {len(prompts)} prompts")
    
    # Initialization
    model = RM.load("ImageReward-v1.0", device='cuda:0')
    
    # Calculate scores
    scores = calculate_scores(prompts, image_dir, model)
    
    # Print statistics
    print_statistics(scores)

if __name__ == "__main__":
    main()