import os
import torch
import numpy as np
from pathlib import Path
from statistics import mean
import t2v_metrics

def load_prompts(file_path):
    """Load prompts from text file"""
    with open(file_path, 'r', encoding='utf-8') as f:
        return [line.strip() for line in f]

def calculate_scores(prompts, image_dir, scorer):
    """Calculate scores"""
    scores = []
    for idx, prompt in enumerate(prompts):
        image_path = os.path.join(image_dir, f"img{idx}.jpg")
        try:
            score = scorer(images=[image_path], texts=[prompt])
            score_value = score.item()
            scores.append(score_value)
            print(f"Score for image {idx}: {score_value}")
        except Exception as e:
            print(f"Error processing image {idx}: {e}")
    return scores

def print_statistics(scores):
    """Print statistical analysis of scores"""
    mean_score = mean(scores)
    std_score = np.std(scores, ddof=1)
    separator = '*' * 50
    
    print(f"Complete score list: {scores}")
    print(f"Mean score: {mean_score:.4f}")
    print(f"Standard deviation: {std_score:.4f}")
    print(f"\n{separator}Results Summary{separator}")
    print(f"Mean score: {mean_score:.4f}")
    print(f"Standard deviation: {std_score:.4f}")

def main():
    torch.set_grad_enabled(False)
    
    # Setup the path
    base_dir = Path("./evaluating")
    prompt_file = base_dir / "image_generation/T2I-Com/T2I-Com/texture.txt"
    image_dir = base_dir / "image_generation/T2I-Com/Generation/RainbowPA/texture"
    
    # Load prompts
    example_prompts = load_prompts(prompt_file)
    print(f"Loaded {len(example_prompts)} prompts")
    
    # Calculate scores
    clip_flant5_score = t2v_metrics.VQAScore(model='clip-flant5-xxl', device='cuda:0')
    
    scores = calculate_scores(example_prompts, image_dir, clip_flant5_score)
    
    # Print statistics
    print_statistics(scores)

if __name__ == "__main__":
    main()