import os
import torch
from pathlib import Path
from diffusers import StableDiffusionPipeline, UNet2DConditionModel
from tqdm import tqdm

def setup_pipeline():
    """Set up and initialize Stable Diffusion pipeline with custom UNet"""
    # Configure CUDA device
    os.environ["CUDA_VISIBLE_DEVICES"] = "0"
    device = torch.device("cuda:0" if torch.cuda.is_available() else "cpu")
    
    # Load custom UNet model
    unet = UNet2DConditionModel.from_pretrained(
        "UCLA-AGI/SPIN-Diffusion-iter3",
        torch_dtype=torch.float16 
    ).to(device)
    
    # Load base diffusion model
    pipeline = StableDiffusionPipeline.from_pretrained(
        "runwayml/stable-diffusion-v1-5",
        torch_dtype=torch.float16,
        safety_checker=None
    )
    
    # Replace UNet and move to device
    pipeline.unet = unet
    return pipeline.to(device)

def generate_image(prompt, pipeline, seed=42):
    """Generate a single image"""
    generator = torch.Generator(device='cuda:0').manual_seed(seed)
    image = pipeline(
        prompt=prompt,
        generator=generator,
        guidance_scale=7.5
    ).images[0]
    return image

def load_prompts(file_path):
    """Load prompts from file"""
    with open(file_path, 'r') as f:
        return [line.strip() for line in f]

def main():
    # Set the path
    input_file = './DPG-Bench.txt'
    save_dir = Path('./Generation/SPIN-Diffusion')
    
    # Create save directory
    save_dir.mkdir(parents=True, exist_ok=True)
    
    # Initialize the pipeline
    pipeline = setup_pipeline()
    
    # Load prompts
    prompts = load_prompts(input_file)
    print(f"Loaded {len(prompts)} prompts")
    
    # Generate images
    for idx, prompt in enumerate(tqdm(prompts, desc="Generating images")):
        image_path = save_dir / f"img{idx}.jpg"
        try:
            image = generate_image(prompt, pipeline)
            image.save(image_path)
        except Exception as e:
            print(f"Error generating image {idx}: {e}")

if __name__ == "__main__":
    main()