import os
import torch
from pathlib import Path
from diffusers import StableDiffusionPipeline, UNet2DConditionModel
from tqdm import tqdm

def setup_pipeline():
    """Set up and initialize Stable Diffusion pipeline with SPIN-Diffusion UNet"""
    # Configure CUDA device
    os.environ["CUDA_VISIBLE_DEVICES"] = "0"
    device = torch.device("cuda:0" if torch.cuda.is_available() else "cpu")
    inference_dtype = torch.float16
    
    # Load the model
    unet = UNet2DConditionModel.from_pretrained(
        "UCLA-AGI/SPIN-Diffusion-iter3",
        torch_dtype=inference_dtype
    ).to(device)
    
    pipeline = StableDiffusionPipeline.from_pretrained(
        "runwayml/stable-diffusion-v1-5",
        torch_dtype=inference_dtype,
    )
    
    pipeline.safety_checker = None
    pipeline = pipeline.to(device)
    
    pipeline.unet = unet
    
    return pipeline

def generate_image(prompt, pipeline, seed=42):
    """Generate a single image"""
    generator = torch.Generator(device='cuda:0').manual_seed(seed)
    image = pipeline(
        prompt=prompt,
        generator=generator,
        guidance_scale=7.5
    ).images[0]
    return image

def load_prompts(file_path):
    """Load prompts from text file"""
    prompts = []
    with open(file_path, 'r', encoding='utf-8') as f:
        for line in f:
            prompts.append(line.strip())
    return prompts

def main():
    # Set the path; when you need to generate images of a new category, you need to change the path
    prompt_file = './T2I-Com/texture.txt'
    save_dir = Path('./Generation/SPIN-Diffusion/texture')
    
    # Create save directory
    save_dir.mkdir(parents=True, exist_ok=True)
    
    # Initialize pipeline
    pipeline = setup_pipeline()
    
    # Load prompts
    prompts = load_prompts(prompt_file)
    print(f"Loaded {len(prompts)} prompts")
    
    # Generation
    for idx, prompt in enumerate(tqdm(prompts, desc="Generating texture images")):
        image_path = save_dir / f"img{idx}.jpg"
        try:
            image = generate_image(prompt, pipeline)
            image.save(image_path)
        except Exception as e:
            print(f"Error generating image {idx}: {e}")

if __name__ == "__main__":
    main()