import argparse
from argparse import Namespace

from tqdm.notebook import tqdm
import sys

usr_name = "none"
sys.path.insert(2, f"/home/{usr_name}/workspace/kergnns")

from model import kergnn

from distances import compare_size, iou_distance, weighted_ged_distance
from utils import *
import torch.nn.functional as F
from torch.utils.data import DataLoader
from torch_geometric.loader import DataLoader as GDataLoader

from torch.utils.data import Subset
from torch_geometric.utils import dense_to_sparse, to_dense_adj, subgraph
from torch_geometric.data import Data
import numpy as np

KERGNN_PATH = f"/home/{usr_name}/workspace/kergnns"
METRICS_NAMES = list(METRICS.keys())
device = torch.device("cuda" if torch.cuda.is_available() else "cpu")


def args_parser():
    parser = argparse.ArgumentParser()
    parser.add_argument("--dataset", type=str, required=True, help="Dataset name")
    parser.add_argument("--split", type=int, default=-1, help="Split index")
    parser.add_argument("--seed", type=int, default=0, help="Seed")
    parser.add_argument(
        "--metrics",
        type=str,
        nargs="+",
        choices=METRICS_NAMES,
        default=METRICS_NAMES,
        help="Metrics to evaluate",
    )
    parser.add_argument("--batch_size", type=int, default=32, help="Batch size")
    parser.add_argument("--node_mask_fn", type=str, default="elbow_softmax:0", help="Thresholding of nodes")
    parser.add_argument("--edge_mask_fn", type=str, default="none", help="Thresholding of edges")
    args = parser.parse_args()
    return args


class Classifier(torch.nn.Module):
    def __init__(self, model):
        super(Classifier, self).__init__()
        self.model = copy.deepcopy(model)

    def forward(self, x):
        preds = 0
        idx = 0
        for layer in range(len(self.model.linears_prediction)):
            try:
                in_features = self.model.linears_prediction[layer].linear.in_features
            except:
                in_features = self.model.linears_prediction[layer].linears[0].in_features
            h = x[:, idx : idx + in_features]
            # idx += in_features
            preds += F.dropout(
                self.model.linears_prediction[layer](h), self.model.dropout_rate, training=False
            )
        return preds


def forward(model, classifier, data):

    x, adj, nidxs, batch = data.x, data.adj, data.nidx, data.batch
    nidx = nidxs.clone()
    nidx[nidx < 0] = len(x)

    unique, counts = torch.unique(batch, return_counts=True)
    n_graphs = unique.size(0)

    responses = [x]
    mlp_inputs = [x]
    h = x

    for layer in range(model.num_layers):
        h = model.ker_layers[layer](adj, h, nidx)
        responses.append(h)
        h = model.batch_norms[layer](h)
        h = F.relu(h)
        mlp_inputs.append(h)

    def pool(x):
        pooled = list()
        for layer, h in enumerate(x):
            pooled_h = torch.zeros(batch.max() + 1, h.shape[1], device=device).index_add_(0, batch, h)
            if not model.no_norm:
                norm = counts.unsqueeze(1).repeat(1, pooled_h.shape[1])
                pooled_h = pooled_h / norm
            pooled += [pooled_h]
        return torch.cat(pooled, -1)

    pooled_mlp_inputs = pool(mlp_inputs)
    pooled_responses = pool(responses)
    mlp_inputs = torch.cat(mlp_inputs, dim=1)
    responses = torch.cat(responses, dim=1)
    preds = classifier(pooled_mlp_inputs)
    return preds, responses, pooled_responses, mlp_inputs, pooled_mlp_inputs, batch


def get_responses_and_logits(model, dataloader, forward_fn):
    logits, responses = list(), list()
    for data in dataloader:
        data = data.to(device)
        l, r = forward_fn(model, data)
        responses.append(r), logits.append(l)
    return torch.cat(responses, 0), torch.cat(logits)


def get_truncated_explanations(explanations, node_mask_fn, edge_mask_fn):
    def _get_truncated_explanations(e):
        subset, edge_index = extract_subset_and_edge_index(
            e.edge_index,
            len(e.x) - 1,
            e.node_mask.sum(1)[:-1],
            node_mask_fn,
            e.edge_mask.sum(1),
            edge_mask_fn,
        )
        edge_index, _ = subgraph(subset, edge_index, relabel_nodes=True, num_nodes=len(e.x))
        y_pred = e.pred.argmax(keepdim=True)[0]
        return Data(edge_index=edge_index, x=e.x[subset], y=y_pred, nodes_subset=subset)

    return list(map(_get_truncated_explanations, explanations))


def get_prototypes(model, dataset):
    prototypes = list()
    all_x = torch.cat([data.x for data in dataset])
    device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
    possible_x = torch.unique(all_x, sorted=False, dim=0).to(device)

    for ker_layer in model.ker_layers:
        if ker_layer.hidden_dim:
            encoded_possible_x = torch.nn.ReLU()(ker_layer.fc_in(possible_x))
        x = ker_layer.features_hidden.permute(0, 2, 1)
        x_shape = x.shape
        x = x.reshape(-1, x.shape[-1])
        sim = encoded_possible_x @ x.T
        x = possible_x[sim.argmax(0)]
        x = x.reshape(x_shape[0], x_shape[1], -1)
        adj_hidden_norm = torch.zeros(
            ker_layer.size_graph_filter, ker_layer.size_graph_filter, ker_layer.out_dim
        ).to(device)
        idx = torch.triu_indices(ker_layer.size_graph_filter, ker_layer.size_graph_filter, 1)
        adj_hidden_norm[idx[0], idx[1], :] = ker_layer.relu(ker_layer.adj_hidden)
        adj = adj_hidden_norm + torch.transpose(adj_hidden_norm, 0, 1)
        adj[adj < 0] = 0
        for i in range(adj.shape[0]):
            edge_index = dense_to_sparse(adj[i] > 0.0)[0]
            edge_attr = torch.zeros(edge_index.shape[1]).to(edge_index.device)
            for j in range(edge_index.shape[1]):
                edge_attr[j] = float(min(adj[i, edge_index[0, j], edge_index[1, j]], 1) > 0)
            prototypes.append(Data(edge_index=edge_index, x=x[i], edge_attr=edge_attr))
    return prototypes


def perturb_model(model, dataset, p_x=0.2, p_edge=0.2):
    model_copy = copy.deepcopy(model)
    model_copy.eval()
    all_x = torch.cat([data.x for data in dataset])

    for ker_layer in model_copy.ker_layers:
        x = ker_layer.features_hidden.detach()
        adj = ker_layer.adj_hidden.detach()
        if p_x > 0:
            x = x.permute((0, 2, 1))
            x_shape = x.shape
            x = x.reshape(-1, x.shape[-1])
            keys = torch.tensor(np.random.choice([0, 1], p=[1 - p_x, p_x], size=len(x))).bool()
            size = keys.sum().item()
            if size > 0:
                x[keys] = torch.nn.ReLU()(
                    ker_layer.fc_in(all_x[np.random.randint(low=0, high=len(all_x), size=size)].to(x.device))
                )

            ker_layer.features_hidden.data = x.reshape(x_shape[0], x_shape[1], -1).permute((0, 2, 1))
        if p_edge > 0:
            adj_shape = adj.shape
            adj = adj.reshape(-1)
            keys = (
                torch.tensor(np.random.choice([0, 1], p=[1 - p_edge, p_edge], size=len(adj)))
                .bool()
                .to(adj.device)
            )
            adj[keys] = (adj[keys] < 0).float()
            ker_layer.adj_hidden.data = adj.reshape(adj_shape)
    return model_copy


def load_model(path, dataset, degree_attr):
    params = torch.load(path, map_location=torch.device("cpu"))
    if "args" in params:
        loaded_args = params["args"]
    else:
        loaded_args = {
            "hidden_dims": [16, 32],
            "kernel": "wl",
            "max_stop": 1,
            "num_mlp_layers": 1,
            "mlp_hidden_dim": 16,
            "dropout_rate": 0.4,
            "size_graph_filter": [6],
            "size_subgraph": 10,
            "no_norm": dataset in ["MUTAG", "PROTEINS", "NCI1"],
        }

    loaded_args = Namespace(**loaded_args)
    if degree_attr:
        features_dim = 1
    else:
        if hasattr(dataset, "x"):
            features_dim = dataset.x.shape[1]
        else:
            features_dim = 10
    n_classes = dataset.y.max() + 1

    # Create models
    model = kergnn(
        features_dim,
        n_classes,
        hidden_dims=loaded_args.hidden_dims,
        kernel=loaded_args.kernel,
        max_step=loaded_args.max_step,
        num_mlp_layers=loaded_args.num_mlp_layers,
        mlp_hidden_dim=loaded_args.mlp_hidden_dim,
        dropout_rate=loaded_args.dropout_rate,
        size_graph_filter=loaded_args.size_graph_filter,
        size_subgraph=loaded_args.size_subgraph,
        no_norm=loaded_args.no_norm,
    )
    model.load_state_dict(params["state_dict"])
    return model, loaded_args


def k_hop_n(node_idx, num_hops, edge_index, max_size, num_nodes, relabel_nodes=True):
    device = edge_index.device
    col, row = edge_index
    node_mask = row.new_empty(num_nodes, dtype=torch.bool, device=device)
    edge_mask = row.new_empty(row.size(0), dtype=torch.bool, device=device)
    subset = torch.tensor([node_idx], dtype=torch.long, device=device)
    for _ in range(num_hops):
        node_mask.fill_(False)
        node_mask[subset] = True
        torch.index_select(node_mask, 0, row, out=edge_mask)
        s = col[edge_mask]
        s = torch.cat((subset, s))
        _, inverse = np.unique(s.cpu(), return_index=True)
        subset = torch.tensor([s[index.item()] for index in sorted(inverse)], dtype=torch.long, device=device)
        if len(subset) > max_size:
            break

    subset = subset[:max_size]

    node_mask.fill_(False)
    node_mask[subset] = True

    edge_mask = node_mask[row] & node_mask[col]
    edge_index = edge_index[:, edge_mask]
    if relabel_nodes:
        mapping = row.new_full((num_nodes,), -1)
        mapping[subset] = torch.arange(subset.size(0), device=row.device)
        edge_index = mapping[edge_index]

    return subset, edge_index, edge_mask, mapping


def extract_subgraphs(edge_index, x, k, subgraph_size):
    device = edge_index.device
    num_nodes = len(x)

    def _k_hop_subgraph(idx):
        nidx, sub_edge_index, _, mapping = k_hop_n(
            idx, num_hops=k, edge_index=edge_index, max_size=subgraph_size, num_nodes=num_nodes
        )
        sub_adj = to_dense_adj(sub_edge_index, max_num_nodes=subgraph_size)
        return nidx, sub_adj, mapping

    subgraphs = [_k_hop_subgraph(idx) for idx in range(len(x))]
    nidx, adj, mapping = zip(*subgraphs)
    adj = torch.cat(adj)
    nidx = [torch.nn.functional.pad(n, pad=(0, subgraph_size - len(n)), value=-1) for n in nidx]
    nidx = torch.stack(nidx)
    return x, adj, nidx


def collate_fn(batch):
    graph_indicator = torch.cat(
        tuple(i * torch.ones(len(b.x), dtype=torch.long, device=device) for i, b in enumerate(batch))
    )
    x = torch.cat([b.x for b in batch]).to(device)
    adj = torch.cat([b.adj for b in batch]).to(device)
    nidx = torch.cat([b.nidx for b in batch]).to(device)
    y = torch.cat([b.y for b in batch]).to(device)

    _, counts = torch.unique(graph_indicator, return_counts=True)
    counts_sum = torch.cumsum(counts, dim=0)[:-1].long()
    nidx_diff = torch.zeros_like(graph_indicator, device=device)
    nidx_diff[counts_sum.long()] = counts[:-1]
    nidx_diff = torch.cumsum(nidx_diff, dim=0)
    nidx[nidx >= 0] = (nidx + nidx_diff.unsqueeze(-1))[nidx >= 0]

    edge_index = list()
    num_nodes = 0
    for b in batch:
        edge_index.append(b.edge_index + num_nodes)
        num_nodes += len(b.x)
    edge_index = torch.cat(edge_index, dim=-1)

    return Data(x=x, edge_index=edge_index, adj=adj, nidx=nidx, batch=graph_indicator, y=y)


def get_explanations(model, explainer, dataloader, forward_fn):
    explanations = list()
    device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
    for data in dataloader:
        data = data.to(device)
        num_features = data.x.shape[-1]
        with torch.no_grad():
            logits, all_responses, response = forward_fn(model, data)
        pred = logits.argmax(dim=-1)
        importance = run_shap(explainer, response).to(device)
        importance = torch.stack([importance[pred[i]][i] for i in range(len(pred))], dim=0).to(device)
        response[response == 0] = 1e-36

        num_nodes = 0
        for b in tqdm(range(len(data.y)), total=len(data.y)):
            mask = data.batch == b
            im, r, ar = importance[b], response[b], all_responses[mask]
            im = (im / r) * ar
            im_features = im[:, :num_features]
            im = im[:, num_features:]

            nidx = data.nidx[mask] - num_nodes
            x = data.x[mask]

            edge_index = data.edge_index[:, data.batch[data.edge_index[0]] == b] - num_nodes
            num_nodes += len(x)
            node_mask = torch.zeros(len(x), im.shape[-1] + im_features.shape[-1], device=device)
            edge_mask = torch.zeros(edge_index.shape[1], im.shape[-1], device=device)
            for i in range(len(nidx)):
                nidx_i = i + 0 * nidx[i, 0:1]
                norm = len(nidx_i)
                if norm == 0:
                    continue
                node_mask[nidx_i, num_features:] += im[i] / norm
            for i in range(len(x)):
                node_mask[i, :num_features] += im_features[i]
            e = Data(
                x=x.cpu(),
                edge_index=edge_index.cpu(),
                y=data.y[b : b + 1].cpu(),
                pred=logits[b : b + 1].cpu(),
                node_mask=node_mask.cpu(),
                edge_mask=edge_mask.cpu(),
            )
            explanations.append(e)
    return explanations


def main():
    args = args_parser()
    print(args)
    set_seed(args.seed)
    degree_attr = args.dataset in ["REDDIT-BINARY", "IMDB-BINARY", "IMDB-MULTI"]
    dataset = get_dataset(args.dataset, degree_attr=degree_attr, use_node_attr=True)
    s = args.seed if args.split < 0 else args.split

    device = torch.device("cuda" if torch.cuda.is_available() else "cpu")
    model_path = f"{KERGNN_PATH}/save/{args.dataset}/cv{s}/prefix_val_model.pth.tar"
    model, loaded_args = load_model(model_path, dataset, degree_attr)
    model = model.to(device)
    model.eval()
    classifier = Classifier(model).to(device)
    classifier.eval()
    model = (model, classifier)

    def preprocess_graph(batch):
        x, adj, nidx = extract_subgraphs(batch.edge_index, batch.x, loaded_args.k, loaded_args.size_subgraph)
        if args.dataset.startswith("BA"):
            x = 0.1 * torch.ones_like(x).to(x.device)
        data = Data(
            x=x,
            edge_index=batch.edge_index,
            edge_attr=batch.edge_attr,
            adj=adj,
            nidx=nidx,
            y=batch.y,
        )
        if hasattr(batch, "explanation_graph"):
            data.explanation_graph = batch.explanation_graph
            data.explanation_nodes_subset = batch.explanation_nodes_subset
        return data

    def preprocess_dataset(dataset):
        return list(map(preprocess_graph, dataset))

    train_idxs, val_idxs, test_idxs = get_splits(
        args.dataset, size=len(dataset), seed=args.seed, split=args.split
    )

    dataset_train, dataset_val, dataset_test = (
        Subset(dataset, train_idxs),
        Subset(dataset, val_idxs),
        Subset(dataset, test_idxs),
    )

    dataset_test = preprocess_dataset(dataset_test)
    dataloader_test = DataLoader(
        dataset_test, batch_size=args.batch_size, shuffle=False, collate_fn=collate_fn
    )

    forward_fn = lambda model, x: forward(model[0], model[1], x)[0]

    results = {}

    node_mask_fn = get_fn(args.node_mask_fn)
    edge_mask_fn = get_fn(args.edge_mask_fn)

    if "A1" in args.metrics:
        results["A1"] = get_acc(model, dataloader_test, forward_fn)
        print_results(results)

    if args.metrics == ["A1"]:
        print("Final")
        print_results(results)
        exit(0)

    def logits_and_responses_fn(model, data):
        model, classifier = model
        logits, _, _, _, responses, _ = forward(model, classifier, data)
        return logits, responses

    def logits_and_all_responses_fn(model, data):
        model, classifier = model
        logits, _, _, all_responses, responses, _ = forward(model, classifier, data)
        return logits, all_responses, responses

    dataloader = DataLoader(
        preprocess_dataset(dataset), batch_size=args.batch_size, shuffle=False, collate_fn=collate_fn
    )
    responses, logits = get_responses_and_logits(model, dataloader, logits_and_responses_fn)

    shap_values, explainer = get_shap(classifier, responses)

    explanations = get_explanations(model, explainer, dataloader_test, logits_and_all_responses_fn)
    truncated_explanations = get_truncated_explanations(explanations, node_mask_fn, edge_mask_fn)
    prototypes = get_prototypes(model[0], dataset)

    if "A1" in args.metrics:
        distances = compare_with_gt_instance(args.dataset, dataset_test, truncated_explanations, iou_distance)
        if len(distances) == 0:
            results["A1"] = None
        else:
            results["A1"] = 1 - np.mean(distances)
        print_results(results)

    if "A2" in args.metrics:
        ground_truths = get_gt_explanations_model(args.dataset)
        if len(ground_truths) > 0:
            distances = compare_with_gt_model(prototypes, ground_truths, weighted_ged_distance)
            results["A2"] = 1 - np.mean(distances)
        else:
            results["A2"] = None
        print_results(results)

    if bool({"I1", "I2"} & set(args.metrics)):
        with_explanations, without_explanations = remove_explanation_perturb(
            explanations, node_mask_fn, edge_mask_fn, skip_last_node=True
        )
        if "I1" in args.metrics:
            with_explanations = [data for data in with_explanations if data is not None]
            with_explanations = preprocess_dataset(with_explanations)
            dataloader_with = DataLoader(
                with_explanations, batch_size=args.batch_size, shuffle=False, collate_fn=collate_fn
            )
            results["I1"] = get_acc(model, dataloader_with, forward_fn)
            print_results(results)
        if "I2" in args.metrics:
            without_explanations = [data for data in without_explanations if data is not None]
            without_explanations = preprocess_dataset(without_explanations)
            dataloader_without = DataLoader(
                without_explanations, batch_size=args.batch_size, shuffle=False, collate_fn=collate_fn
            )
            acc_without = 1 - get_acc(model, dataloader_without, forward_fn)
            results["I2"] = acc_without * len(without_explanations) / len(explanations)
            print_results(results)

    if "I3" in args.metrics:
        _, new_explainer = get_shap(classifier, responses)
        new_explanations = get_explanations(
            model, new_explainer, dataloader_test, logits_and_all_responses_fn
        )
        new_truncated_explanations = get_truncated_explanations(new_explanations, node_mask_fn, edge_mask_fn)
        distances = compare_lists(truncated_explanations, new_truncated_explanations, iou_distance)
        results["I3"] = 1 - np.mean(distances)
        print_results(results)

    if "I4" in args.metrics:
        dataset_noisy_node = add_noise_perturb(dataset_test, p_x=0.05, p_edges_add=0.0, p_edges_del=0.0, explanations=truncated_explanations)
        dataloader_new = DataLoader(
            preprocess_dataset(dataset_noisy_node),
            batch_size=args.batch_size,
            shuffle=False,
            collate_fn=collate_fn,
        )
        new_explanations = get_explanations(model, explainer, dataloader_new, logits_and_all_responses_fn)
        new_truncated_explanations = get_truncated_explanations(new_explanations, node_mask_fn, edge_mask_fn)
        distances = compare_lists(truncated_explanations, new_truncated_explanations, iou_distance)
        results["I4"] = 1 - np.mean(distances)
        print_results(results)

    if "I5" in args.metrics:
        dataset_noisy_edge = add_noise_perturb(dataset_test, p_x=0.0, p_edges_add=0.005, p_edges_del=0.005, explanations=truncated_explanations)
        dataloader_new = DataLoader(
            preprocess_dataset(dataset_noisy_edge),
            batch_size=args.batch_size,
            shuffle=False,
            collate_fn=collate_fn,
        )
        new_explanations = get_explanations(model, explainer, dataloader_new, logits_and_all_responses_fn)
        new_truncated_explanations = get_truncated_explanations(new_explanations, node_mask_fn, edge_mask_fn)
        distances = compare_lists(truncated_explanations, new_truncated_explanations, iou_distance)
        results["I5"] = 1 - np.mean(distances)
        print_results(results)

    if "I6" in args.metrics:
        dataloader_train = DataLoader(
            preprocess_dataset(dataset_train), batch_size=args.batch_size, shuffle=True, collate_fn=collate_fn
        )
        dataloader_val = DataLoader(
            preprocess_dataset(dataset_val), batch_size=args.batch_size, shuffle=False, collate_fn=collate_fn
        )

        explanations_train = get_explanations(model, explainer, dataloader_train, logits_and_all_responses_fn)
        truncated_explanations_train = get_truncated_explanations(
            explanations_train, node_mask_fn, edge_mask_fn
        )
        explanations_val = get_explanations(model, explainer, dataloader_val, logits_and_all_responses_fn)
        truncated_explanations_val = get_truncated_explanations(explanations_val, node_mask_fn, edge_mask_fn)
        new_dataloader_train = GDataLoader(
            truncated_explanations_train, batch_size=args.batch_size, shuffle=True
        )
        new_dataloader_val = GDataLoader(
            truncated_explanations_val, batch_size=args.batch_size, shuffle=False
        )
        new_dataloader_test = GDataLoader(truncated_explanations, batch_size=args.batch_size, shuffle=False)
        model_args = torch.load(f"models/{args.dataset}_s{s}.pt", map_location=torch.device("cpu"))["args"]
        model_args["num_node_features"] = 1 if degree_attr else dataset_test[0].x.shape[-1]

        new_model = get_model(**model_args).to(device)
        new_model = train_model(
            new_model,
            new_dataloader_train,
            new_dataloader_val,
            new_dataloader_test,
            epochs=500 if args.dataset == "MUTAG" else 200,
            lr=0.001,
            weight_decay=0.005,
        )
        new_model.eval()
        results["I6"] = get_acc(
            new_model,
            new_dataloader_test,
            lambda model, data: model(data.x, data.edge_index, batch=data.batch),
        )
        print_results(results)

    if "I7" in args.metrics:
        sizes = compare_lists(dataset_test, truncated_explanations, compare_size)
        results["I7"] = 1 - np.mean(sizes)
        print_results(results)

    if "M1" in args.metrics:
        try:
            perturbed_model = perturb_model(model[0], dataset, p_x=0.5, p_edge=0)
            perturbed_responses, perturbed_logits = get_responses_and_logits(
                (perturbed_model, model[1]), dataloader, logits_and_responses_fn
            )
            _, perturbed_explainer = get_shap(model[1], perturbed_responses)
            perturbed_explanations = get_explanations(
                (perturbed_model, model[1]),
                perturbed_explainer,
                dataloader_test,
                logits_and_all_responses_fn,
            )
            perturbed_explanations = get_truncated_explanations(
                perturbed_explanations, node_mask_fn, edge_mask_fn
            )
            distances = compare_lists(truncated_explanations, perturbed_explanations, iou_distance)
            results["M1"] = np.mean(distances)
        except:
            results["M1"] = 0
        print_results(results)

    if "M2" in args.metrics:
        try:
            perturbed_model = perturb_model(model[0], dataset, p_x=0, p_edge=0.25)
            perturbed_responses, perturbed_logits = get_responses_and_logits(
                (perturbed_model, model[1]), dataloader, logits_and_responses_fn
            )
            _, perturbed_explainer = get_shap(model[1], perturbed_responses)
            perturbed_explanations = get_explanations(
                (perturbed_model, model[1]),
                perturbed_explainer,
                dataloader_test,
                logits_and_all_responses_fn,
            )
            perturbed_explanations = get_truncated_explanations(
                perturbed_explanations, node_mask_fn, edge_mask_fn
            )
            distances = compare_lists(truncated_explanations, perturbed_explanations, iou_distance)
            results["M2"] = np.mean(distances)
        except:
            results["M2"] = 0
        print_results(results)

    if "M3" in args.metrics:
        results["M3"] = 1 - pairwise_list_corr(responses.cpu().detach().numpy(), to_abs=True)
        print_results(results)

    print("Final")
    print_results(results)


if __name__ == "__main__":
    main()
