import argparse

from tqdm.notebook import tqdm
import sys

usr_dir = "none"
sys.path.insert(2, f"/home/{usr_dir}/workspace/ProtGNN")

import load_dataset
from models import GnnNets
from my_mcts import mcts

from torch.utils.data import Subset
from torch_geometric.loader import DataLoader
from torch_geometric.utils import dense_to_sparse

from distances import iou_distance, compare_size, weighted_ged_distance
from utils import *

PROTGNN_PATH = f"/home/{usr_dir}/workspace/ProtGNN"
METRICS_NAMES = list(METRICS.keys())
device = torch.device("cuda" if torch.cuda.is_available() else "cpu")


def args_parser():
    parser = argparse.ArgumentParser()
    parser.add_argument("--dataset", type=str, required=True, help="Dataset name")
    parser.add_argument("--split", type=int, default=-1, help="Split index")
    parser.add_argument("--seed", type=int, default=0, help="Seed")
    parser.add_argument(
        "--metrics", type=str, nargs="+", choices=METRICS_NAMES, default=METRICS_NAMES, help="Metrics to evaluate"
    )
    parser.add_argument("--batch_size", type=int, default=32, help="Batch size")
    parser.add_argument("--node_mask_fn", type=str, default="elbow_softmax:0", help="Thresholding of nodes")
    parser.add_argument("--edge_mask_fn", type=str, default="none", help="Thresholding of edges")

    args = parser.parse_args()
    return args


def fix_model_args(model_args, params):
    model_args.model_name = "gin"
    model_args.latent_dim = [params[f"model.gnn_layers.{i}.nn.0.weight"].shape[0] for i in range(3)]

    model_args.num_prototypes_per_class = (
        params["model.prototype_vectors"].shape[0] // params["model.last_layer.weight"].shape[0]
    )
    return model_args


def fix_data_args(data_args, args):
    data_args.dataset_name = args.dataset
    data_args.random_split = args.dataset == "MUTAG"
    data_args.seed = args.seed
    return data_args


def get_explanations(model, explainer, dataloader, forward_fn, seed):
    explanations = list()
    set_seed(seed)
    for data in dataloader:
        data = data.to(device)
        with torch.no_grad():
            logits, response = forward_fn(model, data)
        pred = logits.argmax(dim=-1)
        importance = run_shap(explainer, response).to(device)
        importance = torch.stack([importance[pred[i]][i] for i in range(len(pred))], dim=0)
        num_nodes = 0
        for b in tqdm(range(len(data.y)), total=len(data.y)):
            im = importance[b]
            mask = data.batch == b
            edge_index = data.edge_index[:, data.batch[data.edge_index[0]] == b] - num_nodes
            d = Data(x=data.x[mask], edge_index=edge_index, y=data.y[b]).to(device)
            num_nodes += mask.sum()
            node_mask = torch.zeros(len(d.x), len(im), device=device)
            edge_mask = torch.zeros(d.edge_index.shape[1], im.shape[-1], device=device)
            for i in range(model.model.last_layer.in_features):
                coalition, similarity, prot = mcts(d, model, model.model.prototype_vectors[i])
                norm = len(coalition)
                if norm == 0:
                    continue
                node_mask[coalition, i] += im[i] / norm

            x, edge_index, y, pred = d.x.clone(), d.edge_index.clone(), d.y.clone(), logits[b].clone()
            e = Data(x=x, edge_index=edge_index, y=y, pred=pred, node_mask=node_mask, edge_mask=edge_mask)
            explanations.append(e)
    return explanations


def get_truncated_explanations(explanations, node_mask_fn, edge_mask_fn):
    def _get_truncated_explanations(e):
        e = e.to(device)
        subset, edge_index = extract_subset_and_edge_index(
            e.edge_index,
            len(e.x),
            e.node_mask.sum(1),
            node_mask_fn,
            e.edge_mask.sum(1),
            edge_mask_fn,
        )
        edge_index, _ = subgraph(subset, edge_index, relabel_nodes=True, num_nodes=len(e.x))
        y_pred = e.pred.argmax()
        return Data(edge_index=edge_index, x=e.x[subset], y=y_pred, nodes_subset=subset)

    return list(map(_get_truncated_explanations, explanations))


def get_prototypes(model, dataset, num_prototypes_per_class):
    prototypes = list()
    for i in range(model.model.last_layer.in_features):
        count = 0
        best_similarity = 0
        label = i // num_prototypes_per_class
        prototypes.append(None)
        for data in tqdm(dataset, total=len(dataset)):
            data = data.to(device)
            if data.y == label:
                count += 1
                coalition, similarity, prot = mcts(data, model, model.model.prototype_vectors[i])
                if similarity > best_similarity:
                    best_similarity = similarity
                    edge_index, _ = subgraph(
                        coalition, data.edge_index, relabel_nodes=True, num_nodes=len(data.x)
                    )
                    prototypes[i] = Data(x=data.x[coalition], edge_index=edge_index, y=data.y)
    return prototypes


def perturb_model(model, prototypes, dataset, p_x=0.2, p_edge=0.2):
    model_copy = copy.deepcopy(model)
    model_copy.eval()
    all_x = torch.cat([data.x for data in dataset])
    prototype_vectors = model_copy.model.prototype_vectors.detach()
    for i in range(prototype_vectors.shape[0]):
        x, edge_index = prototypes[i].x, prototypes[i].edge_index
        if p_x > 0:
            keys = torch.tensor(np.random.choice([0, 1], p=[1 - p_x, p_x], size=len(x))).bool().to(x.device)
            size = keys.sum().item()
            if size > 0:
                x[keys] = all_x[np.random.randint(low=0, high=len(all_x), size=size)].to(x.device)
        if p_edge > 0:
            adj = to_dense_adj(edge_index)[0]
            idx = torch.triu_indices(adj.shape[0], adj.shape[0], 1)
            flatten_adj = adj[idx[0], idx[1]]
            keys = (
                torch.tensor(np.random.choice([0, 1], p=[1 - p_edge, p_edge], size=len(flatten_adj)))
                .bool()
                .to(flatten_adj.device)
            )
            flatten_adj[keys] = 1 - flatten_adj[keys]
            adj = torch.zeros_like(adj)
            adj[idx[0], idx[1]] = flatten_adj
            adj = adj + adj.T
            edge_index = dense_to_sparse(adj)[0]
        data = Data(x=x, edge_index=edge_index, batch=torch.zeros(len(x), dtype=torch.long)).to(x.device)
        _, _, _, emb, _ = model(data)
        prototype_vectors[i] = emb
    model_copy.model.prototype_vectors.data = prototype_vectors
    return model_copy


def main():
    from Configures import data_args, train_args, model_args

    args = args_parser()
    print(args)

    dataset = load_dataset.get_dataset(f"{PROTGNN_PATH}/datasets/", args.dataset, task=None)
    train_idxs, val_idxs, test_idxs = get_splits(
        args.dataset, size=len(dataset), seed=args.seed, split=args.split
    )

    dataset_train, dataset_val, dataset_test = (
        Subset(dataset, train_idxs),
        Subset(dataset, val_idxs),
        Subset(dataset, test_idxs),
    )

    dataloader_test = DataLoader(dataset_test, batch_size=args.batch_size, shuffle=False)

    s = args.seed if args.split < 0 else args.split
    model_path = f"{PROTGNN_PATH}/checkpoint/{args.dataset}/gin_s{s}_best.pth"
    params = torch.load(model_path)
    params = params["net"]
    model_args = fix_model_args(model_args, params)
    model = GnnNets(dataset.num_node_features, dataset.num_classes, model_args).to(device)
    model.update_state_dict(params)
    model.eval()
    forward_fn = lambda model, data: model(data)[0]
    results = {}

    node_mask_fn = get_fn(args.node_mask_fn)
    edge_mask_fn = get_fn(args.edge_mask_fn)

    if "A" in args.metrics:
        results["A"] = get_acc(model, dataloader_test, forward_fn)
        print_results(results)

    def logits_and_responses_fn(model, data):
        logits, _, _, _, min_distances = model(data)
        prototype_activations = torch.log((min_distances + 1) / (min_distances + model.model.epsilon))
        return logits, prototype_activations

    dataloader = DataLoader(
        dataset_train + dataset_val + dataset_test, batch_size=args.batch_size, shuffle=False
    )

    def get_shap(classifier, inputs):
        torch.set_grad_enabled(True)
        explainer = shap.DeepExplainer(classifier, Variable(inputs))
        shap_values = explainer.shap_values(Variable(inputs))
        shap_values = np.stack(shap_values, axis=0)
        return torch.from_numpy(shap_values), explainer

    responses, logits = get_responses_and_logits(model, dataloader, logits_and_responses_fn)
    shap_values, explainer = get_shap(model.model.last_layer, responses)

    explanations = get_explanations(
        model, explainer, dataloader_test, logits_and_responses_fn, args.seed
    )
    truncated_explanations = get_truncated_explanations(explanations, node_mask_fn, edge_mask_fn)
    prototypes = get_prototypes(model, dataset_test, model_args.num_prototypes_per_class)

    if "A1" in args.metrics:
        distances = compare_with_gt_instance(
            args.dataset if args.dataset != "BA_2Motifs" else "BA-2motif",
            dataset_test,
            truncated_explanations,
            iou_distance,
        )
        if len(distances) == 0:
            results["A1"] = None
        else:
            results["A1"] = 1 - np.mean(distances)
        print_results(results)

    if "A2" in args.metrics:
        ground_truths = get_gt_explanations_model(
            args.dataset if args.dataset != "BA_2Motifs" else "BA-2motif"
        )
        if len(ground_truths) > 0:
            distances = compare_with_gt_model(prototypes, ground_truths, weighted_ged_distance)
            results["A2"] = 1 - np.mean(distances)
        else:
            results["A2"] = None
        print_results(results)

    if bool({"I1", "I2"} & set(args.metrics)):
        with_explanations, without_explanations = remove_explanation_perturb(
            explanations,
            node_mask_fn,
            edge_mask_fn,
            skip_last_node=False,
        )
        if "I1" in args.metrics:
            with_explanations = [data for data in with_explanations if data is not None]
            dataloader_with = DataLoader(with_explanations, batch_size=args.batch_size, shuffle=False)
            results["I1"] = get_acc(model, dataloader_with, forward_fn)
            print_results(results)
        if "I2" in args.metrics:
            without_explanations = [data for data in without_explanations if data is not None]
            dataloader_without = DataLoader(without_explanations, batch_size=args.batch_size, shuffle=False)
            acc_without = 1 - get_acc(model, dataloader_without, forward_fn)
            results["I2"] = acc_without * len(without_explanations) / len(explanations)
            print_results(results)

    if "I3" in args.metrics:
        new_explanations = get_explanations(
            model, explainer, dataloader_test, logits_and_responses_fn, args.seed + 1
        )
        new_truncated_explanations = get_truncated_explanations(new_explanations, node_mask_fn, edge_mask_fn)
        distances = compare_lists(truncated_explanations, new_truncated_explanations, iou_distance)
        results["I3"] = 1 - np.mean(distances)
        print_results(results)

    if "I4" in args.metrics:
        dataset_noisy_node = add_noise_perturb(dataset_test, p_x=0.05, p_edges_add=0.0, p_edges_del=0.0, explanations=truncated_explanations)
        dataloader_new = DataLoader(dataset_noisy_node, batch_size=args.batch_size, shuffle=False)
        new_explanations = get_explanations(
            model, explainer, dataloader_new, logits_and_responses_fn, args.seed
        )
        new_truncated_explanations = get_truncated_explanations(new_explanations, node_mask_fn, edge_mask_fn)
        distances = compare_lists(truncated_explanations, new_truncated_explanations, iou_distance)
        results["I4"] = 1 - np.mean(distances)
        print_results(results)

    if "I5" in args.metrics:
        dataset_noisy_edge = add_noise_perturb(dataset_test, p_x=0.0, p_edges_add=0.005, p_edges_del=0.005, explanations=truncated_explanations)
        dataloader_new = DataLoader(dataset_noisy_edge, batch_size=args.batch_size, shuffle=False)
        new_explanations = get_explanations(
            model, explainer, dataloader_new, logits_and_responses_fn, args.seed
        )
        new_truncated_explanations = get_truncated_explanations(new_explanations, node_mask_fn, edge_mask_fn)
        distances = compare_lists(truncated_explanations, new_truncated_explanations, iou_distance)
        results["I5"] = 1 - np.mean(distances)
        print_results(results)

    if "I6" in args.metrics:
        dataloader_train = DataLoader(dataset_train, batch_size=args.batch_size, shuffle=True)
        dataloader_val = DataLoader(dataset_val, batch_size=args.batch_size, shuffle=False)

        explanations_train = get_explanations(
            model, explainer, dataloader_train, logits_and_responses_fn, args.seed
        )
        truncated_explanations_train = get_truncated_explanations(
            explanations_train, node_mask_fn, edge_mask_fn
        )
        explanations_val = get_explanations(
            model, explainer, dataloader_val, logits_and_responses_fn, args.seed
        )
        truncated_explanations_val = get_truncated_explanations(explanations_val, node_mask_fn, edge_mask_fn)
        new_dataloader_train = DataLoader(
            truncated_explanations_train, batch_size=args.batch_size, shuffle=True
        )
        new_dataloader_val = DataLoader(truncated_explanations_val, batch_size=args.batch_size, shuffle=False)
        new_dataloader_test = DataLoader(truncated_explanations, batch_size=args.batch_size, shuffle=False)
        new_dataset_name = "BA-2motif" if args.dataset == "BA_2Motifs" else args.dataset
        new_model_args = torch.load(f"models/{new_dataset_name}_s{s}.pt")["args"]
        new_model_args["num_node_features"] = dataset.x.shape[-1]

        new_model = get_model(**new_model_args).to(device)
        new_model = train_model(
            new_model,
            new_dataloader_train,
            new_dataloader_val,
            new_dataloader_test,
            epochs=500 if args.dataset == "MUTAG" else 200,
            lr=0.001,
            weight_decay=0.005,
        )
        new_model.eval()
        results["I6"] = get_acc(
            new_model,
            new_dataloader_test,
            lambda model, data: model(data.x, data.edge_index, batch=data.batch),
        )
        print_results(results)

    if "I7" in args.metrics:
        sizes = compare_lists(dataset_test, truncated_explanations, compare_size)
        results["I7"] = 1 - np.mean(sizes)
        print_results(results)

    if "M1" in args.metrics:
        try:
            perturbed_model = perturb_model(model, prototypes, dataset, p_x=0.5, p_edge=0)
            perturbed_responses, perturbed_logits = get_responses_and_logits(
                perturbed_model, dataloader, logits_and_responses_fn
            )
            _, perturbed_explainer = get_shap(model.model.last_layer, perturbed_responses)
            perturbed_explanations = get_explanations(
                perturbed_model,
                perturbed_explainer,
                dataloader_test,
                logits_and_responses_fn,
                args.seed,
            )
            perturbed_explanations = get_truncated_explanations(
                perturbed_explanations, node_mask_fn, edge_mask_fn
            )
            distances = compare_lists(truncated_explanations, perturbed_explanations, iou_distance)
            results["M1"] = np.mean(distances)
        except:
            results["M1"] = None
        print_results(results)

    if "M2" in args.metrics:
        try:
            perturbed_model = perturb_model(model, prototypes, dataset, p_x=0, p_edge=0.25)
            perturbed_responses, perturbed_logits = get_responses_and_logits(
                perturbed_model, dataloader, logits_and_responses_fn
            )
            _, perturbed_explainer = get_shap(model.model.last_layer, perturbed_responses)
            perturbed_explanations = get_explanations(
                perturbed_model,
                perturbed_explainer,
                dataloader_test,
                logits_and_responses_fn,
                args.seed,
            )
            perturbed_explanations = get_truncated_explanations(
                perturbed_explanations, node_mask_fn, edge_mask_fn
            )
            distances = compare_lists(truncated_explanations, perturbed_explanations, iou_distance)
            results["M2"] = np.mean(distances)
            print_results(results)
        except:
            results["M2"] = None

    if "M3" in args.metrics:
        results["M3"] = 1 - pairwise_list_corr(responses.cpu().detach().numpy(), to_abs=True)
        print_results(results)

    print("Final")
    print_results(results)


if __name__ == "__main__":
    main()
