# https://github.com/lucidrains/vit-pytorch/blob/main/vit_pytorch/vit.py

import torch
from torch import nn
from torch.nn.init import constant_

from mup import MuReadout
from einops import rearrange, repeat
from einops.layers.torch import Rearrange
from architectures.utils import MyIdentity

# helpers

def pair(t):
    return t if isinstance(t, tuple) else (t, t)

def init_method_normal(sigma):
    """Init method based on N(0, sigma)."""
    def init_(tensor):
        return nn.init.normal_(tensor, mean=0.0, std=sigma)
    return init_

# classes

def get_norm_layer(width, norm):
    if norm == None:
        return MyIdentity()
    elif norm == "ln":
        return nn.LayerNorm(width)
    else:
        raise ValueError()


class PreNorm(nn.Module):
    def __init__(self, dim, fn, norm):
        super().__init__()
        self.norm = get_norm_layer(dim, norm)
        self.fn = fn
    def forward(self, x, **kwargs):
        return self.fn(self.norm(x), **kwargs)


class FeedForward(nn.Module):
    def __init__(self, dim, hidden_dim, dropout = 0.):
        super().__init__()
        self.net = nn.Sequential(
            nn.Linear(dim, hidden_dim),
            nn.GELU(),
            nn.Dropout(dropout),
            nn.Linear(hidden_dim, dim),
            nn.Dropout(dropout)
        )
        self.init_method = init_method_normal((1 / dim)**0.5)
        
    def forward(self, x):
        return self.net(x)
    
    def reset_parameters(self):
        self.init_method(self.net[0].weight)
        self.init_method(self.net[3].weight)
        if self.net[0].bias is not None:
            constant_(self.net[0].bias, 0.)
        if self.net[3].bias is not None:
            constant_(self.net[3].bias, 0.)

class Attention(nn.Module):
    def __init__(self, dim, heads = 8, dim_head = 64, dropout = 0., standparam=False):
        super().__init__()
        inner_dim = dim_head *  heads
        #project_out = not (heads == 1 and dim_head == dim)

        self.dim = dim
        self.heads = heads
        #self.scale = dim_head ** -0.5
        if standparam:
            self.scale = float(dim_head) ** -0.5
        else:
            self.scale = float(dim_head) ** -1

        self.attend = nn.Softmax(dim = -1)
        self.to_qkv = nn.Linear(dim, inner_dim * 3, bias = False)

        self.to_out = nn.Sequential(
            nn.Linear(inner_dim, dim),
            nn.Dropout(dropout)
        )

        self.init_method = init_method_normal((1 / dim)**0.5)
        
        self._reset_parameters()
        
        
    def _reset_parameters(self):
        
        self.init_method(self.to_qkv.weight)
        # zero initializing query head
        constant_(self.to_qkv.weight[:self.dim], 0.)
        if self.to_qkv.bias is not None:
            constant_(self.to_qkv, 0.)
            
        self.init_method(self.to_out[0].weight)
        if self.to_out[0].bias is not None:
            constant_(self.to_out[0].bias, 0.)
            
            
    def forward(self, x):
        qkv = self.to_qkv(x).chunk(3, dim = -1)
        q, k, v = map(lambda t: rearrange(t, 'b n (h d) -> b h n d', h = self.heads), qkv)

        dots = torch.matmul(q, k.transpose(-1, -2)) * self.scale

        attn = self.attend(dots)

        out = torch.matmul(attn, v)
        out = rearrange(out, 'b h n d -> b n (h d)')
        return self.to_out(out)

class Transformer(nn.Module):
    def __init__(self, dim, depth, heads, dim_head, mlp_dim, dropout = 0., res_scaling=1.0, norm='ln'):
        super().__init__()
        self.res_scaling = res_scaling
        self.layers = nn.ModuleList([])
        for _ in range(depth):
            self.layers.append(nn.ModuleList([
                PreNorm(dim, Attention(dim, heads = heads, dim_head = dim_head, dropout = dropout), norm),
                PreNorm(dim, FeedForward(dim, mlp_dim, dropout = dropout), norm)
            ]))
    def forward(self, x):
        for attn, ff in self.layers:
            x = self.res_scaling * attn(x) + x
            x = self.res_scaling * ff(x) + x
        return x

class ViT(nn.Module):
    def __init__(self, *, image_size, num_classes, patch_size=4, init_dim=64, depth=3, heads=6, mlp_dim=128,
                 pool = 'cls', channels = 3, dim_head = -1, dropout = 0., emb_dropout = 0., 
                 wm=1, depth_mult=1, gamma=1, res_scaling=1, norm='ln'):
        super().__init__()
        image_height, image_width = pair(image_size)
        patch_height, patch_width = pair(patch_size)

        assert image_height % patch_height == 0 and image_width % patch_width == 0, 'Image dimensions must be divisible by the patch size.'

        num_patches = (image_height // patch_height) * (image_width // patch_width)
        patch_dim = channels * patch_height * patch_width
        assert pool in {'cls', 'mean'}, 'pool type must be either cls (cls token) or mean (mean pooling)'

        dim = int(init_dim*wm)
        mlp_dim = int(mlp_dim*wm)
        depth = int(depth*depth_mult)
        
        if dim_head == -1:
            dim_head = dim // heads
        
        self.to_patch_embedding = nn.Sequential(
            Rearrange('b c (h p1) (w p2) -> b (h w) (p1 p2 c)', p1 = patch_height, p2 = patch_width),
            nn.Linear(patch_dim, dim),
        )

        self.pos_embedding = nn.Parameter(torch.randn(1, num_patches + 1, dim))
        self.cls_token = nn.Parameter(torch.randn(1, 1, dim))
        self.dropout = nn.Dropout(emb_dropout)

        self.transformer = Transformer(dim, depth, heads, dim_head, mlp_dim, dropout, res_scaling, norm)

        self.pool = pool
        self.to_latent = nn.Identity()

        self.mlp_head = nn.Sequential(
            get_norm_layer(dim, norm),
            MuReadout(dim, num_classes)
        )
        
        self.init_weights()

    def init_weights(self):
        if self.mlp_head[1].bias is not None:
            self.mlp_head[1].bias.data.zero_()
        self.mlp_head[1].weight.data.zero_()
            
    def forward(self, img):
        x = self.to_patch_embedding(img)
        b, n, _ = x.shape

        cls_tokens = repeat(self.cls_token, '() n d -> b n d', b = b)
        x = torch.cat((cls_tokens, x), dim=1)
        x += self.pos_embedding[:, :(n + 1)]
        x = self.dropout(x)

        x = self.transformer(x)

        x = x.mean(dim = 1) if self.pool == 'mean' else x[:, 0]

        x = self.to_latent(x)
        return self.mlp_head(x)