# coding=utf-8
# Copyright 2020 The Gsa Net Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Attention layers."""
import tensorflow.compat.v1 as tf

from gsa_net import ops


def attention_layer(
    inputs,
    depth,
    head_count,
    pooling_kernel_size,
    strides,
    is_training = True,
    batch_norm_momentum = 0.99,
    batch_norm_epsilon = 1e-5,
):
  """An attention layer as a drop-in replacement for a convolution layer.

  Args:
    inputs: Input features to the layer, with shape
        (batch_size, height, width, input_depth).
    depth: Number of output channels.
    head_count: Number of attention heads.
    pooling_kernel_size: Kernel size for the pooling layer for downsampling.
        Skips pooling layer if `strides < 1`.
    strides: Stride of the layer. Currently only supports a single integer for
        both spatial dimensions.
    is_training: Whether in training or evaluation mode.
    batch_norm_momentum: Momentum for the batch normalization layers.
    batch_norm_epsilon: Momentum for the batch normalization layers.

  Returns:
    Output features of the layer with shape
        (batch_size, height // strides, width // strides, depth).
  """
  attn_output = ops.global_self_attention(
      inputs,
      depth,
      head_count,
      is_training=is_training,
      batch_norm_momentum=batch_norm_momentum,
      batch_norm_epsilon=batch_norm_epsilon,
  )

  if strides > 1 and pooling_kernel_size > 1:
    attn_output = tf.nn.avg_pool(
        attn_output,
        ksize=[1, pooling_kernel_size, pooling_kernel_size, 1],
        strides=[1, strides, strides, 1],
        padding='SAME',
        data_format='NHWC',
    )

  return attn_output
