# coding=utf-8
# Copyright 2020 The Gsa Net Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Tests for GSA-Net models."""
from absl.testing import parameterized
import tensorflow.compat.v1 as tf

from gsa_net import models


class ModelComponentTest(parameterized.TestCase, tf.test.TestCase):

  def setUp(self):
    super().setUp()
    self.batch_size = 2
    self.height = 6
    self.width = 4
    self.head_count = 8
    self.input_depth = 64
    self.output_depth = 96
    self.inputs = tf.random_normal((
        self.batch_size, self.height, self.width, self.input_depth))

  @parameterized.named_parameters(
      ('default', True, 1, True, False),
      ('evaluation', False, 1, True, False),
      ('strided', True, 2, True, True),
      ('convolutional', True, 1, False, False),
      ('projected', True, 1, True, True),
  )
  def test_bottleneck_block(
      self, is_training, strides, uses_attention, use_projection):
    depth = self.output_depth // 4 if use_projection else self.input_depth // 4
    outputs = models.bottleneck_block(
        self.inputs,
        depth,
        self.head_count,
        is_training,
        strides,
        uses_attention,
        use_projection,
    )
    self.assertEqual(outputs.shape, (
        self.batch_size,
        self.height // strides,
        self.width // strides,
        4 * depth,
    ))

    with tf.Session() as sess:
      sess.run(tf.global_variables_initializer())
      outputs = sess.run(outputs)
    self.assertNotAllEqual(outputs, 0)

  @parameterized.named_parameters(
      ('default', True, 3, 1, True, False),
      ('single_block', True, 1, 1, True, False),
      ('evaluation', False, 3, 1, True, False),
      ('strided', True, 3, 2, True, True),
      ('convolutional', True, 3, 1, False, False),
  )
  def test_block_group(
      self, is_training, block_count, strides, uses_attention, use_projection):
    outputs = models.block_group(
        self.inputs,
        self.output_depth,
        self.head_count,
        block_count,
        strides,
        uses_attention,
        is_training,
    )
    self.assertEqual(outputs.shape, (
        self.batch_size,
        self.height // strides,
        self.width // strides,
        4 * self.output_depth,
    ))

    with tf.Session() as sess:
      sess.run(tf.global_variables_initializer())
      outputs = sess.run(outputs)
    self.assertNotAllEqual(outputs, 0)


class FullModelTest(parameterized.TestCase, tf.test.TestCase):

  def setUp(self):
    super().setUp()
    self.batch_size = 2
    self.height = 224
    self.width = 224
    self.image_depth = 3
    self.images = tf.random_normal((
        self.batch_size, self.height, self.width, self.image_depth))

  @parameterized.named_parameters(
      ('default', True, ''),
      ('eval', False, ''),
      ('early_attention', True, 'use_attention=[True, True, False, False]'),
      ('late_attention', True, 'use_attention=[False, False, True, True]'),
      ('no_attention', True, 'use_attention=[False, False, False, False]'),
  )
  def test_call(self, is_training, model_hparams_str):
    model_hparams = models.get_hparams(model_hparams_str)
    net = models.GsaNet(model_hparams)
    logits = net(self.images, is_training)
    self.assertEqual(logits.shape, (self.batch_size, model_hparams.class_count))

    with tf.Session() as sess:
      sess.run(tf.global_variables_initializer())
      logits = sess.run(logits)
    self.assertNotAllEqual(logits, 0)

  @parameterized.named_parameters(
      ('default', True, ''),
      ('eval', False, ''),
      ('early_attention', True, 'use_attention=[True, True, False, False]'),
      ('late_attention', True, 'use_attention=[False, False, True, True]'),
      ('no_attention', True, 'use_attention=[False, False, False, False]'),
  )
  def test_build_model(self, is_training, model_hparams_str):
    model_hparams = models.get_hparams(model_hparams_str)
    logits = models.build_model(
        self.images, is_training, model_hparams)
    self.assertEqual(logits.shape, (self.batch_size, model_hparams.class_count))

    with tf.Session() as sess:
      sess.run(tf.global_variables_initializer())
      logits = sess.run(logits)
    self.assertNotAllEqual(logits, 0)


if __name__ == '__main__':
  tf.test.main()
