# coding=utf-8
# Copyright 2020 The Gsa Net Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Tests for convolutional layers."""
from absl.testing import parameterized
import tensorflow.compat.v1 as tf

from gsa_net.layers import conv_layers


class ConvLayersTest(parameterized.TestCase, tf.test.TestCase):

  def setUp(self):
    super().setUp()
    self.batch_size = 2
    self.height = 4
    self.width = 6
    self.output_depth = 24
    input_depth = 16

    self.inputs = tf.random_normal((
        self.batch_size, self.height, self.width, input_depth))

  @parameterized.named_parameters(
      ('default', True, True, False),
      ('evaluation', False, True, False),
      ('no_relu', True, False, False),
      ('init_zero', True, True, True),
  )
  def test_batch_norm_relu(self, is_training, relu, init_zero):
    outputs = conv_layers.batch_norm_relu(
        self.inputs,
        is_training,
        relu,
        init_zero,
    )
    self.assertEqual(outputs.shape, self.inputs.shape)

    with tf.Session() as sess:
      sess.run(tf.global_variables_initializer())
      outputs = sess.run(outputs)
    self.assertNotAllEqual(outputs, 0)

  @parameterized.named_parameters(
      ('default', 1),
      ('strided', 2),
  )
  def test_conv2d(self, strides):
    outputs = conv_layers.conv2d(
        self.inputs,
        self.output_depth,
        kernel_size=3,
        strides=strides,
    )
    self.assertEqual(outputs.shape, (
        self.batch_size,
        self.height // strides,
        self.width // strides,
        self.output_depth,
    ))

    with tf.Session() as sess:
      sess.run(tf.global_variables_initializer())
      outputs = sess.run(outputs)
    self.assertNotAllEqual(outputs, 0)


if __name__ == '__main__':
  tf.test.main()
