import logging
import zmq
import math
import os
import shutil
import sys
import numpy as np


def create_path(path):
    if os.path.exists(path):
        shutil.rmtree(path, ignore_errors=True)
    os.makedirs(path, exist_ok=True)


def setup_logger(name, log_file_path, level=logging.DEBUG):
    create_path(log_file_path)
    log_file = log_file_path + '/log'
    handler = logging.FileHandler(log_file)
    formatter = logging.Formatter('%(asctime)s,%(msecs)d,%(levelname)s::%(message)s')
    handler.setFormatter(formatter)
    logger = logging.getLogger(name)
    logger.setLevel(level)
    logger.addHandler(handler)
    stream_handler = logging.StreamHandler(sys.stdout)
    stream_handler.setFormatter(formatter)
    logger.addHandler(stream_handler)
    return logger


def zmq_nonblocking_recv(socket):
    raw_data_list = []
    while True:
        try:
            data = socket.recv(zmq.NOBLOCK)
            raw_data_list.append(data)
        except zmq.ZMQError as e:
            break
    return raw_data_list


def zmq_nonblocking_multipart_recv(socket):
    raw_data_list = []
    while True:
        try:
            data = socket.recv_multipart(zmq.NOBLOCK)
            raw_data_list.append(data)
        except zmq.ZMQError as e:
            break
    return raw_data_list


def softmax(logits, temp=1):
    e_x = np.exp(np.array(logits) / temp)
    probs = e_x / np.sum(e_x, axis=-1, keepdims=True)
    return probs


def recursive_update(config_dict, update_dict):
    for k, v in update_dict.items():
        if isinstance(v, dict):
            old = config_dict.get(k, {})
            if not isinstance(old, dict):
                config_dict[k] = v
            else:
                config_dict[k] = recursive_update(old, v)
        else:
            config_dict[k] = v
    return config_dict

