from typing import Dict, Any, List, Tuple, Optional
import math
import json
from collections import defaultdict

# ----------------------------
# 可调参数
# ----------------------------
NEAR_DISTANCE_M = 1.5            # 与 AI2-THOR 可交互半径一致
GRID_RESOLUTION_M = 0.25         # 将连续坐标粗量化成网格，可按需修改
USE_KNOWN_IF_VISIBLE = True      # 首次可见则标记 known=True

# 映射：THOR 对象字段 -> ABW 功能状态字段
# （按需增补，避免把不存在的字段硬塞）
FUNCTIONAL_MAP = {
    "isOpen": ("open", bool),
    "isToggled": ("toggle", lambda v: "on" if v else "off"),
    "isDirty": ("clean", lambda v: not bool(v)),  # isDirty True -> clean False
    "isSliced": ("sliced", bool),
}

# ----------------------------
# 工具函数
# ----------------------------
def _dist(p: Dict[str, float], q: Dict[str, float]) -> float:
    return math.sqrt((p["x"] - q["x"])**2 + (p["z"] - q["z"])**2)

def _to_grid(p: Dict[str, float]) -> Dict[str, float]:
    return {
        "x": round(p["x"] / GRID_RESOLUTION_M),
        "y": round(p["z"] / GRID_RESOLUTION_M)  # THOR 的平面是 x-z
    }

def _room_name_from_receptacle(obj: Dict[str, Any]) -> Optional[str]:
    # 占位实现：若你的管线能提供房间分区，就在这里返回房间名
    # 否则可返回 None 或根据坐标/区域划分推断
    return obj.get("room", None)

def _is_receptacle(obj: Dict[str, Any]) -> bool:
    # THOR metadata: "receptacle": True 表示可放置
    return bool(obj.get("receptacle"))

def _safe_get(d: Dict[str, Any], key: str, default=None):
    v = d.get(key, default)
    return v if v is not None else default

# ----------------------------
# 主函数
# ----------------------------
def build_abw_state(
    thor_event_metadata: Dict[str, Any],
    agents_info: Dict[str, Dict[str, Any]],
    task_spec: Dict[str, Any],
    occupied_slots: Optional[List[str]] = None,
    blocked_paths: Optional[List[Tuple[str, str]]] = None,
) -> Dict[str, Any]:
    """
    将 THOR 的 metadata + 多智能体信息 映射到 ABW state.json 结构。

    参数：
      thor_event_metadata: 单步事件的 metadata（controller.step 后的 event.metadata）
      agents_info: {agent_id: {"position": {"x":..., "z":...},
                               "rotation": float, 
                               "holding": <objectId or None>,
                               "room": <optional room name>}}
      task_spec: { "subtasks": [...], "goal_reached": bool, "steps_remaining": int }
      occupied_slots: 例如 ["fridge_1_handle"]
      blocked_paths: 例如 [("agent_A", "drawer_3 blocked by chair")]

    返回：
      abw_state: 符合我们 schema 的 dict
    """
    objects_meta = thor_event_metadata.get("objects", [])
    abw = {
        "agents": {},
        "objects": {},
        "constraints": {"occupied": [], "blocked_paths": []},
        "task": {
            "subtasks": task_spec.get("subtasks", []),
            "goal_reached": task_spec.get("goal_reached", False),
            "steps_remaining": task_spec.get("steps_remaining", 0),
        },
        "metadata": {
            "scene_id": thor_event_metadata.get("sceneName", ""),
            "horizon": task_spec.get("steps_remaining", 0),
        }
    }

    # 1) 先填 agents
    for aid, ainfo in agents_info.items():
        pos = ainfo.get("position", {"x": 0.0, "z": 0.0})
        abw["agents"][aid] = {
            "location": {
                "room": ainfo.get("room", None) or "",  # 若你有房间信息就填
                "grid": _to_grid({"x": pos["x"], "z": pos["z"]}),
            },
            "pose": float(ainfo.get("rotation", 0.0)),
            "holding": ainfo.get("holding") if ainfo.get("holding") is not None else "null",
            "capacity": {"max_carry": 1, "can_carry_more": ainfo.get("holding") in (None, "null")}
        }

    # 2) 构建 objects：基础属性 + per-agent near/visible
    # 先缓存所有 agent 位置
    agent_positions = {aid: a["location"]["grid"] for aid, a in abw["agents"].items()}
    agent_positions_float = {
        aid: agents_info[aid]["position"] for aid in agents_info
    }

    for obj in objects_meta:
        oid = obj.get("objectId") or obj.get("name") or obj.get("id")
        if not oid:
            continue
        otype = obj.get("objectType") or obj.get("type", "Unknown")
        # 可见性（THOR: visible: bool）
        visible = bool(obj.get("visible"))
        # 位置
        pos = _safe_get(obj, "position", {"x": 0.0, "z": 0.0})
        grid = _to_grid({"x": pos["x"], "z": pos["z"]})
        room = _room_name_from_receptacle(obj) or ""

        # per-agent near 计算
        near_map = {}
        for aid, apos in agent_positions_float.items():
            near_map[aid] = (_dist({"x": pos["x"], "z": pos["z"]}, apos) <= NEAR_DISTANCE_M)

        # functional 映射
        functional = {}
        for thor_key, (abw_key, cast_fn) in FUNCTIONAL_MAP.items():
            if thor_key in obj:
                val = obj[thor_key]
                functional[abw_key] = cast_fn(val) if callable(cast_fn) else cast_fn(val)

        # 可放置/容器关系（简单占位：如果 THOR 提供 parentReceptacles，可写 in/on 推断）
        # 这里我们只把 receptacle 的能力记录在 type 上，具体 in/on 在任务时更新
        spatial = {
            "near": near_map,
            "location": {
                "room": room,
                "grid": grid
            }
        }

        # known 标记：可视即可知，或你也可以引入长期记忆判断
        known = bool(obj.get("visible")) if USE_KNOWN_IF_VISIBLE else bool(obj.get("seen", False))

        abw["objects"][oid] = {
            "type": otype,
            "visible": visible,
            "known": known,
            "spatial": spatial,
            "functional": functional if functional else {"usable": True}
        }

    # 3) 约束/冲突占位
    if occupied_slots:
        abw["constraints"]["occupied"] = list(dict.fromkeys(occupied_slots))  # 去重
    if blocked_paths:
        abw["constraints"]["blocked_paths"] = [
            {"id": _id, "reason": reason} for _id, reason in blocked_paths
        ]

    return abw


# ----------------------------
# 示例用法（单智能体）：
# ----------------------------
if __name__ == "__main__":
    # 假设你已从 controller.step(...) 得到 event.metadata:
    fake_event = {
        "sceneName": "Kitchen_01",
        "objects": [
            {"objectId": "Apple_1", "objectType": "Apple", "visible": True,
             "position": {"x": 1.0, "z": 2.0}, "isDirty": True},
            {"objectId": "Fridge_1", "objectType": "Fridge", "visible": True,
             "position": {"x": 2.2, "z": 2.1}, "isOpen": False}
        ]
    }

    agents_info = {
        "agent_A": {
            "position": {"x": 1.1, "z": 2.2},
            "rotation": 90.0,
            "holding": None,
            "room": "Kitchen"
        }
    }

    task_spec = {
        "subtasks": [
            {"id": "store_apple", "description": "Put Apple_1 into Fridge_1",
             "completed": False, "progress": 0.0}
        ],
        "goal_reached": False,
        "steps_remaining": 30
    }

    abw_state = build_abw_state(fake_event, agents_info, task_spec,
                                occupied_slots=[],
                                blocked_paths=[])
    print(json.dumps(abw_state, indent=2))
