
from typing import Optional
import os
from .imagelist import ImageList
from ._util import download as download_data, check_exits


class ImageCLEF(ImageList):
    """Office31 Dataset.

    Args:
        root (str): Root directory of dataset
        task (str): The task (domain) to create dataset. Choices include ``'A'``: amazon, \
            ``'D'``: dslr and ``'W'``: webcam.
        download (bool, optional): If true, downloads the dataset from the internet and puts it \
            in root directory. If dataset is already downloaded, it is not downloaded again.
        transform (callable, optional): A function/transform that  takes in an PIL image and returns a \
            transformed version. E.g, :class:`torchvision.transforms.RandomCrop`.
        target_transform (callable, optional): A function/transform that takes in the target and transforms it.

    .. note:: In `root`, there will exist following files after downloading.
        ::
            amazon/
                images/
                    backpack/
                        *.jpg
                        ...
            dslr/
            webcam/
            image_list/
                amazon.txt
                dslr.txt
                webcam.txt
    """
    download_list = [
        ("image_list", "image_list.zip", "https://cloud.tsinghua.edu.cn/f/0ac1924f514b44788d1d/?dl=1"),
        ("b", "b.tgz", "https://cloud.tsinghua.edu.cn/f/f40e5a484e7d43af9403/?dl=1"),
        ("c", "c.tgz", "https://cloud.tsinghua.edu.cn/f/a6bc0fcca766496cb112/?dl=1"),
        ("i", "i.tgz", "https://cloud.tsinghua.edu.cn/f/2e89cf41a299411093f0/?dl=1"),
        ("p", "p.tgz", "https://cloud.tsinghua.edu.cn/f/2e89cf41a299411093f0/?dl=1"),
    ]
    image_list = {
        "b": "image_list/b.txt",
        "c": "image_list/c.txt",
        "i": "image_list/i.txt",
        "p": "image_list/p.txt"
    }
    CLASSES = ['aeroplane', 'bike', 'bird', 'boat', 'bottle', 'bus', 'car', 'dog','horse','monitor','motorbike','people']

    def __init__(self, root: str, task: str, download: Optional[bool] = False, **kwargs):
        assert task in self.image_list
        data_list_file = os.path.join(root, self.image_list[task])

        if download:
            list(map(lambda args: download_data(root, *args), self.download_list))
        else:
            list(map(lambda file_name, _: check_exits(root, file_name), self.download_list))

        super(ImageCLEF, self).__init__(root, ImageCLEF.CLASSES, data_list_file=data_list_file, **kwargs)

    @classmethod
    def domains(cls):
        return list(cls.image_list.keys())