"""
Configure options to run tests when GPU is available
and you want to use GPU to generate model responses in tests.

You can add @pytest.mark.use_gpu decorator to your tests if you want to let your tests run
only when with option --use-gpu.

https://docs.pytest.org/en/latest/example/simple.html#control-skipping-of-tests-according-to-command-line-option
"""

from typing import Sequence

import pytest


def pytest_addoption(parser: pytest.Parser) -> None:
    parser.addoption(
        "--use-gpu", action="store_true", default=False, help="Use GPU resource to run LLMs"
    )


def pytest_configure(config: pytest.Config) -> None:
    config.addinivalue_line("markers", "use_gpu: mark test as use_gpu to run")


def pytest_collection_modifyitems(config: pytest.Config, items: Sequence[pytest.Item]) -> None:
    if config.getoption("--use-gpu"):
        return
    skip_gpu_test = pytest.mark.skip(reason="need --use-gpu option to run")
    for item in items:
        if "use_gpu" in item.keywords:
            item.add_marker(skip_gpu_test)
