import math
import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.utils.model_zoo as model_zoo
from torch.autograd import Variable

__all__ = ['My_MobileNetV2']

def _make_divisible(v, divisor, min_value=None):
    if min_value is None:
        min_value = divisor
    new_v = max(min_value, int(v + divisor / 2) // divisor * divisor)
    if new_v < 0.9 * v:
        new_v += divisor
    return new_v

def ConvBNReLU(inp, oup, kernel_size=3, stride=1, padding=0, groups=1):
    return nn.Sequential(
            nn.Conv2d(inp,
                      oup,
                      kernel_size=kernel_size,
                      stride=stride,
                      padding=padding,
                      groups=groups,
                      bias=False),
            nn.BatchNorm2d(oup),
            nn.ReLU6(inplace=True)
            )

class InvertedResidual(nn.Module):
    def __init__(self, inp, oup, stride, expand_ratio):
        super(InvertedResidual, self).__init__()
        self.stride = stride
        assert stride in [1, 2]

        #hidden_dim = int(round(inp * expand_ratio))
        self.use_res_connect = self.stride == 1 and inp == oup

        layers = []
        #if expand_ratio != 1:
            # pw
        layers.append(ConvBNReLU(inp, inp*expand_ratio, 1, 1, 0))
        layers.extend([
            # dw
            ConvBNReLU(inp*expand_ratio, inp*expand_ratio, 3, stride=stride, padding=1, groups=inp*expand_ratio),
            # pw-linear
            nn.Conv2d(inp*expand_ratio, oup, 1, 1, 0, bias=False),
            nn.BatchNorm2d(oup)
        ])
        self.conv = nn.Sequential(*layers)

    def forward(self, x):
        if self.use_res_connect:
            return x + self.conv(x)
        else:
            return self.conv(x)

class My_MobileNetV2(nn.Module):
    def __init__(self, dataset, num_branches, width_mult=1.0, inverted_residual_setting=None,
            round_nearest=8):
        super(My_MobileNetV2, self).__init__()
        self.num_branches = num_branches
        block = InvertedResidual
        input_channel = 32
        last_channel = 1280
        H = input_channel // (32//2)

        self.width_mult = width_mult
        if dataset == 'cifar10':
            num_classes = 10
        elif dataset == 'cifar100':
            num_classes = 100
        self.num_classes = num_classes

        if inverted_residual_setting is None:
            inverted_residual_setting = [
                # t, c, n, s
                [1, 16, 1, 1],
                [6, 24, 2, 1], # change stride 2 -> 1 for CIFAR10
                #[6, 24, 2, 2],
                [6, 32, 3, 2],
                [6, 64, 4, 2],
                [6, 96, 3, 1],
                #[6, 160, 3, 2],
                #[6, 320, 1, 1],
            ]

        if len(inverted_residual_setting) == 0 or len(inverted_residual_setting[0]) != 4:
            raise ValueError("inverted_residual_setting should be non-empty "
                             "or a 4-element list, got {}".format(inverted_residual_setting))

        input_channel = _make_divisible(input_channel * width_mult, round_nearest)
        self.last_channel = _make_divisible(last_channel * max(1.0, width_mult), round_nearest)
        features = [ConvBNReLU(3, input_channel, stride=2, padding=1)]

        for t, c, n, s in inverted_residual_setting:
            output_channel = _make_divisible(c * width_mult, round_nearest)
            for i in range(n):
                stride = s if i == 0 else 1
                features.append(block(input_channel, output_channel, stride, expand_ratio=t))
                input_channel = output_channel
        fixed_input_channel1 = input_channel

        self.features = nn.Sequential(*features)

        for i in range(self.num_branches+1):
            input_channel = fixed_input_channel1
            #for j in range(3):
            #    output_channel = _make_divisible(160*width_mult, round_nearest)
            #    setattr(self, 'features0_{}_{}'.format(i, j), block(fixed_input_channel if j==0 else input_channel, output_channel, 2, expand_ratio=6))
            #    input_channel = output_channel
            branch_blocks = []
            output_channel = _make_divisible(160*width_mult, round_nearest)
            for j in range(3):
                stride = 2 if j == 0 else 1
                branch_blocks.append(block(input_channel, output_channel, stride, expand_ratio=6))
                input_channel = output_channel
            setattr(self, 'features1_'+str(i), nn.Sequential(*branch_blocks))
            fixed_input_channel = input_channel
            setattr(self, 'features2_'+str(i), block(fixed_input_channel, _make_divisible(320*width_mult, round_nearest), 1, expand_ratio=6))
            input_channel = _make_divisible(320*width_mult, round_nearest)
            setattr(self, 'CBR_'+str(i), ConvBNReLU(input_channel, self.last_channel, kernel_size=1))
            setattr(self, 'classifier_'+str(i), nn.Linear(self.last_channel, num_classes))

        self.avgpool2d = nn.AvgPool2d(H, ceil_mode=True)

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out')
                if m.bias is not None:
                    nn.init.zeros_(m.bias)
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.ones_(m.weight)
                nn.init.zeros_(m.bias)
            elif isinstance(m, nn.Linear):
                nn.init.normal_(m.weight, 0, 0.01)
                nn.init.zeros_(m.bias)

    def forward(self, x):
        x = self.features(x)
        
        locals()['x_0'] = getattr(self, 'features1_0')(x)
        locals()['x_0'] = getattr(self, 'features2_0')(locals()['x_0'])
        locals()['x_0'] = getattr(self, 'CBR_0')(locals()['x_0'])
        locals()['x_0'] = self.avgpool2d(locals()['x_0'])
        locals()['x_0'] = locals()['x_0'].view(locals()['x_0'].size(0), -1)
        locals()['x_0'] = getattr(self, 'classifier_0')(locals()['x_0'])
        ind = locals()['x_0'].unsqueeze(-1)

        for i in range(1, self.num_branches):
            locals()['x_{}'.format(i)] = getattr(self, 'features1_'+str(i))(x)
            locals()['x_{}'.format(i)] = getattr(self, 'features2_'+str(i))(locals()['x_{}'.format(i)])
            locals()['x_{}'.format(i)] = getattr(self, 'CBR_'+str(i))(locals()['x_{}'.format(i)])
            locals()['x_{}'.format(i)] = self.avgpool2d(locals()['x_{}'.format(i)])
            locals()['x_{}'.format(i)] = locals()['x_{}'.format(i)].view(locals()['x_{}'.format(i)].size(0), -1)
            locals()['x_{}'.format(i)] = getattr(self, 'classifier_'+str(i))(locals()['x_{}'.format(i)])
            ind = torch.cat([ind, locals()['x_{}'.format(i)].unsqueeze(-1)], dim=-1)

        stu = getattr(self, 'features1_'+str(self.num_branches))(x)
        stu = getattr(self, 'features2_'+str(self.num_branches))(stu)
        stu = getattr(self, 'CBR_'+str(self.num_branches))(stu)
        stu = self.avgpool2d(stu)
        stu = stu.view(stu.size(0), -1)
        stu = getattr(self, 'classifier_'+str(self.num_branches))(stu)

        return ind, stu, ind

if __name__ == '__main__':
    model = My_MobileNetV2(dataset='cifar10', width_mult=1.0, num_branches=3)
    print(model)
    x = torch.randn(2, 3, 32, 32)
    y = model(x)
    print(y[0].size(), y[1].size())
