# datagen/create_dfa.py
from __future__ import annotations
import argparse
import json
import pathlib
from typing import Dict, Tuple, List

from automata.dfa import dfa_from_edges, DFA


def load_spec(path: pathlib.Path) -> Dict:
    spec = json.loads(path.read_text())
    required = {"sigma", "num_states", "start", "finals", "transitions"}
    missing = required - set(spec.keys())
    if missing:
        raise ValueError(f"Missing required keys in spec: {sorted(missing)}")
    return spec


def dfa_from_spec(spec: Dict) -> DFA:
    sigma: List[str] = list(spec["sigma"])
    num_states: int = int(spec["num_states"])
    start: int = int(spec["start"])
    finals: List[int] = [int(x) for x in spec["finals"]]

    # Flatten transitions dict {"q": {"sym": r}} -> edges {(q, sym): r}
    trans = spec["transitions"]
    edges: Dict[Tuple[int, str], int] = {}
    for q_str, row in trans.items():
        q = int(q_str)
        for s, r in row.items():
            edges[(q, str(s))] = int(r)

    # Build complete DFA (with dead state if needed)
    dfa = dfa_from_edges(
        num_states=num_states,
        sigma=sigma,
        start=start,
        finals=finals,
        edges=edges,
        complete=True,
        add_dead=False,
    )
    return dfa


def save_dfa_json(dfa: DFA, path: pathlib.Path) -> None:
    obj = {
        "sigma": list(dfa.sigma),
        "start": int(dfa.start),
        "finals": list(map(int, dfa.finals)),
        "delta": [list(map(int, row)) for row in dfa.delta],
        "dead": (None if dfa.dead is None else int(dfa.dead)),
    }
    path.parent.mkdir(parents=True, exist_ok=True)
    path.write_text(json.dumps(obj, ensure_ascii=False, indent=2))


def main():
    ap = argparse.ArgumentParser(description="Create canonical DFA JSON+PNG from a readable spec.")
    ap.add_argument("--name", required=True, help="name (e.g., tomita2).")
    ap.add_argument("--spec", required=True, help="Path to readable DFA JSON (dead transitions omitted).")
    ap.add_argument("--out-dir", default='datagen/assets/dfas', help="Override output dir. Default: ./languages/assets/<name>/")
    args = ap.parse_args()

    name = args.name
    out_dir = pathlib.Path(args.out_dir) / name
    out_dir.mkdir(parents=True, exist_ok=True)

    spec = load_spec(pathlib.Path(args.spec))
    dfa = dfa_from_spec(spec)

    # Save canonical DFA JSON
    dfa_json_path = out_dir / f"{name}.dfa.json"
    save_dfa_json(dfa, dfa_json_path)

    # --- Render FULL view ---
    full_noext = out_dir / f"{name}.dfa"   # path WITHOUT extension
    produced_full = dfa.render(str(full_noext), fmt="png")

    # --- Render SIMPLE view (omit edges to dead; hide dead node) ---
    simple_noext = out_dir / f"{name}.dfa.simple"
    produced_simple = dfa.render_simple(str(simple_noext), fmt="png", hide_dead=True, merge_labels=True)

    print(f"Saved DFA JSON to: {dfa_json_path}")
    print(f"Rendered FULL DFA diagram to: {produced_full}")
    print(f"Rendered SIMPLE DFA diagram to: {produced_simple}")

if __name__ == "__main__":
    main()
