# scripts/make_bounded_dyck.py
from __future__ import annotations
import argparse
import json
import pathlib
from collections import deque
from typing import Dict, List, Tuple

OPEN_TOKENS = ["(", "[", "{", "<"]
CLOSE_TOKENS = [")", "]", "}", ">"]

def build_bounded_dyck_spec(n: int, k: int) -> Dict:
    if not (1 <= n <= 4):
        raise ValueError("n must be in [1,4].")
    if k < 0:
        raise ValueError("k must be >= 0.")

    # Sigma: interleaved open/close for each type, e.g., ["(",")","[","]",...]
    sigma: List[str] = []
    for i in range(n):
        sigma.extend([OPEN_TOKENS[i], CLOSE_TOKENS[i]])

    # State space = all stacks (tuples of ints in 0..n-1) of length 0..k reachable by push/pop rules
    # We BFS from the empty stack to ensure reachability, numbering states as we discover them.
    stack2id: Dict[Tuple[int, ...], int] = {}
    id2stack: List[Tuple[int, ...]] = []

    def get_id(stack: Tuple[int, ...]) -> int:
        if stack not in stack2id:
            stack2id[stack] = len(id2stack)
            id2stack.append(stack)
        return stack2id[stack]

    q = deque()
    start_id = get_id(())  # empty stack
    q.append(())

    while q:
        st = q.popleft()
        depth = len(st)

        # All possible valid opens (if depth < k)
        if depth < k:
            for t in range(n):
                st2 = st + (t,)
                if st2 not in stack2id:
                    get_id(st2)
                    q.append(st2)

        # All possible valid closes (if depth > 0)
        if depth > 0:
            top = st[-1]
            st2 = st[:-1]
            # Only the close that matches the top is valid
            if st2 not in stack2id:
                get_id(st2)  # usually visited already
                q.append(st2)

    num_states = len(id2stack)

    # Build transitions map (omit invalids → dead via create_dfa)
    transitions: Dict[str, Dict[str, int]] = {}
    for sid, st in enumerate(id2stack):
        row: Dict[str, int] = {}
        depth = len(st)

        # Opens
        if depth < k:
            for t in range(n):
                st2 = st + (t,)
                row[OPEN_TOKENS[t]] = stack2id[st2]

        # Closes
        if depth > 0:
            top = st[-1]
            st2 = st[:-1]
            row[CLOSE_TOKENS[top]] = stack2id[st2]

        transitions[str(sid)] = row

    # Accepting only at empty stack
    spec = {
        "sigma": sigma,
        "num_states": num_states,  # dead omitted; will be added automatically if needed
        "start": 0,
        "finals": [0],
        "transitions": transitions
    }
    return spec


def main():
    ap = argparse.ArgumentParser(description="Generate human-readable bounded Dyck(n,k) DFA spec.")
    ap.add_argument("--n", type=int, required=True, help="Number of bracket types (1..4).")
    ap.add_argument("--k", type=int, required=True, help="Max depth (>=0).")
    ap.add_argument("--out", type=str, required=True, help="Path to write the JSON spec.")
    args = ap.parse_args()

    spec = build_bounded_dyck_spec(args.n, args.k)
    out_path = pathlib.Path(args.out)
    out_path.parent.mkdir(parents=True, exist_ok=True)
    out_path.write_text(json.dumps(spec, ensure_ascii=False, indent=2))
    print(f"Wrote Dyck({args.n},{args.k}) spec to: {out_path}")

if __name__ == "__main__":
    main()
