import torch
import torch.nn as nn
import torch.nn.functional as F
import numpy as np

class SupConLoss(nn.Module):

    def __init__(self, temperature=0.07, contrast_mode='all',
                 base_temperature=0.07):
        super(SupConLoss, self).__init__()
        self.temperature = temperature
        self.contrast_mode = contrast_mode
        self.base_temperature = base_temperature

    def forward(self, features, labels=None, mask=None):

        device = (torch.device('cuda')
                  if features.is_cuda
                  else torch.device('cpu'))

        if len(features.shape) < 3:
            raise ValueError('`features` needs to be [bsz, n_views, ...],'
                             'at least 3 dimensions are required')
        if len(features.shape) > 3:
            features = features.view(features.shape[0], features.shape[1], -1)

        batch_size = features.shape[0]
        if labels is not None and mask is not None:
            raise ValueError('Cannot define both `labels` and `mask`')
        elif labels is None and mask is None:
            mask = torch.eye(batch_size, dtype=torch.float32).to(device)
        elif labels is not None:
            labels = labels.contiguous().view(-1, 1)
            if labels.shape[0] != batch_size:
                raise ValueError('Num of labels does not match num of features')
            mask = torch.eq(labels, labels.T).float().to(device)
        else:
            mask = mask.float().to(device)

        contrast_count = features.shape[1]
        contrast_feature = torch.cat(torch.unbind(features, dim=1), dim=0)  # change features from [bsz, n_views, ...] to [bsz*n_views, ...]
        if self.contrast_mode == 'one':
            anchor_feature = features[:, 0]
            anchor_count = 1
        elif self.contrast_mode == 'all':
            anchor_feature = contrast_feature
            anchor_count = contrast_count
        else:
            raise ValueError('Unknown mode: {}'.format(self.contrast_mode))

        # compute logits  (2bsz, 2bsz) if mode = 'all'
        anchor_dot_contrast = torch.div(
            torch.matmul(anchor_feature, contrast_feature.T),
            self.temperature)     # similarity between all samples.
        # for numerical stability
        logits_max, _ = torch.max(anchor_dot_contrast, dim=1, keepdim=True)
        logits = anchor_dot_contrast - logits_max.detach()

        # tile mask
        mask = mask.repeat(anchor_count, contrast_count)
        # mask-out self-contrast cases
        logits_mask = torch.scatter(
            torch.ones_like(mask),
            1,
            torch.arange(batch_size * anchor_count).view(-1, 1).to(device),
            0
        )    # (2bsz, 2bsz) if mode = 'all'. all elements are 1 except diagonal is 0.
        mask = mask * logits_mask  # mask for all positive pair

        # compute log_prob
        exp_logits = torch.exp(logits) * logits_mask   # exp similarity of all prositive and negative pair.
        log_prob = logits - torch.log(exp_logits.sum(1, keepdim=True))   # all similarity divide the sum of (exp similarity of all prositive and negative pair.)

        # compute mean of log-likelihood over positive
        mean_log_prob_pos = (mask * log_prob).sum(1) / mask.sum(1)    # mask * log_prob means only consider positive pair / all pair

        # loss
        loss = - (self.temperature / self.base_temperature) * mean_log_prob_pos
        loss = loss.view(anchor_count, batch_size).mean()

        return loss

def fea_proj(args, dim_in, feat_dim=512, head='linear'):
    # feature projector from PCL code.
    # dropout = nn.Dropout(0.25)
    # hidden_size = 256 if args.dataset == 'PACS' else 512
    # out_dim = 256 if args.dataset == 'PACS' else 512

    # fc_proj = nn.Parameter(torch.FloatTensor(out_dim, out_dim))
    # if args.dataset == 'PACS':
    #     fea_proj = nn.Sequential(
    #         nn.Linear(out_dim, out_dim),
    #     )
    # else:
    #     fea_proj = nn.Sequential(
    #         nn.Linear(out_dim, hidden_size),
    #         dropout,
    #         nn.Linear(hidden_size, out_dim),
    #     )  
    # return fea_proj, fc_proj, out_dim

    # feature projector from super contrastive loss code.
    if head == 'linear':
            proj_head = nn.Linear(dim_in, feat_dim)
    elif head == 'mlp':
        proj_head = nn.Sequential(
            nn.Linear(dim_in, dim_in),
            nn.ReLU(inplace=True),
            nn.Linear(dim_in, feat_dim)
        )
    return proj_head
    
