# Neural Spline Flows (https://arxiv.org/abs/1906.04032)
# https://github.com/bayesiains/nsf

import nf
import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.distributions as td

def searchsorted(bin_locations, inputs, eps=1e-6):
    bin_locations[..., -1] += eps
    return torch.sum(inputs[..., None] >= bin_locations, dim=-1) - 1

def rational_quadratic_spline(inputs,
                              unnormalized_widths,
                              unnormalized_heights,
                              unnormalized_derivatives,
                              inverse=False,
                              left=0., right=1., bottom=0., top=1.,
                              min_bin_width=1e-2,
                              min_bin_height=1e-2,
                              min_derivative=1e-3):

    if torch.min(inputs) < left or torch.max(inputs) > right:
        raise transforms.InputOutsideDomain()

    num_bins = unnormalized_widths.shape[-1]

    if min_bin_width * num_bins > 1.0:
        raise ValueError('Minimal bin width too large for the number of bins')
    if min_bin_height * num_bins > 1.0:
        raise ValueError('Minimal bin height too large for the number of bins')

    widths = F.softmax(unnormalized_widths, dim=-1)
    widths = min_bin_width + (1 - min_bin_width * num_bins) * widths
    cumwidths = torch.cumsum(widths, dim=-1)
    cumwidths = F.pad(cumwidths, pad=(1, 0), mode='constant', value=0.0)
    cumwidths = (right - left) * cumwidths + left
    cumwidths[..., 0] = left
    cumwidths[..., -1] = right
    widths = cumwidths[..., 1:] - cumwidths[..., :-1]

    derivatives = min_derivative + F.softplus(unnormalized_derivatives)

    heights = F.softmax(unnormalized_heights, dim=-1)
    heights = min_bin_height + (1 - min_bin_height * num_bins) * heights
    cumheights = torch.cumsum(heights, dim=-1)
    cumheights = F.pad(cumheights, pad=(1, 0), mode='constant', value=0.0)
    cumheights = (top - bottom) * cumheights + bottom
    cumheights[..., 0] = bottom
    cumheights[..., -1] = top
    heights = cumheights[..., 1:] - cumheights[..., :-1]

    if inverse:
        bin_idx = searchsorted(cumheights, inputs)[..., None]
    else:
        bin_idx = searchsorted(cumwidths, inputs)[..., None]

    input_cumwidths = cumwidths.gather(-1, bin_idx)[..., 0]
    input_bin_widths = widths.gather(-1, bin_idx)[..., 0]

    input_cumheights = cumheights.gather(-1, bin_idx)[..., 0]
    delta = heights / widths
    input_delta = delta.gather(-1, bin_idx)[..., 0]

    input_derivatives = derivatives.gather(-1, bin_idx)[..., 0]
    input_derivatives_plus_one = derivatives[..., 1:].gather(-1, bin_idx)[..., 0]

    input_heights = heights.gather(-1, bin_idx)[..., 0]

    if inverse:
        a = (((inputs - input_cumheights) * (input_derivatives
                                             + input_derivatives_plus_one
                                             - 2 * input_delta)
              + input_heights * (input_delta - input_derivatives)))
        b = (input_heights * input_derivatives
             - (inputs - input_cumheights) * (input_derivatives
                                              + input_derivatives_plus_one
                                              - 2 * input_delta))
        c = - input_delta * (inputs - input_cumheights)

        discriminant = b.pow(2) - 4 * a * c
        if (discriminant < 0).any():
            raise ValueError('Discriminant should be non-negative everywhere.')

        root = (2 * c) / (-b - torch.sqrt(discriminant))
        outputs = root * input_bin_widths + input_cumwidths

        theta_one_minus_theta = root * (1 - root)
        denominator = input_delta + ((input_derivatives + input_derivatives_plus_one - 2 * input_delta)
                                     * theta_one_minus_theta)
        derivative_numerator = input_delta.pow(2) * (input_derivatives_plus_one * root.pow(2)
                                                     + 2 * input_delta * theta_one_minus_theta
                                                     + input_derivatives * (1 - root).pow(2))
        logabsdet = torch.log(derivative_numerator) - 2 * torch.log(denominator)

        return outputs, -logabsdet
    else:
        theta = (inputs - input_cumwidths) / input_bin_widths
        theta_one_minus_theta = theta * (1 - theta)

        numerator = input_heights * (input_delta * theta.pow(2)
                                     + input_derivatives * theta_one_minus_theta)
        denominator = input_delta + ((input_derivatives + input_derivatives_plus_one - 2 * input_delta)
                                     * theta_one_minus_theta)
        outputs = input_cumheights + numerator / denominator

        derivative_numerator = input_delta.pow(2) * (input_derivatives_plus_one * theta.pow(2)
                                                     + 2 * input_delta * theta_one_minus_theta
                                                     + input_derivatives * (1 - theta).pow(2))
        logabsdet = torch.log(derivative_numerator) - 2 * torch.log(denominator)

    return outputs, logabsdet

# unconstrained rational quadratic spline
def spline(inputs,
           unnormalized_widths,
           unnormalized_heights,
           unnormalized_derivatives,
           lower,
           upper,
           inverse=False,
           tails='linear',
           min_bin_width=1e-3,
           min_bin_height=1e-3,
           min_derivative=1e-3):

    unnormalized_widths = unnormalized_widths.expand(*inputs.shape, -1)
    unnormalized_heights = unnormalized_heights.expand(*inputs.shape, -1)
    unnormalized_derivatives = unnormalized_derivatives.expand(*inputs.shape, -1)

    inside_interval_mask = (inputs >= lower) & (inputs <= upper)
    outside_interval_mask = ~inside_interval_mask

    outputs = torch.zeros_like(inputs)
    logabsdet = torch.zeros_like(inputs)

    if tails == 'linear':
        unnormalized_derivatives = F.pad(unnormalized_derivatives, pad=(1, 1))
        constant = np.log(np.exp(1 - min_derivative) - 1)
        unnormalized_derivatives[..., 0] = constant
        unnormalized_derivatives[..., -1] = constant

        outputs[outside_interval_mask] = inputs[outside_interval_mask]
        logabsdet[outside_interval_mask] = 0
    else:
        raise RuntimeError('{} tails are not implemented.'.format(tails))

    if not inside_interval_mask.any():
        return outputs, logabsdet

    outputs[inside_interval_mask], logabsdet[inside_interval_mask] = rational_quadratic_spline(
        inputs=inputs[inside_interval_mask],
        unnormalized_widths=unnormalized_widths[inside_interval_mask, :],
        unnormalized_heights=unnormalized_heights[inside_interval_mask, :],
        unnormalized_derivatives=unnormalized_derivatives[inside_interval_mask, :],
        inverse=inverse,
        left=lower, right=upper, bottom=lower, top=upper,
        min_bin_width=min_bin_width,
        min_bin_height=min_bin_height,
        min_derivative=min_derivative
    )

    return outputs, logabsdet

class Spline(nn.Module):
    def __init__(self, dim, n_bins=5, lower=0, upper=1, latent_dim=None, **kwargs):
        super().__init__()
        self.lower = lower
        self.upper = upper
        self.dim = dim
        self.n_bins = n_bins

        if latent_dim is None:
            bound = 0.001
            self.width = nn.Parameter(torch.Tensor(dim, n_bins).uniform_(-bound, bound))
            self.height = nn.Parameter(torch.Tensor(1, dim, n_bins).uniform_(-bound, bound))
            self.derivative = nn.Parameter(torch.Tensor(1, dim, n_bins + 1).uniform_(-bound, bound))
        else:
            bound = 1 / latent_dim
            self.proj = nn.Linear(latent_dim, dim * (n_bins * 3 + 1))
            self.proj.weight.data.uniform_(-bound, bound)
            self.proj.bias.data.fill_(0.0)

    def get_params(self, latent):
        if latent is None:
            return self.width, self.height, self.derivative
        else:
            params = self.proj(latent)
            params = params.view(*params.shape[:-1], self.dim, self.n_bins * 3 + 1)
            width = params[...,:self.n_bins]
            height = params[...,self.n_bins:2*self.n_bins]
            derivative = params[...,2*self.n_bins:]
            return width, height, derivative

    def forward(self, x, latent=None, **kwargs):
        w, h, d = self.get_params(latent)
        return spline(x, w, h, d, inverse=False, lower=self.lower, upper=self.upper)

    def inverse(self, y, latent=None, **kwargs):
        w, h, d = self.get_params(latent)
        return spline(y, w, h, d, inverse=True, lower=self.lower, upper=self.upper)
