import torch
import pandas as pd
import numpy as np
import torchvision.transforms as trans
import torchvision
import math
from scipy.fftpack import dct, idct
import random
from PIL import Image
import os

# mean and std for different datasets
IMAGENET_SIZE = 224
IMAGENET_MEAN = [0.485, 0.456, 0.406]
IMAGENET_STD = [0.229, 0.224, 0.225]
IMAGENET_TRANSFORM = trans.Compose([
    trans.Resize(256),
    trans.CenterCrop(224),
    trans.ToTensor()])

INCEPTION_SIZE = 299
INCEPTION_TRANSFORM = trans.Compose([
    trans.Resize(342),
    trans.CenterCrop(299),
    trans.ToTensor()])

CIFAR_SIZE = 32
CIFAR_MEAN = [0.4914, 0.4822, 0.4465]
CIFAR_STD = [0.2023, 0.1994, 0.2010]
CIFAR_TRANSFORM = trans.Compose([
    trans.ToTensor()])

MNIST_SIZE = 28
MNIST_MEAN = [0.1307]
MNIST_STD = [0.3081]
MNIST_TRANSFORM = trans.Compose([
    trans.ToTensor()])

SkinCancer_TRANSFORM = trans.ToTensor()
DiabeticRetinopathy_TRANSFORM = transforms_test = trans.Compose([trans.ToTensor(), trans.RandomCrop(32)])


class SkinCancerData(torch.utils.data.Dataset):
    def __init__(self, df_dir, transform=None):
        super().__init__()
        self.data_array = pd.read_csv(df_dir).to_numpy()
        self.transform = transform

    def __len__(self):
        return self.data_array.shape[0]

    def __getitem__(self, index):
        # Load data and get label
        X = Image.fromarray(np.uint8(self.data_array[index, :-1].reshape(28, 28, 3)))
        y = torch.tensor(int(self.data_array[index, -1]))

        if self.transform:
            X = self.transform(X)

        return X, y


class DiabeticRetinopathyData(torch.utils.data.Dataset):
    def __init__(self, base_dir, transform=None):
        self.base_dir = os.path.join(base_dir, 'images')
        self.df = pd.read_csv(os.path.join(base_dir, 'test.csv'))
        self.transform = transform

    def __len__(self):
        return len(self.df)

    def __getitem__(self, index):
        # Load data and get label
        X = Image.open(os.path.join(self.base_dir, self.df['id_code'][index] + ".png"))
        y = torch.tensor(int(self.df['diagnosis'][index]))

        if self.transform:
            X = self.transform(X)

        return X, y


# add gaussian noise
def add_gaussian_noise(img, sigma=0):
    noise_img = img + sigma * torch.randn_like(img)
    return noise_img


# reverses the normalization transformation
def invert_normalization(imgs, dataset):
    if dataset == 'imagenet':
        mean = IMAGENET_MEAN
        std = IMAGENET_STD
    elif dataset == 'cifar':
        mean = CIFAR_MEAN
        std = CIFAR_STD
    elif dataset == 'mnist':
        mean = MNIST_MEAN
        std = MNIST_STD
    imgs_trans = imgs.clone()
    if len(imgs.size()) == 3:
        for i in range(imgs.size(0)):
            imgs_trans[i, :, :] = imgs_trans[i, :, :] * std[i] + mean[i]
    else:
        for i in range(imgs.size(1)):
            imgs_trans[:, i, :, :] = imgs_trans[:, i, :, :] * std[i] + mean[i]
    return imgs_trans


# applies the normalization transformations
def apply_normalization(imgs, dataset, sigma=0):
    if dataset == 'imagenet':
        mean = IMAGENET_MEAN
        std = IMAGENET_STD
    elif dataset == 'cifar':
        mean = CIFAR_MEAN
        std = CIFAR_STD
    elif dataset == 'mnist':
        mean = MNIST_MEAN
        std = MNIST_STD
    else:
        mean = [0, 0, 0]
        std = [1, 1, 1]
    imgs_tensor = imgs.clone()
    if dataset == 'mnist':
        imgs_tensor = add_gaussian_noise(imgs_tensor, sigma)
        imgs_tensor = (imgs_tensor - mean[0]) / std[0]
    else:
        imgs_tensor = add_gaussian_noise(imgs_tensor, sigma)
        if imgs.dim() == 3:
            for i in range(imgs_tensor.size(0)):
                imgs_tensor[i, :, :] = (imgs_tensor[i, :, :] - mean[i]) / std[i]
        else:
            for i in range(imgs_tensor.size(1)):
                imgs_tensor[:, i, :, :] = (imgs_tensor[:, i, :, :] - mean[i]) / std[i]
    return imgs_tensor


# get most likely predictions and probabilities for a set of inputs
def get_preds(model, inputs, dataset_name, correct_class=None, batch_size=25, return_cpu=True, sigma=0):
    num_batches = int(math.ceil(inputs.size(0) / float(batch_size)))
    softmax = torch.nn.Softmax(dim=1)
    all_preds, all_probs = None, None
    transform = trans.Normalize(IMAGENET_MEAN, IMAGENET_STD)
    with torch.no_grad():
        for i in range(num_batches):
            upper = min((i + 1) * batch_size, inputs.size(0))
            input = apply_normalization(inputs[(i * batch_size):upper], dataset_name, sigma).cuda()
            output = softmax.forward(model.forward(input))
            if correct_class is None:
                prob, pred = output.max(1)
            else:
                prob, pred = output[:, correct_class], torch.autograd.Variable(
                    torch.ones(output.size()) * correct_class)
            if return_cpu:
                prob = prob.data.cpu()
                pred = pred.data.cpu()
            else:
                prob = prob.data
                pred = pred.data
            if i == 0:
                all_probs = prob
                all_preds = pred
            else:
                all_probs = torch.cat((all_probs, prob), 0)
                all_preds = torch.cat((all_preds, pred), 0)
    return all_preds, all_probs


# get least likely predictions and probabilities for a set of inputs
def get_least_likely(model, inputs, dataset_name, batch_size=25, return_cpu=True, sigma=0):
    num_batches = int(math.ceil(inputs.size(0) / float(batch_size)))
    softmax = torch.nn.Softmax()
    all_preds, all_probs = None, None
    transform = trans.Normalize(IMAGENET_MEAN, IMAGENET_STD)
    for i in range(num_batches):
        upper = min((i + 1) * batch_size, inputs.size(0))
        input = apply_normalization(inputs[(i * batch_size):upper], dataset_name, sigma)
        input_var = torch.autograd.Variable(input.cuda(), volatile=True)
        output = softmax.forward(model.forward(input_var))
        prob, pred = output.min(1)
        if return_cpu:
            prob = prob.data.cpu()
            pred = pred.data.cpu()
        else:
            prob = prob.data
            pred = pred.data
        if i == 0:
            all_probs = prob
            all_preds = pred
        else:
            all_probs = torch.cat((all_probs, prob), 0)
            all_preds = torch.cat((all_preds, pred), 0)
    return all_preds, all_probs


# defines a diagonal order
# order is fixed across diagonals but are randomized across channels and within the diagonal
# e.g.
# [1, 2, 5]
# [3, 4, 8]
# [6, 7, 9]
# def diagonal_order(image_size, channels):
#     x = torch.arange(0, image_size).cumsum(0)
#     order = torch.zeros(image_size, image_size)
#     for i in range(image_size):
#         order[i, :(image_size - i)] = i + x[i:]
#     for i in range(1, image_size):
#         reverse = order[image_size - i - 1].index_select(0, torch.LongTensor([i for i in range(i - 1, -1, -1)]))
#         order[i, (image_size - i):] = image_size * image_size - 1 - reverse
#     if channels > 1:
#         order_2d = order
#         order = torch.zeros(channels, image_size, image_size)
#         for i in range(channels):
#             order[i, :, :] = 3 * order_2d + i
#     return order.view(1, -1).squeeze().long().sort()[1]

def diagonal_order(image_size, channels):
    order = torch.zeros((image_size, image_size))
    dic = {}
    for i in range(image_size):
        for j in range(image_size):
            dic[(i, j)] = (i**2 + j**2) + 0.01 * (i+j)

    x = dict(sorted(dic.items(), key=lambda item: item[1]))
    for idx, (i, j) in enumerate(x.keys()):
        order[i, j] = int(idx)

    if channels > 1:
        order_2d = order
        order = torch.zeros(channels, image_size, image_size)
        for i in range(channels):
            order[i, :, :] = 3 * order_2d + i
    return order.view(1, -1).squeeze().long().sort()[1]


# defines a block order, starting with top-left (initial_size x initial_size) submatrix
# expanding by stride rows and columns whenever exhausted
# randomized within the block and across channels
# e.g. (initial_size=2, stride=1)
# [1, 3, 6]
# [2, 4, 9]
# [5, 7, 8]
def block_order(image_size, channels, initial_size=1, stride=1):
    order = torch.zeros(channels, image_size, image_size)
    total_elems = channels * initial_size * initial_size
    perm = torch.randperm(total_elems)
    order[:, :initial_size, :initial_size] = perm.view(channels, initial_size, initial_size)
    for i in range(initial_size, image_size, stride):
        num_elems = channels * (2 * stride * i + stride * stride)
        perm = torch.randperm(num_elems) + total_elems
        num_first = channels * stride * (stride + i)
        order[:, :(i + stride), i:(i + stride)] = perm[:num_first].view(channels, -1, stride)
        order[:, i:(i + stride), :i] = perm[num_first:].view(channels, stride, -1)
        total_elems += num_elems
    return order.view(1, -1).squeeze().long().sort()[1]


# zeros all elements outside of the top-left (block_size * ratio) submatrix for every block
def block_zero(x, block_size=8, ratio=0.5):
    z = torch.zeros(x.size())
    num_blocks = int(x.size(2) / block_size)
    mask = torch.zeros(x.size(0), x.size(1), block_size, block_size)
    mask[:, :, :int(block_size * ratio), :int(block_size * ratio)] = 1
    for i in range(num_blocks):
        for j in range(num_blocks):
            z[:, :, (i * block_size):((i + 1) * block_size), (j * block_size):((j + 1) * block_size)] = x[:, :, (i * block_size):((i + 1) * block_size),(j * block_size):((j + 1) * block_size)] * mask
    return z


# applies DCT to each block of size block_size
def block_dct(x, block_size=8, masked=False, ratio=0.5):
    z = torch.zeros(x.size())
    num_blocks = int(x.size(2) / block_size)
    mask = np.zeros((x.size(0), x.size(1), block_size, block_size))
    mask[:, :, :int(block_size * ratio), :int(block_size * ratio)] = 1
    for i in range(num_blocks):
        for j in range(num_blocks):
            submat = x[:, :, (i * block_size):((i + 1) * block_size), (j * block_size):((j + 1) * block_size)].numpy()
            submat_dct = dct(dct(submat, axis=2, norm='ortho'), axis=3, norm='ortho')
            if masked:
                submat_dct = submat_dct * mask
            submat_dct = torch.from_numpy(submat_dct)
            z[:, :, (i * block_size):((i + 1) * block_size), (j * block_size):((j + 1) * block_size)] = submat_dct
    return z


# applies IDCT to each block of size block_size
def block_idct(x, block_size=8, masked=False, ratio=0.5, linf_bound=0.0):
    z = torch.zeros(x.size())
    num_blocks = int(x.size(2) / block_size)
    mask = np.zeros((x.size(0), x.size(1), block_size, block_size))
    if type(ratio) != float:
        for i in range(x.size(0)):
            mask[i, :, :int(block_size * ratio[i]), :int(block_size * ratio[i])] = 1
    else:
        mask[:, :, :int(block_size * ratio), :int(block_size * ratio)] = 1
    for i in range(num_blocks):
        for j in range(num_blocks):
            submat = x[:, :, (i * block_size):((i + 1) * block_size), (j * block_size):((j + 1) * block_size)].numpy()
            if masked:
                submat = submat * mask
            z[:, :, (i * block_size):((i + 1) * block_size),
            (j * block_size):((j + 1) * block_size)] = torch.from_numpy(
                idct(idct(submat, axis=3, norm='ortho'), axis=2, norm='ortho'))
    if linf_bound > 0:
        return z.clamp(-linf_bound, linf_bound)
    else:
        return z


def setup_seed(seed):
    np.random.seed(seed)
    random.seed(seed)
    torch.manual_seed(seed)
    if torch.cuda.is_available():
        torch.cuda.manual_seed(seed)
        torch.cuda.manual_seed_all(seed)
    np.random.seed(seed)
    torch.backends.cudnn.benchmark = False
    torch.backends.cudnn.deterministic = True


def get_dataset(dataset_name, dataset_dir, batch_size=64):
    if dataset_name == "CIFAR10":
        transform_test = trans.Compose([
            trans.ToTensor(),
            trans.Normalize((0.4914, 0.4822, 0.4465), (0.2023, 0.1994, 0.2010))
        ])
        test_data = torchvision.datasets.CIFAR10(root=dataset_dir, train=False, download=True, transform=transform_test)
        test_dataloader = torch.utils.data.DataLoader(test_data, batch_size=batch_size, shuffle=False)

    elif dataset_name == "MNIST":
        transform_test = trans.Compose([
            trans.ToTensor(),
            trans.Normalize((0.1307,), (0.3081,))
        ])
        test_data = torchvision.datasets.MNIST(root=dataset_dir, train=False, download=True, transform=transform_test)
        test_dataloader = torch.utils.data.DataLoader(test_data, batch_size=batch_size, shuffle=False)

    elif dataset_name == "SkinCancer":
        transforms_test = trans.ToTensor()

        class TestData(torch.utils.data.Dataset):
            def __init__(self, df_dir, transform=None):
                super().__init__()
                self.data_array = pd.read_csv(df_dir).to_numpy()
                self.transform = transform

            def __len__(self):
                return self.data_array.shape[0]

            def __getitem__(self, index):
                # Load data and get label
                X = Image.fromarray(np.uint8(self.data_array[index, :-1].reshape(28, 28, 3)))
                y = torch.tensor(int(self.data_array[index, -1]))

                if self.transform:
                    X = self.transform(X)

                return X, y

        test_data = TestData(os.path.join(dataset_dir, 'SkinCancer', 'test.csv'), transforms_test)
        test_dataloader = torch.utils.data.DataLoader(test_data, batch_size=batch_size, shuffle=False)

    else:
        test_dataloader = None

    return test_dataloader


def get_target_labels(true_labels, label_set):
    target_labels = true_labels.clone()
    for i in range(target_labels.size(0)):
        while target_labels[i] == true_labels[i]:
            target_labels[i] = torch.tensor(random.choice(label_set))
    return target_labels

if __name__=="__main__":
    a = diagonal_order(10, 1)
    b = block_order(10, 1, initial_size=2, stride=2)